<?php
namespace DigitalProductPassport;

if (!defined('ABSPATH')) { exit; }

class Shortcode {
    public static function init() {
        // Shortcode tag (matches plugin slug)
        add_shortcode('digital_product_passport', [__CLASS__, 'render']);
        // Prevent WordPress from texturizing content inside our shortcode (curly quotes break JSON)
        add_filter('no_texturize_shortcodes', [__CLASS__, 'no_texturize']);
    }

    public static function no_texturize($shortcodes) {
        if (!is_array($shortcodes)) { $shortcodes = []; }
        foreach (['digital_product_passport'] as $tag) {
            if (!in_array($tag, $shortcodes, true)) {
                $shortcodes[] = $tag;
            }
        }
        return $shortcodes;
    }

    public static function render($atts = [], $content = null, $tag = '') {
        $atts = shortcode_atts([
            'json' => '',
            'show_raw' => 'false', // deprecated, kept for BC
            'view' => 'html', // html | json (raw accepted for BC)
            'title' => '',
            // Viewer configuration
            'limit' => '0', // number of sections/items to show before "Show more"; 0 = all
            'include' => '', // comma-separated list of keys/labels to include
            // Not a WP_Query arg; used only for front-end filtering
            'exclude' => '', // phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
            'order' => '', // comma-separated preferred order (keys/labels)
            'more_label' => '', // override "Show more"
            'less_label' => '', // override "Show less"
        ], $atts, $tag);

        $json = self::resolve_json($atts, $content);

        // Enqueue assets only when used
        wp_enqueue_style('dpp-viewer');
        wp_enqueue_script('dpp-viewer');

        $title_html = '';
        if (!empty($atts['title'])) {
            $title_html = '<div class="dpp-header"><strong>' . esc_html($atts['title']) . '</strong></div>';
        }

        if ($json === null) {
            return '<div class="dpp-viewer dpp-error">' . esc_html__('No Digital Product Passport JSON found.', 'digital-product-passport') . '</div>';
        }
        // Determine start view: honor show_raw first for BC, else use view
        $start_view = 'html';
        if (filter_var($atts['show_raw'], FILTER_VALIDATE_BOOLEAN)) {
            $start_view = 'raw';
        } else {
            $candidate_view = strtolower((string)$atts['view']);
            // Accept 'json' (preferred) and legacy 'raw' to mean the JSON tab
            if ($candidate_view === 'json' || $candidate_view === 'raw') {
                $start_view = 'raw';
            } elseif ($candidate_view === 'html') {
                $start_view = 'html';
            }
        }

        // Prepare JSON for safe embedding within a <script type="application/json"> tag.
        // Re-encode without JSON_UNESCAPED_SLASHES so </script> becomes <\/script>.
        $json_for_script = 'null';
        $decoded_for_script = json_decode($json, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            $json_for_script = wp_json_encode($decoded_for_script);
        }

        ob_start();
        ?>
        <div class="dpp-viewer"
             data-start-view="<?php echo esc_attr($start_view); ?>"
             data-limit="<?php echo esc_attr((string)$atts['limit']); ?>"
             data-include="<?php echo esc_attr((string)$atts['include']); ?>"
             data-exclude="<?php echo esc_attr((string)$atts['exclude']); ?>"
             data-order="<?php echo esc_attr((string)$atts['order']); ?>"
             data-more-label="<?php echo esc_attr((string)$atts['more_label']); ?>"
             data-less-label="<?php echo esc_attr((string)$atts['less_label']); ?>"
        >
            <?php echo $title_html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
            <?php $dashicons_active = wp_style_is('dashicons', 'enqueued'); ?>
            <div class="dpp-toolbar dpp-tabs" role="tablist" aria-label="<?php echo esc_attr__('Digital Product Passport view', 'digital-product-passport'); ?>">
                <button type="button" class="dpp-tab dpp-tab-html" role="tab" aria-selected="false" data-view="html">
                    <?php if ($dashicons_active) { echo '<span class="dashicons dashicons-media-text" aria-hidden="true"></span> '; } ?>
                    <?php echo esc_html__('HTML', 'digital-product-passport'); ?>
                </button>
                <button type="button" class="dpp-tab dpp-tab-raw" role="tab" aria-selected="false" data-view="raw">
                    <?php if ($dashicons_active) { echo '<span class="dashicons dashicons-media-code" aria-hidden="true"></span> '; } ?>
                    <?php echo esc_html__('JSON', 'digital-product-passport'); ?>
                </button>
            </div>
            <div class="dpp-html" role="tabpanel" hidden></div>
            <pre class="dpp-raw" role="tabpanel" hidden></pre>
            <?php
            // Print the data script tag via WP helper for CSP/nonces per WP 5.7.
            if (function_exists('wp_print_inline_script_tag')) {
                wp_print_inline_script_tag(
                    $json_for_script,
                    [ 'type' => 'application/json', 'class' => 'dpp-data' ]
                );
            } elseif (function_exists('wp_get_inline_script_tag')) {
                // Fallback to get_* + echo on WP 5.7 if print_* is unavailable in some environments.
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- HTML produced by core helper
                echo wp_get_inline_script_tag(
                    $json_for_script,
                    [ 'type' => 'application/json', 'class' => 'dpp-data' ]
                );
            } else {
                // Back-compat: raw tag on very old WP; JSON already safe via wp_json_encode above.
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- printing safe JSON in script tag
                echo '<script type="application/json" class="dpp-data">' . $json_for_script . '</script>';
            }
            ?>
        </div>
        <?php
        return ob_get_clean();
    }

    private static function resolve_json($atts, $content) {
        $candidate = '';
        $origin = '';
        if (!empty($atts['json'])) {
            $candidate = $atts['json'];
            $origin = 'atts';
        } elseif (!empty($content)) {
            $candidate = $content;
            $origin = 'content';
        } else {
            $post_id = get_the_ID();
            if ($post_id) {
                $meta = get_post_meta($post_id, MetaBox::META_KEY, true);
                if (!empty($meta)) {
                    $candidate = $meta;
                    $origin = 'meta';
                }
            }
        }
        $candidate = trim((string)$candidate);
        if ($candidate === '') { return null; }

        $decoded = json_decode($candidate, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            // Attempt to clean up common WP editor transformations
            $clean = self::cleanup_pasted_json($candidate);
            $decoded = json_decode($clean, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                return null;
            }
        }
        // Allow developers to filter/redact data before output (privacy/PII control)
        $decoded = apply_filters('digital_product_passport_viewer_data', $decoded, ['origin' => $origin]);
        // Re-encode normalized
        return wp_json_encode($decoded, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }

    private static function cleanup_pasted_json($s) {
        $s = (string)$s;
        // Replace common WP auto-format tags with newlines
        $s = preg_replace('~<br\s*/?>~i', "\n", $s);
        $s = preg_replace('~</p>\s*<p>~i', "\n", $s);
        // Strip any remaining HTML tags
        $s = wp_strip_all_tags($s);
        // Decode entities (quotes, etc.)
        $s = html_entity_decode($s, ENT_QUOTES | ENT_HTML5, get_bloginfo('charset') ?: 'UTF-8');
        // Normalize curly quotes to straight quotes
        $map = [
            "\xE2\x80\x9C" => '"', // “
            "\xE2\x80\x9D" => '"', // ”
            "\xE2\x80\x98" => "'", // ‘
            "\xE2\x80\x99" => "'", // ’
        ];
        $s = strtr($s, $map);
        // Remove optional Markdown fences
        if (preg_match('/^```[a-zA-Z0-9\-_*]*\s*/', $s)) {
            $s = preg_replace('/^```[\s\S]*?\n/', '', $s);
            $s = preg_replace('/\n```\s*$/', '', $s);
        }
        $s = trim($s);
        // Try to salvage a JSON object/array region if extra text remains
        if (strpos($s, '{') !== false && strrpos($s, '}') !== false) {
            $start = strpos($s, '{');
            $end = strrpos($s, '}');
            if ($end > $start) {
                $slice = substr($s, $start, $end - $start + 1);
                // Prefer the sliced object if it looks plausible
                if (substr($slice, 0, 1) === '{' && substr($slice, -1) === '}') {
                    $s = $slice;
                }
            }
        } elseif (strpos($s, '[') !== false && strrpos($s, ']') !== false) {
            $start = strpos($s, '[');
            $end = strrpos($s, ']');
            if ($end > $start) {
                $slice = substr($s, $start, $end - $start + 1);
                if (substr($slice, 0, 1) === '[' && substr($slice, -1) === ']') {
                    $s = $slice;
                }
            }
        }
        return $s;
    }
}
