<?php
namespace DigitalProductPassport;

if (!defined('ABSPATH')) { exit; }

class MetaBox {
    const META_KEY = '_dpp_json';

    public static function init() {
        add_action('add_meta_boxes', [__CLASS__, 'add_metabox']);
        add_action('save_post', [__CLASS__, 'save']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin_assets']);
    }

    public static function add_metabox() {
        $post_types = get_post_types(['public' => true], 'names');
        // Add WooCommerce product if installed
        if (post_type_exists('product') && !in_array('product', $post_types, true)) {
            $post_types[] = 'product';
        }
        foreach ($post_types as $pt) {
            add_meta_box(
                'dpp_metabox',
                __('Digital Product Passport', 'digital-product-passport'),
                [__CLASS__, 'render_metabox'],
                $pt,
                'normal',
                'default'
            );
        }
    }

    public static function render_metabox($post) {
        wp_nonce_field('dpp_metabox_save', 'dpp_metabox_nonce');
        $json = get_post_meta($post->ID, self::META_KEY, true);
        // Pretty print for editor display if valid JSON is stored
        $display_json = $json;
        if (is_string($json) && $json !== '') {
            $decoded = json_decode($json, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                $display_json = wp_json_encode($decoded, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
            }
        }
        $placeholder = "{\n" .
            "  \"id\": \"urn:uuid:123e4567-e89b-12d3-a456-426614174000\",\n" .
            "  \"type\": \"Product\",\n" .
            "  \"name\": \"Acme Widget 2000\",\n" .
            "  \"brand\": \"ACME\",\n" .
            "  \"model\": \"W-2000\",\n" .
            "  \"serialNumber\": \"SN-001234\",\n" .
            "  \"manufacturer\": {\n" .
            "    \"name\": \"ACME Corporation\",\n" .
            "    \"address\": \"123 Industrial Ave, Springfield\",\n" .
            "    \"website\": \"https://example.com\"\n" .
            "  },\n" .
            "  \"properties\": [\n" .
            "    { \"name\": \"Material\", \"value\": \"Aluminum\" },\n" .
            "    { \"name\": \"Color\", \"value\": \"Silver\" },\n" .
            "    { \"name\": \"Weight\", \"value\": \"1.2 kg\" }\n" .
            "  ],\n" .
            "  \"documentation\": [\n" .
            "    { \"name\": \"User Manual\", \"url\": \"https://example.com/manual.pdf\" }\n" .
            "  ],\n" .
            "  \"warranty\": {\n" .
            "    \"period\": \"24 months\",\n" .
            "    \"terms\": \"Defects in materials and workmanship\"\n" .
            "  }\n" .
            "}";

        echo '<div class="dpp-metabox">';
        echo '<p>' . esc_html__('Paste your Digital Product Passport JSON below. It must be valid JSON. The placeholder shows a typical structure.', 'digital-product-passport') . '</p>';
        echo '<textarea name="dpp_json" id="dpp_json" placeholder="' . esc_attr($placeholder) . '" spellcheck="false">' . esc_textarea($display_json) . '</textarea>';
        // Actions row: validation note + format button
        echo '<div class="dpp-actions">'
                . '<span class="dpp-json-validation-note" aria-live="polite"></span>'
                . '<button type="button" class="button button-secondary" id="dpp_json_format_btn">' . esc_html__('Format JSON', 'digital-product-passport') . '</button>'
            . '</div>';
        echo '</div>';
    }

    public static function enqueue_admin_assets($hook) {
        if (in_array($hook, ['post.php', 'post-new.php'], true)) {
            // Enqueue the WordPress Code Editor (CodeMirror) for JSON if available
            $settings = function_exists('wp_enqueue_code_editor') ? wp_enqueue_code_editor(['type' => 'application/json']) : false;
            if ($settings) {
                // Ensure core code editor assets are present
                wp_enqueue_script('code-editor');
                wp_enqueue_style('code-editor');
            }
            // Enqueue plugin admin styles
            wp_enqueue_style('dpp-admin');
            // Add small inline CSS hooks for validation states
            wp_add_inline_style('dpp-admin', '.dpp-json-validation-note.ok{color:#15803d}.dpp-json-validation-note.err{color:#b91c1c}');

            // Provide settings to admin script via inline JS before it loads
            if ($settings) {
                wp_add_inline_script('dpp-admin', 'window.dppAdminCodeEditorSettings = ' . wp_json_encode($settings) . ';', 'before');
            }
            wp_enqueue_script('dpp-admin');
        }
    }

    public static function save($post_id) {
        // Nonce check (unslash + sanitize for safety before verification)
        if (!isset($_POST['dpp_metabox_nonce'])) {
            return;
        }
        $nonce = sanitize_text_field(wp_unslash($_POST['dpp_metabox_nonce']));
        if (!wp_verify_nonce($nonce, 'dpp_metabox_save')) {
            return;
        }
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) { return; }
        // Use actual post type to determine capability checks rather than trusting $_POST
        $post_type = get_post_type($post_id);
        if ($post_type === 'page') {
            if (!current_user_can('edit_page', $post_id)) { return; }
        } else {
            if (!current_user_can('edit_post', $post_id)) { return; }
        }

        if (isset($_POST['dpp_json'])) {
            $raw = sanitize_textarea_field( wp_unslash( $_POST['dpp_json'] ) );
            $trimmed = trim($raw);
            if ($trimmed === '') {
                delete_post_meta($post_id, self::META_KEY);
                return;
            }

            // Validate JSON
            $decoded = json_decode($trimmed, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                // Safer to not save invalid JSON. Redirect with a verified nonce for the admin notice.
                $notice_nonce = wp_create_nonce('dpp_json_error_' . $post_id);
                add_filter('redirect_post_location', function($location) use ($notice_nonce) {
                    return add_query_arg(
                        [
                            'dpp_json_error' => '1',
                            'dpp_json_error_nonce' => $notice_nonce,
                        ],
                        $location
                    );
                });
                return;
            }
            // Save as a compact JSON to avoid unwanted whitespace
            $encoded = wp_json_encode($decoded, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
            update_post_meta($post_id, self::META_KEY, $encoded);
        }
    }
}

// Admin notice for invalid JSON — protected by a contextual nonce tied to the post ID
add_action('admin_notices', function() {
    if (!isset($_GET['dpp_json_error'], $_GET['dpp_json_error_nonce'])) {
        return;
    }
    $nonce = sanitize_text_field( wp_unslash( $_GET['dpp_json_error_nonce'] ) );
    $post_id = 0;
    if (isset($_GET['post'])) {
        $post_id = absint( wp_unslash( $_GET['post'] ) );
    }
    if (!$post_id) {
        return;
    }
    if (!wp_verify_nonce($nonce, 'dpp_json_error_' . $post_id)) {
        return;
    }
    echo '<div class="notice notice-error"><p>' . esc_html__('Digital Product Passport JSON is invalid and was not saved.', 'digital-product-passport') . '</p></div>';
});
