/* Digital Product Passport viewer: transforms JSON into human-readable HTML */
(function(){
  var DPP_UID = 0;
  function t(s){
    try { return (window.wp && wp.i18n && typeof wp.i18n.__ === 'function') ? wp.i18n.__(s, 'digital-product-passport') : s; }
    catch(e){ return s; }
  }

  function isObject(v){ return Object.prototype.toString.call(v) === '[object Object]'; }
  function isArray(v){ return Array.isArray(v); }

  function esc(str){
    return String(str).replace(/[&<>"']/g, function(s){
      return ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'})[s];
    });
  }

  function renderValue(value){
    if (value === null) return '<span class="dpp-leaf">null</span>';
    if (typeof value === 'boolean' || typeof value === 'number') {
      return '<span class="dpp-leaf">' + esc(value) + '</span>';
    }
    if (typeof value === 'string') {
      return '<span class="dpp-leaf">' + esc(value) + '</span>';
    }
    if (isArray(value)) {
      var items = value.map(function(item){ return '<li class="dpp-kv">' + renderValue(item) + '</li>'; }).join('');
      return '<ul class="dpp-list">' + items + '</ul>';
    }
    if (isObject(value)) {
      var out = Object.keys(value).map(function(key){
        return '<div class="dpp-kv"><span class="dpp-key">' + esc(key) + ':</span> ' + renderValue(value[key]) + '</div>';
      }).join('');
      return '<div class="dpp-object">' + out + '</div>';
    }
    return '<span class="dpp-leaf">' + esc(String(value)) + '</span>';
  }

  // Humanize keys like "EndOfLifeAndCircularity" -> "End of Life and Circularity"
  // Handles camelCase/PascalCase, underscores, dashes, digits, and common acronyms.
  var DPP_MINOR_WORDS = {
    'and':1,'or':1,'of':1,'for':1,'the':1,'in':1,'on':1,'with':1,
    'to':1,'from':1,'by':1,'at':1,'as':1,'per':1,'via':1
  };
  var DPP_ACRONYMS = {
    'id':'ID','sku':'SKU','gtin':'GTIN','ean':'EAN','upc':'UPC','qr':'QR','nfc':'NFC','rfid':'RFID',
    'url':'URL','uri':'URI','urn':'URN','iri':'IRI','eu':'EU','ip':'IP','cpu':'CPU','gpu':'GPU',
    'co2':'CO2','ghg':'GHG','iso':'ISO','iec':'IEC','ce':'CE','kwh':'kWh'
  };
  function humanizeTitle(str){
    var s = String(str || '').trim();
    if (!s) return '';
    // Normalize separators and split camel/pascal case boundaries
    s = s.replace(/[_\-]+/g, ' ');
    s = s.replace(/([A-Z]+)([A-Z][a-z])/g, '$1 $2');
    s = s.replace(/([a-z0-9])([A-Z])/g, '$1 $2');
    s = s.replace(/([0-9])([A-Za-z])/g, '$1 $2');
    s = s.replace(/([A-Za-z])([0-9])/g, '$1 $2');
    s = s.replace(/\s+/g, ' ').trim();
    var parts = s.split(' ');
    for (var i=0;i<parts.length;i++){
      var w = parts[i];
      var lw = w.toLowerCase();
      if (DPP_ACRONYMS[lw]) { parts[i] = DPP_ACRONYMS[lw]; continue; }
      if (w === w.toUpperCase() && /[A-Z]/.test(w)) { parts[i] = w; continue; }
      var cap = w.charAt(0).toUpperCase() + w.slice(1).toLowerCase();
      if (i > 0 && i < parts.length - 1 && DPP_MINOR_WORDS[lw]) cap = lw;
      parts[i] = cap;
    }
    return parts.join(' ');
  }

  function normalizeKey(key){
    if (!key) return '';
    var k = String(key).trim();
    if (k[0] === '@') k = k.slice(1);
    if (/^context$/i.test(k)) return '';
    if (/^id$/i.test(k)) return 'ID';
    if (/^type$/i.test(k)) return 'Type';
    return humanizeTitle(k);
  }

  function get(obj, keys){
    for (var i=0;i<keys.length;i++){
      var k = keys[i];
      if (obj != null && Object.prototype.hasOwnProperty.call(obj, k) && obj[k] != null) return obj[k];
    }
    return undefined;
  }

  function isPropertyValue(obj){
    if (!isObject(obj)) return false;
    var t = get(obj, ['@type','@Type','type','Type']);
    if (t && String(t).toLowerCase() === 'propertyvalue') return true;
    var hasName = get(obj, ['Name','name']) != null;
    var hasValue = get(obj, ['Value','value']) != null;
    return !!(hasName && hasValue);
  }

  function isPropertyValueArray(arr){
    if (!isArray(arr) || arr.length === 0) return false;
    for (var i=0;i<arr.length;i++){
      if (!isPropertyValue(arr[i])) return false;
    }
    return true;
  }

  function renderPropertyValue(pv){
    var name = get(pv, ['Name','name']);
    var value = get(pv, ['Value','value']);
    var label = humanizeTitle(name || 'Property');
    var content = (value == null ? '' : String(value));
    return '<div class="dpp-kv"><span class="dpp-key">' + esc(label) + ':</span> <span class="dpp-leaf">' + esc(content) + '</span></div>';
  }

  function renderPropertyList(list){
    return '<div class="dpp-properties">' + list.map(renderPropertyValue).join('') + '</div>';
  }

  function renderKeyVals(obj, options){
    options = options || {};
    var openRemaining = typeof options.openFirst === 'number' ? options.openFirst : 0;
    return Object.keys(obj).map(function(k){
      var v = obj[k];
      var label = normalizeKey(k);
      if (label === '') return '';
      if (isArray(v) && isPropertyValueArray(v)) {
        var isOpen = openRemaining > 0; if (openRemaining > 0) openRemaining--;
        return renderAccordionSection(t('Properties'), v, { open: isOpen });
      }
      if (isObject(v) || isArray(v)) {
        var isOpen2 = openRemaining > 0; if (openRemaining > 0) openRemaining--;
        return renderAccordionSection(label, v, { open: isOpen2 });
      }
      return '<div class="dpp-kv"><span class="dpp-key">' + esc(label) + ':</span> <span class="dpp-leaf">' + esc(String(v)) + '</span></div>';
    }).join('');
  }

  function renderAccordionSection(label, value, opts){
    opts = opts || {};
    function countRenderableKeys(obj){
      if (!isObject(obj)) return 0;
      var keys = Object.keys(obj);
      var n = 0;
      for (var i=0;i<keys.length;i++){
        if (normalizeKey(keys[i]) !== '') n++;
      }
      return n;
    }
    function buildMeta(value){
      try {
        if (isArray(value)) {
          var kind = isPropertyValueArray(value) ? 'properties' : 'items';
          return '<span class="dpp-acc-meta" aria-hidden="true">' + value.length + ' ' + (kind === 'properties' ? t('properties') : t('items')) + '</span>';
        } else if (isObject(value)) {
          var c = countRenderableKeys(value);
          return '<span class="dpp-acc-meta" aria-hidden="true">' + c + ' ' + t('fields') + '</span>';
        }
      } catch(e){}
      return '';
    }
    var inner;
    if (isArray(value)) {
      if (isPropertyValueArray(value)) {
        inner = renderPropertyList(value);
      } else {
        inner = '<ol class="dpp-acc-list">' + value.map(function(v, i){
          if (isObject(v)) {
            if (isPropertyValue(v)) {
              return '<li>' + renderPropertyValue(v) + '</li>';
            }
            // For array items, only first one open by default
            return '<li><div class="dpp-object">' + renderKeyVals(v, { openFirst: (i === 0 ? 1 : 0) }) + '</div></li>';
          } else if (isArray(v)) {
            return '<li>' + renderValue(v) + '</li>';
          } else {
            return '<li><span class="dpp-leaf">' + esc(String(v)) + '</span></li>';
          }
        }).join('') + '</ol>';
      }
    } else if (isObject(value)) {
      inner = '<div class="dpp-object">' + renderKeyVals(value, { openFirst: 1 }) + '</div>';
    } else {
      inner = '<div class="dpp-leaf">' + esc(String(value)) + '</div>';
    }
    var openAttr = opts.open ? ' open' : '';
    var meta = buildMeta(value);
    return '<details class="dpp-acc-section"' + openAttr + '>' +
           '<summary><span class="dpp-acc-label">' + esc(label) + '</span>' + meta + '</summary>' +
           '<div class="dpp-acc-content">' + inner + '</div>' +
           '</details>';
  }

  // Parse comma separated attribute into normalized tokens
  function parseListAttr(s){
    if (!s) return [];
    if (typeof s !== 'string') return [];
    return s.split(',').map(function(v){ return v.trim(); }).filter(Boolean);
  }

  function normToken(s){
    return String(s || '')
      .replace(/^@/, '')
      .replace(/[_\-\s]+/g, '')
      .toLowerCase();
  }

  function matchKeyAgainstList(rawKey, label, list){
    if (!list || !list.length) return false;
    var rk = normToken(rawKey);
    var lk = normToken(label);
    for (var i=0;i<list.length;i++){
      var t = normToken(list[i]);
      if (t && (rk === t || lk === t)) return true;
    }
    return false;
  }

  function orderKeys(keys, dataObj, include, exclude, order){
    // Filter include first (if present)
    var filtered = [];
    keys.forEach(function(k){
      var disp = normalizeKey(k);
      if (disp === '') return; // skip meta keys
      if (include && include.length && !matchKeyAgainstList(k, disp, include)) return;
      if (exclude && exclude.length && matchKeyAgainstList(k, disp, exclude)) return;
      filtered.push({ key: k, label: disp });
    });
    if (!order || !order.length) return filtered;
    // Stable sort: first by order index if present
    var idxMap = {};
    order.forEach(function(tok, i){ idxMap[normToken(tok)] = i; });
    filtered.sort(function(a, b){
      var ai = idxMap[normToken(a.key)] != null ? idxMap[normToken(a.key)] : (idxMap[normToken(a.label)] != null ? idxMap[normToken(a.label)] : Infinity);
      var bi = idxMap[normToken(b.key)] != null ? idxMap[normToken(b.key)] : (idxMap[normToken(b.label)] != null ? idxMap[normToken(b.label)] : Infinity);
      if (ai !== bi) return ai - bi;
      return 0; // keep original order otherwise
    });
    return filtered;
  }

  function renderWithConfig(data, cfg){
    cfg = cfg || {};
    var limit = parseInt(cfg.limit, 10);
    if (!isFinite(limit) || limit < 0) limit = 0;
    var include = parseListAttr(cfg.include);
    var exclude = parseListAttr(cfg.exclude);
    var order = parseListAttr(cfg.order);
    var moreLabel = cfg.moreLabel || (window.wp && wp.i18n ? wp.i18n.__('Show more', 'digital-product-passport') : 'Show more');
    var lessLabel = cfg.lessLabel || (window.wp && wp.i18n ? wp.i18n.__('Show less', 'digital-product-passport') : 'Show less');

    // Arrays
    if (isArray(data)) {
      if (isPropertyValueArray(data)) {
        // Limit property list items if requested
        var items = data.slice(0);
        var first = (limit > 0 ? items.slice(0, limit) : items);
        var rest = (limit > 0 ? items.slice(limit) : []);
        var html = renderAccordionSection(t('Properties'), first, { open: true });
        if (rest.length) {
          // Append a second section for the remaining items with a toggle container
          var moreId = 'dpp-more-' + (++DPP_UID);
          html += '<div class="dpp-more" id="' + moreId + '" hidden>' + renderAccordionSection(t('More Properties'), rest, { open: false }) + '</div>';
          html += '<div class="dpp-more-ctl"><button type="button" class="dpp-more-toggle" aria-controls="' + moreId + '" aria-expanded="false" data-more-label="' + esc(moreLabel) + '" data-less-label="' + esc(lessLabel) + '">' + esc(moreLabel) + '</button></div>';
        }
        return html;
      }
      // Generic array of items
      var blocks = data.map(function(item, i){
        var label;
        if (isObject(item) && (item.name || item.title)) { label = String(item.name || item.title); }
        else {
          try {
            if (window.wp && wp.i18n && typeof wp.i18n.sprintf === 'function') {
              label = wp.i18n.sprintf(wp.i18n.__('Item %d', 'digital-product-passport'), (i+1));
            } else {
              label = 'Item ' + (i+1);
            }
          } catch(e){ label = 'Item ' + (i+1); }
        }
        return renderAccordionSection(label, item, { open: i === 0 });
      });
      var firstA = (limit > 0 ? blocks.slice(0, limit) : blocks);
      var restA = (limit > 0 ? blocks.slice(limit) : []);
      var htmlA = firstA.join('');
      if (restA.length) {
        var moreIdA = 'dpp-more-' + (++DPP_UID);
        htmlA += '<div class="dpp-more" id="' + moreIdA + '" hidden>' + restA.join('') + '</div>';
        htmlA += '<div class="dpp-more-ctl"><button type="button" class="dpp-more-toggle" aria-controls="' + moreIdA + '" aria-expanded="false" data-more-label="' + esc(moreLabel) + '" data-less-label="' + esc(lessLabel) + '">' + esc(moreLabel) + '</button></div>';
      }
      return htmlA;
    }

    // Objects
    if (isObject(data)) {
      var keys = Object.keys(data);
      var ordered = orderKeys(keys, data, include, exclude, order);
      var openFirst = 3;
      var sections = ordered.map(function(kobj, idx){
        var open = openFirst > 0; if (openFirst > 0) openFirst--;
        return renderAccordionSection(kobj.label, data[kobj.key], { open: open });
      });
      var first = (limit > 0 ? sections.slice(0, limit) : sections);
      var rest = (limit > 0 ? sections.slice(limit) : []);
      var html = first.join('');
      if (rest.length) {
        var moreId = 'dpp-more-' + (++DPP_UID);
        html += '<div class="dpp-more" id="' + moreId + '" hidden>' + rest.join('') + '</div>';
        html += '<div class="dpp-more-ctl"><button type="button" class="dpp-more-toggle" aria-controls="' + moreId + '" aria-expanded="false" data-more-label="' + esc(moreLabel) + '" data-less-label="' + esc(lessLabel) + '">' + esc(moreLabel) + '</button></div>';
      }
      return html;
    }

    return '<div class="dpp-leaf">' + esc(String(data)) + '</div>';
  }

  function initViewer(root){
    try {
      var script = root.querySelector('script.dpp-data');
      if (!script) return;
      var jsonText = script.textContent || script.innerText || '';
      var data = JSON.parse(jsonText);
      var pretty = root.querySelector('.dpp-pretty');
      var htmlView = root.querySelector('.dpp-html');
      var raw = root.querySelector('.dpp-raw');
      if (pretty) pretty.innerHTML = renderValue(data);
      // Read config from data-attrs or optional script.dpp-config
      var cfg = {
        limit: root.getAttribute('data-limit') || '0',
        include: root.getAttribute('data-include') || '',
        exclude: root.getAttribute('data-exclude') || '',
        order: root.getAttribute('data-order') || '',
        moreLabel: root.getAttribute('data-more-label') || '',
        lessLabel: root.getAttribute('data-less-label') || ''
      };
      var cfgNode = root.querySelector('script.dpp-config');
      if (cfgNode) {
        try { var fromScript = JSON.parse(cfgNode.textContent || cfgNode.innerText || '{}'); if (fromScript && typeof fromScript === 'object') { Object.keys(fromScript).forEach(function(k){ cfg[k] = fromScript[k]; }); } } catch(e){}
      }
      if (htmlView) htmlView.innerHTML = '<div class="dpp-accordion">' + renderWithConfig(data, cfg) + '</div>';
      if (raw) raw.textContent = JSON.stringify(data, null, 2);

      var tabs = Array.prototype.slice.call(root.querySelectorAll('.dpp-tab'));
      var panels = { html: htmlView, raw: raw, pretty: pretty };

      // Derive available views from the rendered tabs
      var availableViews = tabs.map(function(t){ return t.getAttribute('data-view'); }).filter(Boolean);

      // Copy JSON button + status (only if raw panel exists)
      var toolbar = root.querySelector('.dpp-toolbar.dpp-tabs');
      var copyBtn = null;
      var statusNode = null;
      if (toolbar && raw) {
        var labelCopy = (window.wp && wp.i18n) ? wp.i18n.__('Copy JSON', 'digital-product-passport') : 'Copy JSON';
        copyBtn = document.createElement('button');
        copyBtn.type = 'button';
        copyBtn.className = 'dpp-copy';
        // Add Dashicon only when the Dashicons stylesheet is active on the page
        var hasDashicons = !!document.getElementById('dashicons-css');
        if (hasDashicons) {
          copyBtn.innerHTML = '<span class="dashicons dashicons-clipboard" aria-hidden="true"></span><span class="dpp-copy-label">' + labelCopy + '</span>';
        } else {
          copyBtn.textContent = labelCopy;
        }
        copyBtn.setAttribute('aria-label', labelCopy);
        copyBtn.setAttribute('title', labelCopy);
        toolbar.appendChild(copyBtn);
        copyBtn.hidden = true;

        statusNode = document.createElement('span');
        statusNode.className = 'dpp-copy-status';
        statusNode.setAttribute('role', 'status');
        statusNode.setAttribute('aria-live', 'polite');
        statusNode.setAttribute('aria-atomic', 'true');
        statusNode.hidden = true;
        toolbar.appendChild(statusNode);

        function showStatus(msg, ok) {
          statusNode.textContent = msg;
          statusNode.hidden = false;
          statusNode.classList.remove('ok', 'err');
          statusNode.classList.add(ok ? 'ok' : 'err');
          clearTimeout(showStatus._t);
          showStatus._t = setTimeout(function(){ statusNode.hidden = true; }, 2200);
        }

        function copyRawJSON(){
          var text = raw ? raw.textContent : '';
          if (!text) return;
          var okMsg = (window.wp && wp.i18n) ? wp.i18n.__('JSON copied', 'digital-product-passport') : 'JSON copied';
          var errMsg = (window.wp && wp.i18n) ? wp.i18n.__('Copy failed', 'digital-product-passport') : 'Copy failed';
          if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(text).then(function(){ showStatus(okMsg, true); }, function(){ showStatus(errMsg, false); });
          } else {
            try {
              var range = document.createRange();
              range.selectNodeContents(raw);
              var sel = window.getSelection();
              sel.removeAllRanges();
              sel.addRange(range);
              var ok = document.execCommand('copy');
              sel.removeAllRanges();
              showStatus(ok ? okMsg : errMsg, !!ok);
            } catch(e) {
              showStatus(errMsg, false);
            }
          }
        }
        copyBtn.addEventListener('click', copyRawJSON);
      }

      // ARIA wiring: id, labelledby, controls, roving tabindex
      var uid = (++DPP_UID);
      availableViews.forEach(function(view){
        var tab = root.querySelector('.dpp-tab-' + view);
        var panel = panels[view];
        if (!tab || !panel) return;
        var tabId = 'dpp-tab-' + uid + '-' + view;
        var panelId = 'dpp-panel-' + uid + '-' + view;
        tab.id = tabId;
        tab.setAttribute('role','tab');
        tab.setAttribute('aria-controls', panelId);
        tab.setAttribute('tabindex','-1');
        panel.id = panelId;
        panel.setAttribute('role','tabpanel');
        panel.setAttribute('aria-labelledby', tabId);
      });

      function setView(view){
        var views = { pretty: pretty, html: htmlView, raw: raw };
        Object.keys(views).forEach(function(k){ if (views[k]) views[k].hidden = (k !== view); });
        tabs.forEach(function(btn){
          var selected = btn.getAttribute('data-view') === view;
          btn.setAttribute('aria-selected', String(selected));
          btn.setAttribute('tabindex', selected ? '0' : '-1');
        });
        if (copyBtn) { copyBtn.hidden = (view !== 'raw'); }
        if (statusNode && view !== 'raw') { statusNode.hidden = true; }
      }
      var startView = root.getAttribute('data-start-view');
      if (!startView) {
        // BC: fallback to old data-show-raw
        startView = (root.getAttribute('data-show-raw') === 'true') ? 'raw' : 'html';
      }
      if (!/^(html|raw)$/.test(startView)) startView = 'html';
      if (availableViews.indexOf(startView) === -1) {
        startView = availableViews[0] || 'html';
      }
      setView(startView);
      // Mouse
      tabs.forEach(function(btn){ btn.addEventListener('click', function(){ setView(btn.getAttribute('data-view')); btn.focus(); }); });
      // Keyboard: Left/Right/Home/End
      root.addEventListener('keydown', function(e){
        var active = document.activeElement;
        if (!active || tabs.indexOf(active) === -1) return;
        var idx = tabs.indexOf(active);
        var nextIdx = idx;
        if (e.key === 'ArrowRight') nextIdx = (idx + 1) % tabs.length;
        else if (e.key === 'ArrowLeft') nextIdx = (idx - 1 + tabs.length) % tabs.length;
        else if (e.key === 'Home') nextIdx = 0;
        else if (e.key === 'End') nextIdx = tabs.length - 1;
        else return;
        e.preventDefault();
        var nextTab = tabs[nextIdx];
        if (nextTab) {
          setView(nextTab.getAttribute('data-view'));
          nextTab.focus();
        }
      });

      // Show more/less toggles
      function bindMoreToggles(scope){
        var toggles = (scope || root).querySelectorAll('.dpp-more-toggle');
        if (!toggles || !toggles.length) return;
        toggles.forEach(function(btn){
          var targetId = btn.getAttribute('aria-controls');
          var target = targetId ? document.getElementById(targetId) : null;
          var moreLabel = btn.getAttribute('data-more-label') || btn.textContent || ((window.wp && wp.i18n) ? wp.i18n.__('Show more', 'digital-product-passport') : 'Show more');
          var lessLabel = btn.getAttribute('data-less-label') || ((window.wp && wp.i18n) ? wp.i18n.__('Show less', 'digital-product-passport') : 'Show less');
          btn.addEventListener('click', function(){
            if (!target) return;
            var expanded = btn.getAttribute('aria-expanded') === 'true';
            expanded = !expanded;
            btn.setAttribute('aria-expanded', String(expanded));
            target.hidden = !expanded;
            btn.textContent = expanded ? lessLabel : moreLabel;
          });
        });
      }
      bindMoreToggles(root);
    } catch(e) {
      root.innerHTML = '<div class="dpp-error">' + (window.wp && wp.i18n ? wp.i18n.__('Unable to render Digital Product Passport.', 'digital-product-passport') : 'Unable to render Digital Product Passport.') + '</div>';
    }
  }

  function boot(){
    var nodes = document.querySelectorAll('.dpp-viewer');
    nodes.forEach(initViewer);
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', boot);
  } else {
    boot();
  }
})();
