<?php
/**
 * Plugin Name: Dialekt
 * Plugin URI: https://dialekt.io
 * Description: Automatic website translation by Dialekt
 * Version: 1.0.1
 * Author: Dialekt
 * License: GPL v2 or later
 * Text Domain: dialekt
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// ============================================================================
// CONSTANTS
// ============================================================================

define( 'DIALEKT_SCRIPT_URL', 'https://cdn.dialekt.io/dialektweb.js' );
define( 'DIALEKT_OPTION_API_KEY', 'dialekt_api_key' );

// ============================================================================
// HELPERS
// ============================================================================

/**
 * Get sanitized request path from SERVER.
 *
 * @return string Sanitized path without leading/trailing slashes.
 */
function dialekt_get_request_path() {
	if ( ! isset( $_SERVER['REQUEST_URI'] ) ) {
		return '';
	}
	$uri  = sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) );
	$path = wp_parse_url( $uri, PHP_URL_PATH );
	return $path ? trim( $path, '/' ) : '';
}

/**
 * Detect language prefix from current request.
 *
 * @return string|false Two-letter language code or false.
 */
function dialekt_get_lang_from_uri() {
	static $lang = null;
	if ( null !== $lang ) {
		return $lang;
	}

	$path = dialekt_get_request_path();
	$lang = preg_match( '#^([a-z]{2})(?:/|$)#i', $path, $m ) ? strtolower( $m[1] ) : false;
	return $lang;
}

/**
 * Get request path without language prefix.
 *
 * @return string Clean path.
 */
function dialekt_get_clean_path() {
	$path = dialekt_get_request_path();
	return preg_replace( '#^[a-z]{2}(?:/|$)#i', '', $path );
}

// ============================================================================
// ACTIVATION
// ============================================================================

register_activation_hook( __FILE__, function () {
	add_option( 'dialekt_do_activation_redirect', true );
} );

// ============================================================================
// ADMIN
// ============================================================================

add_action( 'admin_init', function () {
	if ( get_option( 'dialekt_do_activation_redirect', false ) ) {
		delete_option( 'dialekt_do_activation_redirect' );
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Standard WP activation redirect pattern.
		if ( ! isset( $_GET['activate-multi'] ) ) {
			wp_safe_redirect( admin_url( 'admin.php?page=dialekt' ) );
			exit;
		}
	}
} );

add_action( 'admin_menu', function () {
	add_menu_page(
		'Dialekt',
		'Dialekt',
		'manage_options',
		'dialekt',
		'dialekt_admin_page',
		'dashicons-translation',
		30
	);
} );

add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), function ( $links ) {
	array_unshift( $links, '<a href="' . esc_url( admin_url( 'admin.php?page=dialekt' ) ) . '">' . esc_html__( 'Settings', 'dialekt' ) . '</a>' );
	return $links;
} );

/**
 * Show admin notice if API key is not configured.
 */
add_action( 'admin_notices', function () {
	$api_key = get_option( DIALEKT_OPTION_API_KEY, '' );
	if ( ! empty( $api_key ) ) {
		return;
	}
	?>
	<div class="notice notice-error">
		<p>
			<?php
			printf(
				/* translators: %s: link to settings page */
				esc_html__( 'Dialekt Translation plugin is installed but not configured yet. Set it up on the %s — it only takes a minute!', 'dialekt' ),
				'<a href="' . esc_url( admin_url( 'admin.php?page=dialekt' ) ) . '">' . esc_html__( 'Dialekt Translation plugin settings page', 'dialekt' ) . '</a>'
			);
			?>
		</p>
	</div>
	<?php
} );

/**
 * Render admin settings page.
 */
function dialekt_admin_page() {
	if ( isset( $_POST['dialekt_save'], $_POST['api_key'] ) && check_admin_referer( 'dialekt_settings' ) ) {
		update_option( DIALEKT_OPTION_API_KEY, sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) );
		echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'dialekt' ) . '</p></div>';
	}

	$api_key = get_option( DIALEKT_OPTION_API_KEY, '' );
	?>
	<div class="wrap">
		<h1><?php echo esc_html__( 'Dialekt', 'dialekt' ); ?></h1>

		<form method="post">
			<?php wp_nonce_field( 'dialekt_settings' ); ?>
			<table class="form-table">
				<tr>
					<th><label for="api_key"><?php echo esc_html__( 'API Key', 'dialekt' ); ?></label></th>
					<td>
						<input type="text" id="api_key" name="api_key"
							   value="<?php echo esc_attr( $api_key ); ?>"
							   class="regular-text" placeholder="dia_xxxxxxxxxxxxx">
						<p class="description">
							<?php
							printf(
								/* translators: %s: link to Dialekt Dashboard */
								esc_html__( 'You can get your API Key from the %s.', 'dialekt' ),
								'<a href="https://dashboard.dialekt.io/signup" target="_blank">Dialekt Dashboard</a>'
							);
							?>
						</p>
					</td>
				</tr>
			</table>
			<?php submit_button( esc_html__( 'Save', 'dialekt' ), 'primary', 'dialekt_save' ); ?>
		</form>
	</div>
	<?php
}

// ============================================================================
// SCRIPT INJECTION (in <head> to prevent flash)
// ============================================================================

add_action( 'wp_head', function () {
	$api_key = get_option( DIALEKT_OPTION_API_KEY, '' );
	if ( empty( $api_key ) ) {
		return;
	}
	?>
	<script data-api-key="<?php echo esc_attr( $api_key ); ?>" data-nav-mode="subdirectories" src="<?php echo esc_url( DIALEKT_SCRIPT_URL ); ?>"></script>
	<?php
}, 1 );

// ============================================================================
// LANGUAGE PREFIX HANDLING
// ============================================================================

add_filter( 'request', function ( $query_vars ) {
	if ( is_admin() ) {
		return $query_vars;
	}

	$lang = dialekt_get_lang_from_uri();
	if ( ! $lang ) {
		return $query_vars;
	}

	$clean_path = dialekt_get_clean_path();

	// Homepage.
	if ( '' === $clean_path ) {
		return array();
	}

	// Try URL to post ID.
	$url     = home_url( '/' . $clean_path );
	$post_id = url_to_postid( $url );
	if ( $post_id ) {
		$post = get_post( $post_id );
		if ( $post ) {
			return array(
				'page_id'   => $post_id,
				'post_type' => $post->post_type,
			);
		}
	}

	// Try as page by path.
	$page = get_page_by_path( $clean_path );
	if ( $page ) {
		return array( 'pagename' => $clean_path );
	}

	// Try as post by name.
	$parts     = explode( '/', $clean_path );
	$post_name = end( $parts );
	$posts     = get_posts( array(
		'name'        => $post_name,
		'post_type'   => 'post',
		'numberposts' => 1,
	) );
	if ( ! empty( $posts ) ) {
		return array(
			'name'      => $post_name,
			'post_type' => 'post',
		);
	}

	// Nothing found - flag for 404.
	if ( ! defined( 'DIALEKT_FORCE_404' ) ) {
		define( 'DIALEKT_FORCE_404', true );
	}
	return array();
}, 1 );

/**
 * Force 404 template when content not found.
 */
add_action( 'template_redirect', function () {
	if ( ! defined( 'DIALEKT_FORCE_404' ) || ! DIALEKT_FORCE_404 ) {
		return;
	}

	global $wp_query;
	$wp_query->set_404();
	status_header( 404 );
	nocache_headers();

	$template = get_404_template();
	if ( ! $template ) {
		$template = get_index_template();
	}
	if ( $template ) {
		include $template;
		exit;
	}
} );
