<?php
/**
 * Provide a public-facing view for the plugin
 *
 * This file is used to rename uploaded files based on the plugin settings.
 *
 * @link       https://wordpress.org/plugins/dh-rename-uploaded-files/
 * @since      1.0.0
 *
 * @package    dh-rename-uploaded-files
 * @subpackage dh-rename-uploaded-files/public/partials
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Get plugin options
$options = get_option('dh_renameuploadedfiles');

if (isset($options['dhruf_enable'])) {
    /**
     * Rename uploaded files based on plugin settings
     *
     * @param string $dhruf_filename The original filename
     * @return string The processed filename
     */
    function dh__ruf($dhruf_filename) {
        if (empty($dhruf_filename)) {
            return $dhruf_filename;
        }

        $options = get_option('dh_renameuploadedfiles');
        $dhruf_info = pathinfo($dhruf_filename);
        $dhruf_ext = !empty($dhruf_info['extension']) ? '.' . $dhruf_info['extension'] : '';
        $dhruf_name = basename($dhruf_filename, $dhruf_ext);

        switch ($options['dhruf_renametype'] ?? 'norename') {
            case 'filename':
                $dhruf_namefile = pathinfo($dhruf_filename, PATHINFO_FILENAME);
                $dhruf_random = substr(str_shuffle($dhruf_namefile), 0);
                $dhruf_append = apply_filters('dh__ruf', true, $dhruf_filename, $dhruf_random);
                
                if (!$dhruf_append) {
                    return $dhruf_filename;
                }
                
                $dhruf_name = basename($dhruf_random, $dhruf_ext);
                break;

            case 'filenamerename':
                $dhruf_namefile = pathinfo($dhruf_filename, PATHINFO_FILENAME);
                $dhruf_random = substr(str_shuffle($dhruf_namefile), 0);
                $dhruf_append = apply_filters('dh__ruf', true, $dhruf_filename, $dhruf_random);
                
                if (!$dhruf_append) {
                    return $dhruf_filename;
                }
                
                $dhruf_name = $dhruf_name . '_' . $dhruf_random;
                break;

            case 'allrename':
                $dhruf_namefile = pathinfo($dhruf_filename, PATHINFO_FILENAME);
                $dhruf_namefiles = str_replace(
                    array('[', ']', '(', ')', '{', '}', '-', '_', '.', ',', '/', '|', '+', '^', '&', '%', '#', '$'),
                    '',
                    $dhruf_namefile
                );
                
                $dhruf_alphabet = 'abcdefghijklmnopqrstuvwxyz';
                $dhruf_numeric = '1234567890';
                $dhruf_random = substr(str_shuffle($dhruf_namefiles . $dhruf_alphabet . $dhruf_numeric), 0, 30);
                $dhruf_append = apply_filters('dh__ruf', true, $dhruf_filename, $dhruf_random);
                
                if (!$dhruf_append) {
                    return $dhruf_filename;
                }
                
                $dhruf_name = basename($dhruf_random, $dhruf_ext);
                break;
                
            case 'norename':
            default:
                // Keep original filename
                break;
        }

        return $dhruf_name . $dhruf_ext;
    }

    add_filter('sanitize_file_name', 'dh__ruf', 10);
}