<?php
/**
 * Admin settings for DevLab Maintenance Mode
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MMBD_Admin {

    private $settings;

    public function __construct() {
        $this->settings = mmbd_get_settings();

        add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );

        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        add_action( 'admin_notices', array( $this, 'admin_notice' ) );
        add_filter( 'plugin_action_links_' . MMBD_PLUGIN_BASENAME, array( $this, 'add_settings_link' ) );
    }

    public function add_settings_page() {
        add_options_page(
            __( 'DevLab Maintenance Mode', 'devlab-maintenance-mode' ),
            __( 'Maintenance Mode', 'devlab-maintenance-mode' ),
            'manage_options',
            'devlab-maintenance-mode',
            array( $this, 'render_settings_page' )
        );
    }

    public function register_settings() {
        register_setting(
            'mmbd_settings_group',
            'mmbd_settings',
            array(
                'type'              => 'array',
                'sanitize_callback' => array( $this, 'sanitize_settings' ),
                'default'           => mmbd_get_default_settings(),
            )
        );

        add_settings_section( 'mmbd_main_section', __( 'Maintenance Mode Settings', 'devlab-maintenance-mode' ), array( $this, 'render_main_section' ), 'devlab-maintenance-mode' );
        add_settings_field( 'mmbd_enabled', __( 'Enable Maintenance Mode', 'devlab-maintenance-mode' ), array( $this, 'render_enabled_field' ), 'devlab-maintenance-mode', 'mmbd_main_section' );
        add_settings_field( 'mmbd_message', __( 'Maintenance Message', 'devlab-maintenance-mode' ), array( $this, 'render_message_field' ), 'devlab-maintenance-mode', 'mmbd_main_section' );

        add_settings_section( 'mmbd_template_section', __( 'Template Selection', 'devlab-maintenance-mode' ), array( $this, 'render_template_section' ), 'devlab-maintenance-mode' );
        add_settings_field( 'mmbd_template', __( 'Choose Template', 'devlab-maintenance-mode' ), array( $this, 'render_template_selection_field' ), 'devlab-maintenance-mode', 'mmbd_template_section' );

        add_settings_field( 'mmbd_countdown_date', __( 'Countdown Date', 'devlab-maintenance-mode' ), array( $this, 'render_countdown_date_field' ), 'devlab-maintenance-mode', 'mmbd_template_section' );

        add_settings_section( 'mmbd_bypass_section', __( 'Bypass Options', 'devlab-maintenance-mode' ), array( $this, 'render_bypass_section' ), 'devlab-maintenance-mode' );
        add_settings_field( 'mmbd_bypass_roles', __( 'Bypass Roles', 'devlab-maintenance-mode' ), array( $this, 'render_bypass_roles_field' ), 'devlab-maintenance-mode', 'mmbd_bypass_section' );
        add_settings_field( 'mmbd_ip_whitelist', __( 'IP Whitelist', 'devlab-maintenance-mode' ), array( $this, 'render_ip_whitelist_field' ), 'devlab-maintenance-mode', 'mmbd_bypass_section' );
        add_settings_field( 'mmbd_secret_phrase', __( 'Secret Phrase', 'devlab-maintenance-mode' ), array( $this, 'render_secret_phrase_field' ), 'devlab-maintenance-mode', 'mmbd_bypass_section' );
        add_settings_field( 'mmbd_cookie_duration', __( 'Cookie Duration', 'devlab-maintenance-mode' ), array( $this, 'render_cookie_duration_field' ), 'devlab-maintenance-mode', 'mmbd_bypass_section' );

        add_settings_section( 'mmbd_modules_section', __( 'Modules', 'devlab-maintenance-mode' ), array( $this, 'render_modules_section' ), 'devlab-maintenance-mode' );
        add_settings_field( 'mmbd_analytics_id', __( 'Google Analytics ID', 'devlab-maintenance-mode' ), array( $this, 'render_analytics_id_field' ), 'devlab-maintenance-mode', 'mmbd_modules_section' );
        add_settings_field( 'mmbd_social_facebook', __( 'Facebook URL', 'devlab-maintenance-mode' ), array( $this, 'render_social_facebook_field' ), 'devlab-maintenance-mode', 'mmbd_modules_section' );
        add_settings_field( 'mmbd_social_twitter', __( 'Twitter / X URL', 'devlab-maintenance-mode' ), array( $this, 'render_social_twitter_field' ), 'devlab-maintenance-mode', 'mmbd_modules_section' );
        add_settings_field( 'mmbd_social_instagram', __( 'Instagram URL', 'devlab-maintenance-mode' ), array( $this, 'render_social_instagram_field' ), 'devlab-maintenance-mode', 'mmbd_modules_section' );
        add_settings_field( 'mmbd_social_linkedin', __( 'LinkedIn URL', 'devlab-maintenance-mode' ), array( $this, 'render_social_linkedin_field' ), 'devlab-maintenance-mode', 'mmbd_modules_section' );
    }

    public function sanitize_settings( $input ) {
        $sanitized = array();

        $sanitized['enabled'] = ! empty( $input['enabled'] );
        $sanitized['message'] = isset( $input['message'] ) ? wp_kses_post( $input['message'] ) : mmbd_get_default_settings()['message'];

        $sanitized['bypass_roles'] = array();
        if ( isset( $input['bypass_roles'] ) && is_array( $input['bypass_roles'] ) ) {
            $wp_roles = wp_roles()->get_names();
            foreach ( $input['bypass_roles'] as $role ) {
                $role = sanitize_key( $role );
                if ( array_key_exists( $role, $wp_roles ) ) {
                    $sanitized['bypass_roles'][] = $role;
                }
            }
        }

        $sanitized['ip_whitelist'] = '';
        if ( isset( $input['ip_whitelist'] ) ) {
            $ips = array_filter( array_map( 'trim', explode( "\n", $input['ip_whitelist'] ) ) );
            $valid_ips = array();
            foreach ( $ips as $ip ) {
                $ip = sanitize_text_field( $ip );
                if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
                    $valid_ips[] = $ip;
                }
            }
            $sanitized['ip_whitelist'] = implode( "\n", $valid_ips );
        }

        $sanitized['secret_phrase'] = isset( $input['secret_phrase'] ) ? sanitize_text_field( $input['secret_phrase'] ) : '';

        $sanitized['cookie_duration'] = isset( $input['cookie_duration'] ) ? absint( $input['cookie_duration'] ) : 1;
        $sanitized['cookie_duration'] = max( 1, min( 30, $sanitized['cookie_duration'] ) );

        $sanitized['template'] = isset( $input['template'] ) ? sanitize_key( $input['template'] ) : 'classic';
        $sanitized['template'] = MMBD_Templates::validate_template_id( $sanitized['template'] );

        $sanitized['countdown_date'] = '';
        if ( ! empty( $input['countdown_date'] ) ) {
            $sanitized['countdown_date'] = sanitize_text_field( $input['countdown_date'] );
        }

        $sanitized['analytics_id'] = '';
        if ( ! empty( $input['analytics_id'] ) ) {
            $sanitized['analytics_id'] = sanitize_text_field( $input['analytics_id'] );
        }

        $sanitized['social_facebook']  = ! empty( $input['social_facebook'] ) ? esc_url_raw( $input['social_facebook'] ) : '';
        $sanitized['social_twitter']   = ! empty( $input['social_twitter'] ) ? esc_url_raw( $input['social_twitter'] ) : '';
        $sanitized['social_instagram'] = ! empty( $input['social_instagram'] ) ? esc_url_raw( $input['social_instagram'] ) : '';
        $sanitized['social_linkedin']  = ! empty( $input['social_linkedin'] ) ? esc_url_raw( $input['social_linkedin'] ) : '';

        $sanitized['version'] = MMBD_VERSION;

        return $sanitized;
    }

    public function enqueue_admin_assets( $hook ) {
        if ( 'settings_page_devlab-maintenance-mode' !== $hook ) {
            return;
        }
        wp_enqueue_style( 'mmbd-admin', MMBD_PLUGIN_URL . 'assets/css/admin.css', array(), MMBD_VERSION );
        wp_enqueue_script( 'mmbd-admin', MMBD_PLUGIN_URL . 'assets/js/admin.js', array(), MMBD_VERSION, true );
    }

    public function admin_notice() {
        if ( empty( $this->settings['enabled'] ) ) {
            return;
        }
        $settings_url = admin_url( 'options-general.php?page=devlab-maintenance-mode' );
        ?>
        <div class="notice notice-warning is-dismissible">
            <p>
                <strong><?php esc_html_e( 'Maintenance Mode is Active', 'devlab-maintenance-mode' ); ?></strong> &mdash;
                <?php esc_html_e( 'Your site is currently showing a maintenance page to visitors.', 'devlab-maintenance-mode' ); ?>
                <a href="<?php echo esc_url( $settings_url ); ?>"><?php esc_html_e( 'Manage Settings', 'devlab-maintenance-mode' ); ?></a>
            </p>
        </div>
        <?php
    }

    public function add_settings_link( $links ) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url( 'options-general.php?page=devlab-maintenance-mode' ),
            __( 'Settings', 'devlab-maintenance-mode' )
        );
        array_unshift( $links, $settings_link );
        return $links;
    }

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $this->settings = mmbd_get_settings();
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            <?php $this->render_status_indicator(); ?>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'mmbd_settings_group' );
                do_settings_sections( 'devlab-maintenance-mode' );
                submit_button( __( 'Save Settings', 'devlab-maintenance-mode' ) );
                ?>
            </form>
            <?php $this->render_preview_section(); ?>

        </div>
        <?php
    }

    private function render_status_indicator() {
        $enabled = ! empty( $this->settings['enabled'] );
        $class = $enabled ? 'mmbd-status-on' : 'mmbd-status-off';
        $text = $enabled
            ? __( 'Maintenance Mode is ON - Site is not accessible to visitors', 'devlab-maintenance-mode' )
            : __( 'Maintenance Mode is OFF - Site is accessible to everyone', 'devlab-maintenance-mode' );
        ?>
        <div class="mmbd-status-indicator <?php echo esc_attr( $class ); ?>">
            <span class="mmbd-status-dot"></span>
            <span class="mmbd-status-text"><?php echo esc_html( $text ); ?></span>
        </div>
        <?php
    }

    private function render_preview_section() {
        ?>
        <div class="mmbd-preview-section">
            <h2><?php esc_html_e( 'Preview', 'devlab-maintenance-mode' ); ?></h2>
            <p><?php esc_html_e( 'Preview how your maintenance page looks to visitors.', 'devlab-maintenance-mode' ); ?></p>
            <a href="<?php echo esc_url( home_url( '?mmbd_preview=1' ) ); ?>" class="button button-secondary" target="_blank" rel="noopener noreferrer">
                <?php esc_html_e( 'Preview Maintenance Page', 'devlab-maintenance-mode' ); ?>
            </a>
            <p class="description">
                <?php esc_html_e( 'Note: Preview opens in a new tab. Since you\'re logged in as admin, use incognito mode to see the actual maintenance page when enabled.', 'devlab-maintenance-mode' ); ?>
            </p>
        </div>
        <?php
    }


    public function render_main_section() {
        echo '<p>' . esc_html__( 'Configure your maintenance mode settings below.', 'devlab-maintenance-mode' ) . '</p>';
    }

    public function render_modules_section() {
        echo '<p>' . esc_html__( 'Add extra functionality to your maintenance page.', 'devlab-maintenance-mode' ) . '</p>';
    }

    public function render_analytics_id_field() {
        $id = isset( $this->settings['analytics_id'] ) ? $this->settings['analytics_id'] : '';
        ?>
        <input type="text" id="mmbd_analytics_id" name="mmbd_settings[analytics_id]" value="<?php echo esc_attr( $id ); ?>" class="regular-text" placeholder="G-XXXXXXXXXX">
        <p class="description">
            <?php esc_html_e( 'Enter your Google Analytics Measurement ID (e.g., G-123456789).', 'devlab-maintenance-mode' ); ?>
        </p>
        <?php
    }

    public function render_social_facebook_field() {
        $url = isset( $this->settings['social_facebook'] ) ? $this->settings['social_facebook'] : '';
        ?>
        <input type="url" id="mmbd_social_facebook" name="mmbd_settings[social_facebook]" value="<?php echo esc_attr( $url ); ?>" class="regular-text">
        <?php
    }

    public function render_social_twitter_field() {
        $url = isset( $this->settings['social_twitter'] ) ? $this->settings['social_twitter'] : '';
        ?>
        <input type="url" id="mmbd_social_twitter" name="mmbd_settings[social_twitter]" value="<?php echo esc_attr( $url ); ?>" class="regular-text">
        <?php
    }

    public function render_social_instagram_field() {
        $url = isset( $this->settings['social_instagram'] ) ? $this->settings['social_instagram'] : '';
        ?>
        <input type="url" id="mmbd_social_instagram" name="mmbd_settings[social_instagram]" value="<?php echo esc_attr( $url ); ?>" class="regular-text">
        <?php
    }

    public function render_social_linkedin_field() {
        $url = isset( $this->settings['social_linkedin'] ) ? $this->settings['social_linkedin'] : '';
        ?>
        <input type="url" id="mmbd_social_linkedin" name="mmbd_settings[social_linkedin]" value="<?php echo esc_attr( $url ); ?>" class="regular-text">
        <?php
    }

    public function render_bypass_section() {
        echo '<p>' . esc_html__( 'Configure who can bypass maintenance mode and access the site normally.', 'devlab-maintenance-mode' ) . '</p>';
    }

    public function render_template_section() {
        echo '<p>' . esc_html__( 'Choose a template design for your maintenance page.', 'devlab-maintenance-mode' ) . '</p>';
    }

    public function render_template_selection_field() {
        $templates = MMBD_Templates::get_templates();
        $selected = isset( $this->settings['template'] ) ? $this->settings['template'] : 'classic';
        $selected = MMBD_Templates::validate_template_id( $selected );
        ?>
        <div class="mmbd-template-grid">
            <?php foreach ( $templates as $template_id => $template ) : ?>
                <label class="mmbd-template-card <?php echo esc_attr( $selected === $template_id ? 'mmbd-template-selected' : '' ); ?>">
                    <input type="radio" name="mmbd_settings[template]" value="<?php echo esc_attr( $template_id ); ?>" <?php checked( $selected, $template_id ); ?> class="mmbd-template-radio">
                    <div class="mmbd-template-preview">
                        <img src="<?php echo esc_url( MMBD_Templates::get_preview_url( $template_id ) ); ?>" alt="<?php echo esc_attr( $template['name'] ); ?>" loading="lazy">
                        <span class="mmbd-template-checkmark">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3">
                                <polyline points="20 6 9 17 4 12"></polyline>
                            </svg>
                        </span>
                    </div>
                    <div class="mmbd-template-info">
                        <span class="mmbd-template-name"><?php echo esc_html( $template['name'] ); ?></span>
                        <span class="mmbd-template-desc"><?php echo esc_html( $template['description'] ); ?></span>
                    </div>
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }

    public function render_countdown_date_field() {
        $countdown_date = isset( $this->settings['countdown_date'] ) ? $this->settings['countdown_date'] : '';
        ?>
        <input type="datetime-local" id="mmbd_countdown_date" name="mmbd_settings[countdown_date]" value="<?php echo esc_attr( $countdown_date ); ?>" class="regular-text">
        <p class="description">
            <?php esc_html_e( 'Set the target date and time for the countdown timer.', 'devlab-maintenance-mode' ); ?>
        </p>
        <?php
    }

    public function render_enabled_field() {
        $enabled = ! empty( $this->settings['enabled'] );
        ?>
        <label for="mmbd_enabled">
            <input type="checkbox" id="mmbd_enabled" name="mmbd_settings[enabled]" value="1" <?php checked( $enabled ); ?> />
            <?php esc_html_e( 'Enable maintenance mode', 'devlab-maintenance-mode' ); ?>
        </label>
        <p class="description">
            <?php esc_html_e( 'When enabled, visitors will see the maintenance page. Logged-in users with bypass roles can still access the site.', 'devlab-maintenance-mode' ); ?>
        </p>
        <?php
    }

    public function render_message_field() {
        $message = isset( $this->settings['message'] ) ? $this->settings['message'] : '';
        ?>
        <textarea id="mmbd_message" name="mmbd_settings[message]" class="large-text code" rows="6"><?php echo esc_textarea( $message ); ?></textarea>
        <p class="description">
            <?php esc_html_e( 'HTML is allowed. Use &lt;h1&gt;, &lt;p&gt;, &lt;a&gt; tags to format your message.', 'devlab-maintenance-mode' ); ?>
        </p>
        <?php
    }

    public function render_bypass_roles_field() {
        $selected_roles = isset( $this->settings['bypass_roles'] ) ? $this->settings['bypass_roles'] : array();
        $wp_roles = wp_roles()->get_names();
        ?>
        <fieldset>
            <?php foreach ( $wp_roles as $role_key => $role_name ) : ?>
                <label class="mmbd-bypass-role-label">
                    <input type="checkbox" name="mmbd_settings[bypass_roles][]" value="<?php echo esc_attr( $role_key ); ?>" <?php checked( in_array( $role_key, $selected_roles, true ) ); ?> />
                    <?php echo esc_html( translate_user_role( $role_name ) ); ?>
                </label>
            <?php endforeach; ?>
        </fieldset>
        <p class="description">
            <?php esc_html_e( 'Select which user roles can bypass maintenance mode. Administrators are recommended.', 'devlab-maintenance-mode' ); ?>
        </p>
        <?php
    }

    public function render_ip_whitelist_field() {
        $whitelist = isset( $this->settings['ip_whitelist'] ) ? $this->settings['ip_whitelist'] : '';
        $current_ip = $this->get_current_ip();
        ?>
        <textarea id="mmbd_ip_whitelist" name="mmbd_settings[ip_whitelist]" class="regular-text code" rows="4" placeholder="192.168.1.1\n10.0.0.1"><?php echo esc_textarea( $whitelist ); ?></textarea>
        <p class="description">
            <?php esc_html_e( 'Enter one IP address per line. These IPs can always access the site.', 'devlab-maintenance-mode' ); ?>
            <?php if ( $current_ip ) : ?>
                <br>
                <?php
                /* translators: %s: IP address */
                printf( esc_html__( 'Your current IP address: %s', 'devlab-maintenance-mode' ), '<code>' . esc_html( $current_ip ) . '</code>' );
                ?>
            <?php endif; ?>
        </p>
        <?php
    }

    public function render_secret_phrase_field() {
        $phrase = isset( $this->settings['secret_phrase'] ) ? $this->settings['secret_phrase'] : '';
        ?>
        <input type="text" id="mmbd_secret_phrase" name="mmbd_settings[secret_phrase]" value="<?php echo esc_attr( $phrase ); ?>" class="regular-text">
        <p class="description">
            <?php esc_html_e( 'Optional. Visitors can access the site by adding ?mmbd_bypass=YOUR_PHRASE to any URL. A cookie will be set to remember their access.', 'devlab-maintenance-mode' ); ?>
            <?php if ( $phrase ) : ?>
                <br>
                <?php
                /* translators: %s: bypass URL */
                printf( esc_html__( 'Bypass URL: %s', 'devlab-maintenance-mode' ), '<code>' . esc_url( home_url( '?mmbd_bypass=' . rawurlencode( $phrase ) ) ) . '</code>' );
                ?>
            <?php endif; ?>
        </p>
        <?php
    }

    public function render_cookie_duration_field() {
        $duration = isset( $this->settings['cookie_duration'] ) ? $this->settings['cookie_duration'] : 1;
        ?>
        <input type="number" id="mmbd_cookie_duration" name="mmbd_settings[cookie_duration]" value="<?php echo esc_attr( $duration ); ?>" min="1" max="30" class="small-text">
        <?php esc_html_e( 'days', 'devlab-maintenance-mode' ); ?>
        <p class="description">
            <?php esc_html_e( 'How long the bypass cookie lasts after using the secret phrase (1-30 days).', 'devlab-maintenance-mode' ); ?>
        </p>
        <?php
    }

    private function get_current_ip() {
        $headers = array( 'HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR' );

        foreach ( $headers as $header ) {
            if ( ! empty( $_SERVER[ $header ] ) ) {
                $ip = sanitize_text_field( wp_unslash( $_SERVER[ $header ] ) );
                if ( strpos( $ip, ',' ) !== false ) {
                    $ip = trim( explode( ',', $ip )[0] );
                }
                if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
                    return $ip;
                }
            }
        }
        return '';
    }
}

new MMBD_Admin();