<?php
/**
 * Класс транслитератора
 * 
 * @package DevBrothers_Cyrillic_Slugs
 */

// Защита от прямого доступа
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Класс для транслитерации кириллических символов в латиницу
 */
class DBCS_Transliterator {
    
    /**
     * Таблица транслитерации ISO 9 для русского языка
     * @var array
     */
    private $translit_table = [];
    
    /**
     * Конструктор
     */
    public function __construct() {
        $this->init_translit_table();
        $this->init_hooks();
    }
    
    /**
     * Инициализация таблицы транслитерации ISO 9
     */
    private function init_translit_table() {
        $this->translit_table = [
            // Строчные буквы
            'а' => 'a',
            'б' => 'b',
            'в' => 'v',
            'г' => 'g',
            'д' => 'd',
            'е' => 'e',
            'ё' => 'yo',
            'ж' => 'zh',
            'з' => 'z',
            'и' => 'i',
            'й' => 'j',
            'к' => 'k',
            'л' => 'l',
            'м' => 'm',
            'н' => 'n',
            'о' => 'o',
            'п' => 'p',
            'р' => 'r',
            'с' => 's',
            'т' => 't',
            'у' => 'u',
            'ф' => 'f',
            'х' => 'h',
            'ц' => 'c',
            'ч' => 'ch',
            'ш' => 'sh',
            'щ' => 'shh',
            'ъ' => '',
            'ы' => 'y',
            'ь' => '',
            'э' => 'e',
            'ю' => 'yu',
            'я' => 'ya',
            
            // Заглавные буквы
            'А' => 'A',
            'Б' => 'B',
            'В' => 'V',
            'Г' => 'G',
            'Д' => 'D',
            'Е' => 'E',
            'Ё' => 'Yo',
            'Ж' => 'Zh',
            'З' => 'Z',
            'И' => 'I',
            'Й' => 'J',
            'К' => 'K',
            'Л' => 'L',
            'М' => 'M',
            'Н' => 'N',
            'О' => 'O',
            'П' => 'P',
            'Р' => 'R',
            'С' => 'S',
            'Т' => 'T',
            'У' => 'U',
            'Ф' => 'F',
            'Х' => 'H',
            'Ц' => 'C',
            'Ч' => 'Ch',
            'Ш' => 'Sh',
            'Щ' => 'Shh',
            'Ъ' => '',
            'Ы' => 'Y',
            'Ь' => '',
            'Э' => 'E',
            'Ю' => 'Yu',
            'Я' => 'Ya',
            
            // Дополнительные символы
            '№' => '',
            '«' => '',
            '»' => '',
            '—' => '-',
            '–' => '-',
        ];
        
        // Позволяем изменить таблицу через фильтр
        $this->translit_table = apply_filters('dbcs_translit_table', $this->translit_table);
    }
    
    /**
     * Инициализация хуков
     * 
     * ВАЖНО: Фильтры sanitize_title, sanitize_file_name и wp_insert_post_data применяются глобально,
     * но методы-обработчики содержат строгие проверки контекста, чтобы транслитерация
     * применялась только когда это действительно необходимо:
     * - только для включенных типов записей в настройках плагина
     * - только в админке или REST API (не на фронтенде)
     * - только для текста с кириллицей
     * - если тип записи не может быть определен, транслитерация НЕ применяется
     * 
     * Это предотвращает изменение глобального поведения WordPress когда контекст неясен.
     */
    private function init_hooks() {
        // Приоритет 9 чтобы выполниться раньше стандартной обработки WordPress
        // Фильтры применяются глобально, но каждый метод содержит строгие проверки контекста
        // и возвращает оригинальное значение без изменений, если транслитерация не нужна
        add_filter('sanitize_title', [$this, 'transliterate_title'], 9, 3);
        add_filter('sanitize_file_name', [$this, 'transliterate_filename'], 9);
        
        // Дополнительный хук для REST API - гарантированно срабатывает при сохранении
        add_filter('wp_insert_post_data', [$this, 'transliterate_post_slug'], 10, 2);
    }
    
    /**
     * Транслитерация slug при сохранении записи (для REST API и Gutenberg)
     * 
     * ВАЖНО: Метод возвращает данные без изменений, если:
     * - это ревизия или auto-draft
     * - тип записи не включен в настройках
     * - данные не содержат кириллицу
     * 
     * @param array $data Данные записи
     * @param array $postarr Массив аргументов
     * @return array Модифицированные данные или оригинал если транслитерация не нужна
     */
    public function transliterate_post_slug($data, $postarr) {
        // Пропускаем ревизии
        if (isset($data['post_type']) && $data['post_type'] === 'revision') {
            return $data;
        }
        
        // Пропускаем auto-draft (первичное создание)
        if (isset($data['post_status']) && $data['post_status'] === 'auto-draft') {
            return $data;
        }
        
        // Проверяем, включен ли данный тип записи
        $settings = get_option('dbcs_settings', []);
        if (empty($settings['post_types'])) {
            $settings['post_types'] = ['post', 'page'];
        }
        
        // Если тип записи не указан или не включен в настройках, НЕ изменяем данные
        if (empty($data['post_type']) || !in_array($data['post_type'], $settings['post_types'])) {
            return $data;
        }
        
        $slug = isset($data['post_name']) ? $data['post_name'] : '';
        $title = isset($data['post_title']) ? $data['post_title'] : '';
        $post_status = isset($data['post_status']) ? $data['post_status'] : '';
        
        // Декодируем URL-encoded slug
        $decoded_slug = urldecode($slug);
        
        // Случай 1: Slug содержит кириллицу - транслитерируем
        if (!empty($decoded_slug) && $this->has_cyrillic($decoded_slug)) {
            $data['post_name'] = $this->transliterate($decoded_slug);
            return $data;
        }
        
        // Случай 2: Slug пустой, есть кириллический заголовок, не draft
        // При публикации генерируем slug из заголовка
        if (empty($slug) && !empty($title) && $this->has_cyrillic($title)) {
            if (!in_array($post_status, ['auto-draft', 'draft'], true)) {
                $data['post_name'] = $this->transliterate($title);
            }
            return $data;
        }
        
        // Случай 3: При публикации проверяем, не был ли slug сгенерирован из статуса "Черновик"
        // Если slug латинский но заголовок кириллический - возможно WordPress сгенерировал slug неправильно
        if ($post_status === 'publish' && !empty($title) && $this->has_cyrillic($title)) {
            // Генерируем правильный slug из заголовка
            $correct_slug = $this->transliterate($title);
            
            // Если текущий slug не соответствует транслитерации заголовка и это короткий автосгенерированный slug
            if (!empty($decoded_slug) && $decoded_slug !== $correct_slug) {
                // Проверяем типичные автосгенерированные slug WordPress
                $auto_slugs = ['chernovik', 'draft', 'auto-draft', 'untitled'];
                $slug_lower = strtolower($decoded_slug);
                
                // Также проверяем slug вида "chernovik-2", "draft-3" и т.д.
                $is_auto_slug = false;
                foreach ($auto_slugs as $auto) {
                    if ($slug_lower === $auto || preg_match('/^' . preg_quote($auto, '/') . '-\d+$/', $slug_lower)) {
                        $is_auto_slug = true;
                        break;
                    }
                }
                
                if ($is_auto_slug) {
                    $data['post_name'] = $correct_slug;
                }
            }
        }
        
        return $data;
    }
    
    /**
     * Основная функция транслитерации
     * 
     * @param string $text Текст для транслитерации
     * @return string Транслитерированный текст
     */
    public function transliterate($text) {
        if (empty($text)) {
            return $text;
        }
        
        // Сохраняем оригинал для логирования
        $original = $text;
        
        // Применяем таблицу транслитерации
        $text = strtr($text, $this->translit_table);
        
        // Приводим к нижнему регистру
        $text = mb_strtolower($text, 'UTF-8');
        
        // Убираем все кроме латинских букв, цифр, пробелов и дефисов
        $text = preg_replace('/[^a-z0-9\s\-]/', '', $text);
        
        // Заменяем множественные пробелы и дефисы на один дефис
        $text = preg_replace('/[\s\-]+/', '-', $text);
        
        // Убираем дефисы в начале и конце
        $text = trim($text, '-');
        
        // Позволяем изменить результат через фильтр
        $text = apply_filters('dbcs_transliterate_result', $text, $original);
        
        return $text;
    }
    
    /**
     * Транслитерация заголовка (slug)
     * 
     * Фильтр применяется глобально, но содержит строгие проверки:
     * 1. Проверка контекста (только админка/REST API, не фронтенд)
     * 2. Проверка наличия кириллицы (ранний выход если нет)
     * 3. Проверка включенного типа записи в настройках
     * 
     * ВАЖНО: Метод возвращает оригинальное значение без изменений, если:
     * - контекст не подходит для транслитерации
     * - текст не содержит кириллицу
     * - тип записи не включен в настройках или не может быть определен
     * 
     * @param string $title Заголовок
     * @param string $raw_title Сырой заголовок
     * @param string $context Контекст
     * @return string Транслитерированный заголовок или оригинал если транслитерация не нужна
     */
    public function transliterate_title($title, $raw_title = '', $context = 'save') {
        // Ранняя проверка: если контекст запроса, не транслитерируем
        if ($context === 'query') {
            return $title;
        }
        
        // Используем raw_title если есть, иначе title
        $text = !empty($raw_title) ? $raw_title : $title;
        
        // Ранняя проверка наличия кириллицы - если нет, сразу возвращаем без дополнительных проверок
        if (!$this->has_cyrillic($text)) {
            return $title;
        }
        
        // Проверяем, нужно ли транслитерировать в данном контексте
        if (!$this->should_transliterate($context)) {
            return $title;
        }
        
        // Проверяем тип записи из контекста
        // Если тип не определен или не включен, НЕ изменяем глобальное поведение
        if (!$this->is_post_type_enabled()) {
            return $title;
        }
        
        return $this->transliterate($text);
    }
    
    /**
     * Транслитерация имени файла
     * 
     * Фильтр применяется глобально, но содержит строгие проверки:
     * 1. Ранняя проверка наличия кириллицы (выход если нет)
     * 2. Проверка настройки attachment в опциях плагина
     * 3. Применяется только для включенных типов записей
     * 
     * @param string $filename Имя файла
     * @return string Транслитерированное имя файла
     */
    public function transliterate_filename($filename) {
        // Ранняя проверка наличия кириллицы - если нет, сразу возвращаем без дополнительных проверок
        if (!$this->has_cyrillic($filename)) {
            return $filename;
        }
        
        // Проверяем настройки для attachment - только если включен в настройках плагина
        $settings = get_option('dbcs_settings', []);
        if (empty($settings['post_types']) || !in_array('attachment', $settings['post_types'])) {
            return $filename;
        }
        
        // Отделяем расширение файла
        $pathinfo = pathinfo($filename);
        $extension = isset($pathinfo['extension']) ? '.' . $pathinfo['extension'] : '';
        $basename = isset($pathinfo['filename']) ? $pathinfo['filename'] : $filename;
        
        // Транслитерируем базовое имя
        $transliterated = $this->transliterate($basename);
        
        return $transliterated . $extension;
    }
    
    /**
     * Проверка наличия кириллических символов в тексте
     * 
     * @param string $text Текст для проверки
     * @return bool true если есть кириллица
     */
    private function has_cyrillic($text) {
        return preg_match('/[а-яёА-ЯЁ]/u', $text) === 1;
    }
    
    /**
     * Проверка, нужно ли транслитерировать в данном контексте
     * 
     * @param string $context Контекст
     * @return bool true если нужно транслитерировать
     */
    private function should_transliterate($context) {
        // Не транслитерируем при запросах (поиск, фильтрация)
        if ($context === 'query') {
            return false;
        }
        
        // Проверяем, это REST API запрос (Gutenberg использует REST API)
        $is_rest = defined('REST_REQUEST') && REST_REQUEST;
        
        // Не транслитерируем во фронтенде при обработке запросов
        // НО разрешаем для REST API запросов (Gutenberg редактор)
        if (!is_admin() && !$is_rest && did_action('wp')) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Проверка, включен ли текущий тип записи в настройках
     * 
     * @return bool true если включен
     */
    private function is_post_type_enabled() {
        // Получаем настройки
        $settings = get_option('dbcs_settings', []);
        
        // Если настроек нет, включаем по умолчанию для post и page
        if (empty($settings['post_types'])) {
            $settings['post_types'] = ['post', 'page'];
        }
        
        // Пытаемся определить текущий тип записи
        $post_type = $this->get_current_post_type();
        
        // Если не удалось определить тип записи, НЕ транслитерируем
        // Это предотвращает изменение глобального поведения WordPress
        // когда контекст неясен
        if ($post_type === null) {
            return false;
        }
        
        // Проверяем, включен ли данный тип записи
        return in_array($post_type, $settings['post_types']);
    }
    
    /**
     * Определение текущего типа записи
     * 
     * @return string|null Тип записи или null
     */
    private function get_current_post_type() {
        global $post, $typenow, $current_screen;
        
        // Пробуем различные способы определения типа записи
        if ($post && isset($post->post_type)) {
            return $post->post_type;
        }
        
        if ($typenow) {
            return $typenow;
        }
        
        if ($current_screen && isset($current_screen->post_type)) {
            return $current_screen->post_type;
        }
        
        // Определение типа записи из запроса (только чтение для контекста, не обработка формы)
        // phpcs:disable WordPress.Security.NonceVerification.Recommended
        // Используется только для определения контекста выполнения, не для обработки данных формы
        if (isset($_REQUEST['post_type'])) {
            return sanitize_text_field(wp_unslash($_REQUEST['post_type']));
        }
        
        // Определение типа записи по ID (только чтение для контекста, не обработка формы)
        if (isset($_REQUEST['post'])) {
            $post_id = absint(wp_unslash($_REQUEST['post']));
            if ($post_id) {
                return get_post_type($post_id);
            }
        }
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
        
        // Для REST API запросов определяем тип из URL
        if (defined('REST_REQUEST') && REST_REQUEST) {
            // Пробуем определить из REST route (несколько способов)
            $rest_route = '';
            
            // Способ 1: из глобальных переменных WordPress
            if (isset($GLOBALS['wp']->query_vars['rest_route'])) {
                $rest_route = $GLOBALS['wp']->query_vars['rest_route'];
            }
            
            // Способ 2: из REQUEST_URI (fallback)
            if (empty($rest_route) && isset($_SERVER['REQUEST_URI'])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Санитизация через sanitize_text_field и проверка паттерном
                $request_uri = sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI']));
                // Ищем REST API паттерн в URI
                if (preg_match('#/wp-json(/wp/v2/[a-z_-]+)#i', $request_uri, $uri_matches)) {
                    $rest_route = sanitize_text_field($uri_matches[1]);
                }
            }
            
            // Стандартные REST маршруты: /wp/v2/posts, /wp/v2/pages, /wp/v2/products
            if (preg_match('#/wp/v2/([a-z_-]+)#i', $rest_route, $matches)) {
                $endpoint = strtolower($matches[1]);
                // Преобразуем множественное число в тип записи
                $type_map = [
                    'posts' => 'post',
                    'pages' => 'page',
                    'products' => 'product',
                    'media' => 'attachment',
                ];
                if (isset($type_map[$endpoint])) {
                    return $type_map[$endpoint];
                }
                // Для кастомных типов записей (обычно без изменений)
                $singular = rtrim($endpoint, 's');
                if (post_type_exists($singular)) {
                    return $singular;
                }
                if (post_type_exists($endpoint)) {
                    return $endpoint;
                }
            }
        }
        
        return null;
    }
}

