<?php
/**
 * Класс настроек плагина
 * 
 * @package DevBrothers_Cyrillic_Slugs
 */

// Защита от прямого доступа
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Класс для управления настройками плагина
 */
class DBCS_Settings {
    
    /**
     * Имя опции в базе данных
     * @var string
     */
    private $option_name = 'dbcs_settings';
    
    /**
     * Конструктор
     */
    public function __construct() {
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_post_dbcs_save_settings', [$this, 'save_settings']);
    }
    
    /**
     * Регистрация настроек
     */
    public function register_settings() {
        register_setting('dbcs_settings_group', $this->option_name, [
            'sanitize_callback' => [$this, 'sanitize_settings'],
        ]);
    }
    
    /**
     * Санитизация настроек
     */
    public function sanitize_settings($input) {
        $sanitized = [];
        
        if (isset($input['post_types']) && is_array($input['post_types'])) {
            $sanitized['post_types'] = array_map('sanitize_text_field', $input['post_types']);
        }
        
        if (isset($input['taxonomies']) && is_array($input['taxonomies'])) {
            $sanitized['taxonomies'] = array_map('sanitize_text_field', $input['taxonomies']);
        }
        
        return $sanitized;
    }
    
    /**
     * Получение текущих настроек
     * 
     * @return array Массив настроек
     */
    public function get_settings() {
        $defaults = [
            'post_types' => ['post', 'page'],
            'taxonomies' => [],
        ];
        
        $settings = get_option($this->option_name, $defaults);
        return wp_parse_args($settings, $defaults);
    }
    
    /**
     * Получение всех доступных типов записей
     * 
     * @return array Массив типов записей
     */
    private function get_available_post_types() {
        $post_types = get_post_types([
            'public' => true,
        ], 'objects');
        
        // Добавляем attachment отдельно
        $attachment = get_post_type_object('attachment');
        if ($attachment) {
            $post_types['attachment'] = $attachment;
        }
        
        // Сортируем: сначала стандартные, потом кастомные
        $standard_types = ['post', 'page', 'attachment'];
        $sorted = [];
        
        foreach ($standard_types as $type) {
            if (isset($post_types[$type])) {
                $sorted[$type] = $post_types[$type];
            }
        }
        
        foreach ($post_types as $key => $post_type) {
            if (!in_array($key, $standard_types)) {
                $sorted[$key] = $post_type;
            }
        }
        
        return $sorted;
    }
    
    /**
     * Получение всех доступных таксономий
     * 
     * @return array Массив таксономий
     */
    private function get_available_taxonomies() {
        $taxonomies = get_taxonomies([
            'public' => true,
        ], 'objects');
        
        return $taxonomies;
    }
    
    /**
     * Проверка активности WooCommerce
     * 
     * @return bool true если WooCommerce активен
     */
    private function is_woocommerce_active() {
        return class_exists('WooCommerce');
    }
    
    /**
     * Отрисовка страницы настроек
     */
    public function render_settings_page() {
        // Обработка сохранения настроек ДО любого вывода
        $settings_saved = false;
        if (isset($_POST['dbcs_save_settings']) && check_admin_referer('dbcs_settings_nonce')) {
            $this->save_settings_internal();
            $settings_saved = true;
        }
        
        $settings = $this->get_settings();
        $post_types = $this->get_available_post_types();
        $taxonomies = $this->get_available_taxonomies();
        $is_woocommerce_active = $this->is_woocommerce_active();
        
        // Показываем сообщение об успехе если настройки сохранены
        if ($settings_saved) {
            echo '<div class="notice notice-success"><p>' . 
                 esc_html__('Настройки сохранены!', 'devbrothers-cyrillic-url') . 
                 '</p></div>';
        }
        
        ?>
        <form method="post" action="">
            <?php wp_nonce_field('dbcs_settings_nonce'); ?>
            
            <!-- Категория: Основные настройки -->
            <div id="general" class="devbrothers-settings-category">
                <h2>
                    <span class="dashicons dashicons-admin-generic"></span>
                    <?php esc_html_e('Основные настройки', 'devbrothers-cyrillic-url'); ?>
                </h2>
                
                <p class="description">
                    <?php esc_html_e('Выберите типы записей, для которых будет автоматически работать транслитерация URL.', 'devbrothers-cyrillic-url'); ?>
                </p>
                
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Типы записей', 'devbrothers-cyrillic-url'); ?>
                        </th>
                        <td>
                            <fieldset>
                                <legend class="screen-reader-text">
                                    <span><?php esc_html_e('Типы записей', 'devbrothers-cyrillic-url'); ?></span>
                                </legend>
                                
                                <?php foreach ($post_types as $post_type_name => $post_type_obj): ?>
                                    <?php
                                    $checked = in_array($post_type_name, $settings['post_types']);
                                    $label = $post_type_obj->label . ' (' . $post_type_name . ')';
                                    
                                    // Добавляем метку для WooCommerce типов
                                    if (in_array($post_type_name, ['product', 'shop_order', 'shop_coupon'])) {
                                        $label .= ' <span class="dashicons dashicons-cart" title="WooCommerce"></span>';
                                    }
                                    ?>
                                    <label>
                                        <input type="checkbox" 
                                               name="dbcs_post_types[]" 
                                               value="<?php echo esc_attr($post_type_name); ?>"
                                               <?php checked($checked); ?> />
                                        <?php echo esc_html($post_type_obj->label); ?>
                                        <span class="description">(<?php echo esc_html($post_type_name); ?>)</span>
                                        <?php if (in_array($post_type_name, ['product', 'shop_order', 'shop_coupon'])): ?>
                                            <span class="dashicons dashicons-cart" style="color: #96588a;" title="WooCommerce"></span>
                                        <?php endif; ?>
                                    </label>
                                    <br>
                                <?php endforeach; ?>
                            </fieldset>
                            
                            <?php if ($is_woocommerce_active): ?>
                                <p class="description">
                                    <span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
                                    <?php esc_html_e('WooCommerce обнаружен! Типы товаров добавлены в список.', 'devbrothers-cyrillic-url'); ?>
                                </p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Таксономии', 'devbrothers-cyrillic-url'); ?>
                        </th>
                        <td>
                            <fieldset>
                                <legend class="screen-reader-text">
                                    <span><?php esc_html_e('Таксономии', 'devbrothers-cyrillic-url'); ?></span>
                                </legend>
                                
                                <?php foreach ($taxonomies as $taxonomy_name => $taxonomy_obj): ?>
                                    <?php
                                    $checked = in_array($taxonomy_name, $settings['taxonomies']);
                                    ?>
                                    <label>
                                        <input type="checkbox" 
                                               name="dbcs_taxonomies[]" 
                                               value="<?php echo esc_attr($taxonomy_name); ?>"
                                               <?php checked($checked); ?> />
                                        <?php echo esc_html($taxonomy_obj->label); ?>
                                        <span class="description">(<?php echo esc_html($taxonomy_name); ?>)</span>
                                        <?php if (in_array($taxonomy_name, ['product_cat', 'product_tag'])): ?>
                                            <span class="dashicons dashicons-cart" style="color: #96588a;" title="WooCommerce"></span>
                                        <?php endif; ?>
                                    </label>
                                    <br>
                                <?php endforeach; ?>
                            </fieldset>
                            
                            <p class="description">
                                <?php esc_html_e('Таксономии: категории, теги и другие группировки контента.', 'devbrothers-cyrillic-url'); ?>
                            </p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- Категория: Конвертация URL -->
            <div id="conversion" class="devbrothers-settings-category">
                <h2>
                    <span class="dashicons dashicons-update"></span>
                    <?php esc_html_e('Конвертация существующих URL', 'devbrothers-cyrillic-url'); ?>
                </h2>
                
                <p class="description">
                    <?php esc_html_e('Конвертируйте все существующие записи и термины с кириллическими URL в латинские.', 'devbrothers-cyrillic-url'); ?>
                </p>
                
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Массовая конвертация', 'devbrothers-cyrillic-url'); ?>
                        </th>
                        <td>
                            <button type="button" 
                                    id="dbcs-convert-button" 
                                    class="button button-primary">
                                <span class="dashicons dashicons-update"></span>
                                <?php esc_html_e('Конвертировать все URL', 'devbrothers-cyrillic-url'); ?>
                            </button>
                            
                            <p class="description">
                                <?php esc_html_e('Это действие обновит slug всех записей и терминов с кириллицей. Процесс может занять некоторое время.', 'devbrothers-cyrillic-url'); ?>
                            </p>
                            
                            <div id="dbcs-conversion-progress" style="display: none; margin-top: 15px;">
                                <div style="background: #f0f0f1; border-radius: 4px; padding: 15px;">
                                    <p><strong><?php esc_html_e('Идет конвертация...', 'devbrothers-cyrillic-url'); ?></strong></p>
                                    <div style="background: #fff; border: 1px solid #c3c4c7; border-radius: 3px; height: 24px; overflow: hidden;">
                                        <div id="dbcs-progress-bar" style="background: #2271b1; height: 100%; width: 0%; transition: width 0.3s;"></div>
                                    </div>
                                    <p id="dbcs-progress-text" style="margin: 10px 0 0;">0%</p>
                                </div>
                            </div>
                            
                            <div id="dbcs-conversion-result" style="display: none; margin-top: 15px;"></div>
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- Категория: О плагине -->
            <div id="about" class="devbrothers-settings-category">
                <h2>
                    <span class="dashicons dashicons-info"></span>
                    <?php esc_html_e('О плагине', 'devbrothers-cyrillic-url'); ?>
                </h2>
                
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Версия', 'devbrothers-cyrillic-url'); ?>
                        </th>
                        <td>
                            <p><?php echo esc_html(DBCS_VERSION); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Стандарт транслитерации', 'devbrothers-cyrillic-url'); ?>
                        </th>
                        <td>
                            <p><strong>ISO 9</strong> - <?php esc_html_e('международный стандарт транслитерации кириллицы', 'devbrothers-cyrillic-url'); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Информация', 'devbrothers-cyrillic-url'); ?>
                        </th>
                        <td>
                            <p><?php esc_html_e('Плагин автоматически конвертирует кириллические символы в URL в латинские по стандарту ISO 9.', 'devbrothers-cyrillic-url'); ?></p>
                            <p><?php esc_html_e('Это улучшает читаемость URL и совместимость с различными системами.', 'devbrothers-cyrillic-url'); ?></p>
                            
                            <p>
                                <strong><?php esc_html_e('Пример:', 'devbrothers-cyrillic-url'); ?></strong><br>
                                <code>новая-запись</code> → <code>novaya-zapis</code>
                            </p>
                            
                            <?php if ($is_woocommerce_active): ?>
                                <p style="padding: 10px; background: #f0f6fc; border-left: 4px solid #2271b1;">
                                    <span class="dashicons dashicons-cart"></span>
                                    <?php esc_html_e('WooCommerce поддерживается! Плагин работает с товарами и категориями.', 'devbrothers-cyrillic-url'); ?>
                                </p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('WordPress версия', 'devbrothers-cyrillic-url'); ?>
                        </th>
                        <td>
                            <p><?php echo esc_html(get_bloginfo('version')); ?></p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- Кнопка сохранения -->
            <p class="submit">
                <button type="submit" name="dbcs_save_settings" class="button button-primary">
                    <span class="dashicons dashicons-yes"></span>
                    <?php esc_html_e('Сохранить настройки', 'devbrothers-cyrillic-url'); ?>
                </button>
            </p>
        </form>
        <?php
    }
    
    /**
     * Внутренний метод сохранения настроек (без редиректа)
     */
    private function save_settings_internal() {
        // Получаем и очищаем данные
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce проверяется в render_settings_page
        $post_types = isset($_POST['dbcs_post_types']) ? array_map('sanitize_text_field', wp_unslash($_POST['dbcs_post_types'])) : [];
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce проверяется в render_settings_page
        $taxonomies = isset($_POST['dbcs_taxonomies']) ? array_map('sanitize_text_field', wp_unslash($_POST['dbcs_taxonomies'])) : [];
        
        // Сохраняем настройки
        $settings = [
            'post_types' => $post_types,
            'taxonomies' => $taxonomies,
        ];
        
        update_option($this->option_name, $settings);
    }
    
    /**
     * Сохранение настроек (публичный метод для admin_post хука)
     */
    public function save_settings() {
        // Проверка nonce
        // wp_verify_nonce требует санитизации, так как функция pluggable
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'dbcs_settings_nonce')) {
            wp_die(esc_html__('Ошибка безопасности', 'devbrothers-cyrillic-url'));
        }
        
        // Проверка прав
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Недостаточно прав', 'devbrothers-cyrillic-url'));
        }
        
        // Сохраняем настройки
        $this->save_settings_internal();
        
        // Если это AJAX запрос, не перенаправляем
        if (defined('DOING_AJAX') && DOING_AJAX) {
            return;
        }
        
        // Перенаправление обратно с сообщением
        wp_safe_redirect(add_query_arg('settings-updated', 'true', wp_get_referer()));
        exit;
    }
}

