<?php
/**
 * Класс конвертера существующих URL
 * 
 * @package DevBrothers_Cyrillic_Slugs
 */

// Защита от прямого доступа
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Класс для массовой конвертации существующих URL
 */
class DBCS_Converter {
    
    /**
     * Экземпляр транслитератора
     * @var DBCS_Transliterator
     */
    private $transliterator;
    
    /**
     * Лимит обработки за один запрос
     * @var int
     */
    private $batch_limit = 100;
    
    /**
     * Конструктор
     * 
     * @param DBCS_Transliterator $transliterator Экземпляр транслитератора
     */
    public function __construct($transliterator) {
        $this->transliterator = $transliterator;
        $this->init_hooks();
    }
    
    /**
     * Инициализация хуков
     */
    private function init_hooks() {
        // AJAX хук для конвертации
        add_action('wp_ajax_dbcs_convert_urls', [$this, 'ajax_convert_urls']);
    }
    
    /**
     * AJAX обработчик конвертации
     */
    public function ajax_convert_urls() {
        // Проверка nonce
        check_ajax_referer('dbcs_convert_nonce', 'nonce');
        
        // Проверка прав
        if (!current_user_can('manage_options')) {
            wp_send_json_error([
                'message' => __('Недостаточно прав для выполнения этой операции', 'devbrothers-cyrillic-url')
            ]);
        }
        
        // Получаем offset из запроса
        $offset = isset($_POST['offset']) ? absint($_POST['offset']) : 0;
        
        // Выполняем конвертацию
        $result = $this->convert_batch($offset);
        
        // Отправляем результат
        wp_send_json_success($result);
    }
    
    /**
     * Конвертация одной партии записей
     * 
     * @param int $offset Смещение для выборки
     * @return array Результат конвертации
     */
    public function convert_batch($offset = 0) {
        global $wpdb;
        
        $settings = get_option('dbcs_settings', []);
        $post_types = !empty($settings['post_types']) ? $settings['post_types'] : ['post', 'page'];
        $taxonomies = !empty($settings['taxonomies']) ? $settings['taxonomies'] : [];
        
        $converted_posts = 0;
        $converted_terms = 0;
        $errors = [];
        
        // Конвертация записей (posts)
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce проверяется в ajax_convert_urls
        if ($offset === 0 || isset($_POST['type']) && $_POST['type'] === 'posts') {
            $posts_result = $this->convert_posts($post_types, $offset);
            $converted_posts = $posts_result['converted'];
            $errors = array_merge($errors, $posts_result['errors']);
        }
        
        // Конвертация терминов (categories, tags, etc)
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce проверяется в ajax_convert_urls
        if (($offset === 0 && $converted_posts < $this->batch_limit) || (isset($_POST['type']) && $_POST['type'] === 'terms')) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce проверяется в ajax_convert_urls
            $terms_offset = isset($_POST['terms_offset']) ? absint($_POST['terms_offset']) : 0;
            $terms_result = $this->convert_terms($taxonomies, $terms_offset);
            $converted_terms = $terms_result['converted'];
            $errors = array_merge($errors, $terms_result['errors']);
        }
        
        // Получаем общее количество для конвертации
        $total = $this->get_total_count($post_types, $taxonomies);
        $processed = $offset + $converted_posts;
        
        return [
            'converted_posts' => $converted_posts,
            'converted_terms' => $converted_terms,
            'total' => $total,
            'processed' => $processed,
            'has_more' => $processed < $total,
            'errors' => $errors,
        ];
    }
    
    /**
     * Конвертация записей
     * 
     * @param array $post_types Типы записей для конвертации
     * @param int $offset Смещение
     * @return array Результат
     */
    private function convert_posts($post_types, $offset = 0) {
        global $wpdb;
        
        $offset = absint($offset);
        $limit  = absint($this->batch_limit);
        $converted = 0;
        $errors = [];
        
        if (empty($post_types)) {
            return ['converted' => 0, 'errors' => []];
        }

        // Санитизируем типы записей
        $sanitized_post_types = array_map('sanitize_text_field', (array) $post_types);
        
        // Подготовка SQL запроса с использованием $wpdb->prepare() и динамического списка типов
        $placeholders = implode(', ', array_fill(0, count($sanitized_post_types), '%s'));
        
        $sql = "
            SELECT ID, post_name, post_type, post_status
            FROM {$wpdb->posts}
            WHERE post_type IN ($placeholders)
              AND post_status NOT IN ('trash', 'auto-draft')
            LIMIT %d OFFSET %d
        ";
        
        // Собираем аргументы для prepare: все типы + лимит + offset
        $args = array_merge($sanitized_post_types, [$limit, $offset]);
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Динамическое количество типов записей через плейсхолдеры, все данные санитизированы
        $query = call_user_func_array([$wpdb, 'prepare'], array_merge([$sql], $args));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $posts = $wpdb->get_results($query);
        
        // Фильтруем записи с кириллицей в PHP (надежнее чем REGEXP в MySQL)
        // ВАЖНО: декодируем URL перед проверкой на кириллицу!
        $posts = array_filter($posts, function($post) {
            $decoded_slug = urldecode($post->post_name);
            return preg_match('/[а-яёА-ЯЁ]/u', $decoded_slug);
        });
        
        foreach ($posts as $post) {
            try {
                // Декодируем slug (если он в URL-encoded формате)
                $decoded_slug = urldecode($post->post_name);
                
                // Транслитерируем декодированный slug
                $new_slug = $this->transliterator->transliterate($decoded_slug);
                
                // Проверяем уникальность
                $new_slug = wp_unique_post_slug(
                    $new_slug,
                    $post->ID,
                    get_post_status($post->ID),
                    $post->post_type,
                    0
                );
                
                // Обновляем запись
                $updated = wp_update_post([
                    'ID' => $post->ID,
                    'post_name' => $new_slug,
                ], true);
                
                if (is_wp_error($updated)) {
                    $errors[] = sprintf(
                        /* translators: 1: Post ID, 2: Error message */
                        __('Ошибка при обновлении записи #%1$d: %2$s', 'devbrothers-cyrillic-url'),
                        $post->ID,
                        $updated->get_error_message()
                    );
                } else {
                    $converted++;
                }
                
            } catch (Exception $e) {
                $errors[] = sprintf(
                    /* translators: 1: Post ID, 2: Exception message */
                    __('Исключение при обновлении записи #%1$d: %2$s', 'devbrothers-cyrillic-url'),
                    $post->ID,
                    $e->getMessage()
                );
            }
        }
        
        return [
            'converted' => $converted,
            'errors' => $errors,
        ];
    }
    
    /**
     * Конвертация терминов
     * 
     * @param array $taxonomies Таксономии для конвертации
     * @param int $offset Смещение
     * @return array Результат
     */
    private function convert_terms($taxonomies, $offset = 0) {
        global $wpdb;
        
        $offset = absint($offset);
        $limit  = absint($this->batch_limit);
        $converted = 0;
        $errors = [];
        
        // Если таксономии не указаны, конвертируем все публичные
        if (empty($taxonomies)) {
            $taxonomies = get_taxonomies(['public' => true]);
        }
        
        if (empty($taxonomies)) {
            return ['converted' => 0, 'errors' => []];
        }

        // Санитизируем таксономии
        $sanitized_taxonomies = array_map('sanitize_text_field', (array) $taxonomies);
        
        // Подготовка SQL запроса с использованием $wpdb->prepare() и динамического списка таксономий
        $placeholders = implode(', ', array_fill(0, count($sanitized_taxonomies), '%s'));
        
        $sql = "
            SELECT t.term_id, t.slug, tt.taxonomy
            FROM {$wpdb->terms} AS t
            INNER JOIN {$wpdb->term_taxonomy} AS tt ON t.term_id = tt.term_id
            WHERE tt.taxonomy IN ($placeholders)
            LIMIT %d OFFSET %d
        ";
        
        $args = array_merge($sanitized_taxonomies, [$limit, $offset]);
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Динамическое количество таксономий через плейсхолдеры, все данные санитизированы
        $query = call_user_func_array([$wpdb, 'prepare'], array_merge([$sql], $args));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $terms = $wpdb->get_results($query);
        
        // Фильтруем термины с кириллицей в PHP (надежнее чем REGEXP в MySQL)
        // ВАЖНО: декодируем URL перед проверкой!
        $terms = array_filter($terms, function($term) {
            $decoded_slug = urldecode($term->slug);
            return preg_match('/[а-яёА-ЯЁ]/u', $decoded_slug);
        });
        
        foreach ($terms as $term) {
            try {
                // Декодируем slug (если он в URL-encoded формате)
                $decoded_slug = urldecode($term->slug);
                
                // Транслитерируем декодированный slug
                $new_slug = $this->transliterator->transliterate($decoded_slug);
                
                // Обновляем термин
                $updated = wp_update_term($term->term_id, $term->taxonomy, [
                    'slug' => $new_slug,
                ]);
                
                if (is_wp_error($updated)) {
                    $errors[] = sprintf(
                        /* translators: 1: Term ID, 2: Error message */
                        __('Ошибка при обновлении термина #%1$d: %2$s', 'devbrothers-cyrillic-url'),
                        $term->term_id,
                        $updated->get_error_message()
                    );
                } else {
                    $converted++;
                }
                
            } catch (Exception $e) {
                $errors[] = sprintf(
                    /* translators: 1: Term ID, 2: Exception message */
                    __('Исключение при обновлении термина #%1$d: %2$s', 'devbrothers-cyrillic-url'),
                    $term->term_id,
                    $e->getMessage()
                );
            }
        }
        
        return [
            'converted' => $converted,
            'errors' => $errors,
        ];
    }
    
    /**
     * Получение общего количества элементов для конвертации
     * 
     * @param array $post_types Типы записей
     * @param array $taxonomies Таксономии
     * @return int Общее количество
     */
    private function get_total_count($post_types, $taxonomies) {
        global $wpdb;
        
        $total = 0;
        
        // Подсчет записей с кириллицей
        if (!empty($post_types)) {
            $sanitized_post_types = array_map('sanitize_text_field', (array) $post_types);
            // Создаем плейсхолдеры для динамического списка типов записей
            $placeholders = implode(', ', array_fill(0, count($sanitized_post_types), '%s'));
            $sql = "
                SELECT post_name
                FROM {$wpdb->posts}
                WHERE post_type IN ($placeholders)
                  AND post_status NOT IN ('trash', 'auto-draft')
            ";
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Динамическое количество типов записей, все данные санитизированы
            $query = call_user_func_array([$wpdb, 'prepare'], array_merge([$sql], $sanitized_post_types));
            
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $posts = $wpdb->get_col($query);
            foreach ($posts as $post_name) {
                $decoded = urldecode($post_name);
                if (preg_match('/[а-яёА-ЯЁ]/u', $decoded)) {
                    $total++;
                }
            }
        }
        
        // Подсчет терминов с кириллицей
        if (empty($taxonomies)) {
            $taxonomies = get_taxonomies(['public' => true]);
        }
        
        if (!empty($taxonomies)) {
            $sanitized_taxonomies = array_map('sanitize_text_field', (array) $taxonomies);
            // Создаем плейсхолдеры для динамического списка таксономий
            $placeholders = implode(', ', array_fill(0, count($sanitized_taxonomies), '%s'));
            $sql = "
                SELECT t.slug
                FROM {$wpdb->terms} AS t
                INNER JOIN {$wpdb->term_taxonomy} AS tt ON t.term_id = tt.term_id
                WHERE tt.taxonomy IN ($placeholders)
            ";
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Динамическое количество таксономий, все данные санитизированы
            $query = call_user_func_array([$wpdb, 'prepare'], array_merge([$sql], $sanitized_taxonomies));
            
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $slugs = $wpdb->get_col($query);
            foreach ($slugs as $slug) {
                $decoded = urldecode($slug);
                if (preg_match('/[а-яёА-ЯЁ]/u', $decoded)) {
                    $total++;
                }
            }
        }
        
        return $total;
    }
    
    /**
     * Конвертация всех URL (синхронная версия для CLI или крона)
     * 
     * @return array Результат конвертации
     */
    public function convert_all() {
        $settings = get_option('dbcs_settings', []);
        $post_types = !empty($settings['post_types']) ? $settings['post_types'] : ['post', 'page'];
        $taxonomies = !empty($settings['taxonomies']) ? $settings['taxonomies'] : [];
        
        $total_converted = 0;
        $all_errors = [];
        
        // Конвертация записей
        $offset = 0;
        do {
            $result = $this->convert_posts($post_types, $offset);
            $total_converted += $result['converted'];
            $all_errors = array_merge($all_errors, $result['errors']);
            $offset += $this->batch_limit;
            
            // Защита от бесконечного цикла
            if ($result['converted'] === 0) {
                break;
            }
        } while ($result['converted'] > 0);
        
        // Конвертация терминов
        $offset = 0;
        do {
            $result = $this->convert_terms($taxonomies, $offset);
            $total_converted += $result['converted'];
            $all_errors = array_merge($all_errors, $result['errors']);
            $offset += $this->batch_limit;
            
            // Защита от бесконечного цикла
            if ($result['converted'] === 0) {
                break;
            }
        } while ($result['converted'] > 0);
        
        return [
            'total_converted' => $total_converted,
            'errors' => $all_errors,
        ];
    }
}

