<?php
/**
 * Plugin Name: DevBrothers Cyrillic URL
 * Plugin URI: https://devbrothers.ru/cyrillic-slugs
 * Description: Automatic transliteration of Cyrillic URLs to Latin according to ISO 9 standard. Support for all post types including WooCommerce.
 * Version: 1.0.0
 * Author: DevBrothers
 * Author URI: https://devbrothers.ru
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: devbrothers-cyrillic-url
 * Requires at least: 5.8
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * Requires Plugins: devbrothers-admin-panel
 */

// Защита от прямого доступа
if (!defined('ABSPATH')) {
    exit;
}

// Константы плагина
define('DBCS_VERSION', '1.0.0');
define('DBCS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('DBCS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('DBCS_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('DBCS_PREFIX', 'dbcs');

/**
 * Главный класс плагина
 */
class DevBrothers_Cyrillic_Slugs {
    
    /**
     * Единственный экземпляр класса
     * @var DevBrothers_Cyrillic_Slugs
     */
    private static $instance = null;
    
    /**
     * Экземпляр транслитератора
     * @var DBCS_Transliterator
     */
    public $transliterator;
    
    /**
     * Экземпляр настроек
     * @var DBCS_Settings
     */
    public $settings;
    
    /**
     * Экземпляр конвертера
     * @var DBCS_Converter
     */
    public $converter;
    
    /**
     * Получить экземпляр класса (Singleton)
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Конструктор
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
        $this->init_components();
    }
    
    /**
     * Загрузка зависимостей
     */
    private function load_dependencies() {
        require_once DBCS_PLUGIN_DIR . 'includes/class-transliterator.php';
        require_once DBCS_PLUGIN_DIR . 'includes/class-settings.php';
        require_once DBCS_PLUGIN_DIR . 'includes/class-converter.php';
    }
    
    /**
     * Инициализация хуков WordPress
     */
    private function init_hooks() {
        // Загрузка переводов
        add_action('plugins_loaded', [$this, 'load_textdomain']);
        
        // Регистрация в DevBrothers
        add_action('devbrothers_ready', [$this, 'register_in_devbrothers']);
        
        // Подключение стилей и скриптов
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }
    
    /**
     * Инициализация компонентов
     */
    private function init_components() {
        $this->transliterator = new DBCS_Transliterator();
        $this->settings = new DBCS_Settings();
        $this->converter = new DBCS_Converter($this->transliterator);
    }
    
    /**
     * Загрузка переводов
     * С WordPress 4.6+ переводы загружаются автоматически
     */
    public function load_textdomain() {
        // load_plugin_textdomain() больше не требуется
        // WordPress автоматически загружает переводы из wordpress.org
    }
    
    /**
     * Проверка наличия базового плагина DevBrothers
     */
    public function check_devbrothers_dependency() {
        if (!function_exists('devbrothers_register_plugin')) {
            add_action('admin_notices', function() {
                ?>
                <div class="notice notice-error">
                    <p>
                        <strong><?php esc_html_e('DevBrothers Cyrillic URL', 'devbrothers-cyrillic-url'); ?></strong>
                        <?php esc_html_e(' требует установки ', 'devbrothers-cyrillic-url'); ?>
                        <a href="<?php echo esc_url(admin_url('plugin-install.php?s=devbrothers+admin+panel&tab=search')); ?>">
                            DevBrothers Admin Panel
                        </a>
                    </p>
                </div>
                <?php
            });
            return false;
        }
        return true;
    }
    
    /**
     * Регистрация плагина в системе DevBrothers
     */
    public function register_in_devbrothers() {
        if (!$this->check_devbrothers_dependency()) {
            return;
        }
        
        devbrothers_register_plugin([
            'id' => 'cyrillic-slugs',
            'name' => __('Cyrillic URL', 'devbrothers-cyrillic-url'),
            'name_ru' => __('Кириллические URL', 'devbrothers-cyrillic-url'),
            'description' => __('Автоматическая транслитерация кириллических URL в латиницу', 'devbrothers-cyrillic-url'),
            'version' => DBCS_VERSION,
            'icon' => 'dashicons-translation',
            'settings_callback' => [$this->settings, 'render_settings_page'],
            'categories' => [
                [
                    'id' => 'general',
                    'name' => __('Основные настройки', 'devbrothers-cyrillic-url'),
                    'icon' => 'dashicons-admin-generic'
                ],
                [
                    'id' => 'conversion',
                    'name' => __('Конвертация URL', 'devbrothers-cyrillic-url'),
                    'icon' => 'dashicons-update'
                ],
                [
                    'id' => 'about',
                    'name' => __('О плагине', 'devbrothers-cyrillic-url'),
                    'icon' => 'dashicons-info'
                ]
            ]
        ]);
    }
    
    /**
     * Подключение стилей и скриптов
     */
    public function enqueue_admin_assets($hook) {
        // Загружаем только на страницах DevBrothers
        if (strpos($hook, 'devbrothers') === false) {
            return;
        }
        
        // CSS - зависимость от базового плагина для единых стилей
        wp_enqueue_style(
            'dbcs-admin',
            DBCS_PLUGIN_URL . 'assets/css/admin.css',
            ['devbrothers-admin'], // Зависимость от стилей базового плагина
            DBCS_VERSION
        );
        
        // JS
        wp_enqueue_script(
            'dbcs-admin',
            DBCS_PLUGIN_URL . 'assets/js/admin.js',
            ['jquery'],
            DBCS_VERSION,
            true
        );
        
        // Локализация для AJAX
        wp_localize_script('dbcs-admin', 'dbcsData', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('dbcs_convert_nonce'),
            'converting_text' => __('Конвертация...', 'devbrothers-cyrillic-url'),
            'completed_text' => __('Завершено!', 'devbrothers-cyrillic-url'),
            'error_text' => __('Ошибка!', 'devbrothers-cyrillic-url')
        ]);
    }
}

/**
 * Функция-хелпер для получения экземпляра плагина
 */
function dbcs_plugin() {
    return DevBrothers_Cyrillic_Slugs::get_instance();
}

// Инициализация плагина
add_action('plugins_loaded', 'dbcs_plugin', 10);

