<?php

/**
 * Plugin Name: Delete Own Admin Account
 * Description: Adds a secure “Delete Own Account” option for admins in the WordPress Users list, allowing self-removal while safely reassigning content.
 * Version: 1.0.2
 * Author:            SWIT & FreelancersTools (Ivica Delić)
 * Author URI:        https://github.com/sandiwinter/
 * Text Domain: delete-own-admin-account
 * Domain Path: /languages
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main plugin class
 */
class DeleteOwnAdministratorAccount
{

    public function __construct()
    {
        add_action('init', array($this, 'init'));
    }

    public function init()
    {
        // Add hooks only in admin
        if (is_admin()) {
            // Add delete link to user row actions
            add_filter('user_row_actions', array($this, 'add_delete_own_account_link'), 10, 2);

            // Handle delete action
            add_action('admin_init', array($this, 'handle_delete_own_account'));

            // Handle delete confirmation page
            add_action('admin_menu', array($this, 'add_delete_confirmation_page'));

            // Add admin notice after deletion redirect
            add_action('admin_notices', array($this, 'show_deletion_notice'));
        }
    }

    /**
     * Add delete link for current user's own account
     */
    public function add_delete_own_account_link($actions, $user_object)
    {
        // Only show for current user who is an administrator
        $current_user = wp_get_current_user();

        // Check if this is the current user and they are an administrator
        if ($current_user->ID == $user_object->ID && current_user_can('administrator')) {

            // Create nonce for security
            $delete_nonce = wp_create_nonce('delete_own_admin_account_' . $user_object->ID);

            // Create delete URL (redirect to confirmation page)
            $delete_url = add_query_arg(array(
                'page' => 'delete-own-admin-confirm',
                'user_id' => $user_object->ID,
                '_wpnonce' => $delete_nonce
            ), admin_url('admin.php'));

            // Add delete link with red styling similar to WordPress default
            $actions['delete_own'] = sprintf(
                '<a href="%s" style="color: #d63638;">%s</a>',
                esc_url($delete_url),
                esc_html__('Delete Own Account', 'delete-own-admin-account')
            );
        }

        return $actions;
    }

    /**
     * Handle the actual delete own account action (from confirmation form)
     */
    public function handle_delete_own_account()
    {
        // Check if this is our delete confirmation action
        $action = isset($_POST['action']) ? sanitize_text_field(wp_unslash($_POST['action'])) : '';
        if ($action !== 'confirm_delete_own_admin_account') {
            return;
        }

        // Verify nonce
        $nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';
        $posted_user_id = isset($_POST['user_id']) ? sanitize_text_field(wp_unslash($_POST['user_id'])) : '';

        if (empty($nonce) || !wp_verify_nonce($nonce, 'delete_own_admin_account_' . $posted_user_id)) {
            wp_die(esc_html__('Security check failed. Please try again.', 'delete-own-admin-account'));
        }

        $user_id = intval($posted_user_id);
        $reassign_user_id = isset($_POST['reassign_user']) ? intval(sanitize_text_field(wp_unslash($_POST['reassign_user']))) : 0;
        $current_user = wp_get_current_user();

        // Security checks
        if ($user_id !== $current_user->ID) {
            wp_die(esc_html__('You can only delete your own account.', 'delete-own-admin-account'));
        }

        if (!current_user_can('administrator')) {
            wp_die(esc_html__('Only administrators can use this feature.', 'delete-own-admin-account'));
        }

        // Validate reassign user
        if (empty($reassign_user_id) || $reassign_user_id == $user_id) {
            wp_die(esc_html__('Invalid reassignment user selected.', 'delete-own-admin-account'));
        }

        $reassign_user = get_user_by('ID', $reassign_user_id);
        if (!$reassign_user || !user_can($reassign_user_id, 'administrator')) {
            wp_die(esc_html__('Reassignment user must be an administrator.', 'delete-own-admin-account'));
        }

        // Check if this is the only administrator
        $admin_users = get_users(array(
            'role' => 'administrator',
            'number' => 2 // We only need to know if there are 2 or more
        ));

        if (count($admin_users) <= 1) {
            wp_die(esc_html__('Cannot delete your account as you are the only administrator. Please create another administrator account first.', 'delete-own-admin-account'));
        }

        // Perform the deletion
        require_once(ABSPATH . 'wp-admin/includes/user.php');

        $deleted = wp_delete_user($user_id, $reassign_user_id);

        if ($deleted) {
            // Log out the user immediately
            wp_clear_auth_cookie();
            wp_set_current_user(0);

            // Redirect to login page with success message
            $redirect_url = add_query_arg('deleted', '1', wp_login_url());
            wp_redirect($redirect_url);
            exit;
        } else {
            wp_die(esc_html__('Failed to delete account. Please try again.', 'delete-own-admin-account'));
        }
    }

    /**
     * Add delete confirmation page to admin menu (hidden)
     */
    public function add_delete_confirmation_page()
    {
        add_submenu_page(
            null, // No parent menu (hidden page)
            __('Delete Administrator Account', 'delete-own-admin-account'),
            __('Delete Administrator Account', 'delete-own-admin-account'),
            'administrator',
            'delete-own-admin-confirm',
            array($this, 'render_delete_confirmation_page')
        );
    }

    /**
     * Render the delete confirmation page
     */
    public function render_delete_confirmation_page()
    {
        // Verify access and nonce
        if (!current_user_can('administrator')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'delete-own-admin-account'));
        }

        if (!isset($_GET['user_id']) || !isset($_GET['_wpnonce'])) {
            wp_die(esc_html__('Invalid request.', 'delete-own-admin-account'));
        }

        $user_id = intval($_GET['user_id']);
        $current_user = wp_get_current_user();

        // Verify nonce
        $get_nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';
        if (empty($get_nonce) || !wp_verify_nonce($get_nonce, 'delete_own_admin_account_' . $user_id)) {
            wp_die(esc_html__('Security check failed. Please try again.', 'delete-own-admin-account'));
        }

        // Security checks
        if ($user_id !== $current_user->ID) {
            wp_die(esc_html__('You can only delete your own account.', 'delete-own-admin-account'));
        }

        // Get other administrators
        $admin_users = get_users(array(
            'role' => 'administrator',
            'exclude' => array($user_id) // Exclude current user
        ));

        if (empty($admin_users)) {
            wp_die(esc_html__('Cannot delete your account as you are the only administrator. Please create another administrator account first.', 'delete-own-admin-account'));
        }

        // Check if user has posts
        $post_count = count_user_posts($user_id);

?>
        <div class="wrap">
            <h1><?php echo esc_html__('Delete Administrator Account', 'delete-own-admin-account'); ?></h1>

            <div class="notice notice-warning">
                <p><strong><?php echo esc_html__('Warning:', 'delete-own-admin-account'); ?></strong>
                    <?php echo esc_html__('You are about to delete your administrator account. This action cannot be undone.', 'delete-own-admin-account'); ?></p>
            </div>

            <?php if ($post_count > 0): ?>
                <p><?php
                    /* translators: %d: number of posts */
                    printf(esc_html__('You have %d posts. What should be done with your posts?', 'delete-own-admin-account'), (int) $post_count); ?></p>
            <?php endif; ?>

            <form method="post" action="">
                <?php wp_nonce_field('delete_own_admin_account_' . $user_id); ?>
                <input type="hidden" name="action" value="confirm_delete_own_admin_account">
                <input type="hidden" name="user_id" value="<?php echo esc_attr($user_id); ?>">

                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="reassign_user"><?php echo esc_html__('Reassign posts to:', 'delete-own-admin-account'); ?></label>
                        </th>
                        <td>
                            <select name="reassign_user" id="reassign_user" required>
                                <option value=""><?php echo esc_html__('Select an administrator...', 'delete-own-admin-account'); ?></option>
                                <?php foreach ($admin_users as $admin): ?>
                                    <option value="<?php echo esc_attr($admin->ID); ?>">
                                        <?php echo esc_html($admin->display_name . ' (' . $admin->user_login . ')'); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description">
                                <?php echo esc_html__('All your posts and content will be transferred to the selected administrator.', 'delete-own-admin-account'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <input type="submit" name="submit" id="submit" class="button button-primary"
                        value="<?php esc_attr_e('Confirm Deletion', 'delete-own-admin-account'); ?>"
                        onclick="return confirm('<?php echo esc_js(__('Are you absolutely sure you want to delete your account? This cannot be undone!', 'delete-own-admin-account')); ?>');">
                    <a href="<?php echo esc_url(admin_url('users.php')); ?>" class="button">
                        <?php echo esc_html__('Cancel', 'delete-own-admin-account'); ?>
                    </a>
                </p>
            </form>
        </div>
<?php
    }

    /**
     * Show deletion notice on login page
     */
    public function show_deletion_notice()
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is just displaying a notice, not processing sensitive data
        $deleted = isset($_GET['deleted']) ? sanitize_text_field(wp_unslash($_GET['deleted'])) : '';
        if ($deleted === '1') {
            echo '<div class="notice notice-success"><p>' .
                esc_html__('Your administrator account has been successfully deleted.', 'delete-own-admin-account') .
                '</p></div>';
        }
    }
}

// Initialize the plugin
new DeleteOwnAdministratorAccount();
