<?php
/**
 * DEED Pay Payment Gateway
 *
 * Main gateway class extending WooCommerce payment gateway.
 *
 * @package DEEDPay
 * @since 0.1.1
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * WC_Gateway_DeedPay class
 */
// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedClassFound -- WooCommerce gateway class naming convention.
class WC_Gateway_DeedPay extends WC_Payment_Gateway {
    /**
     * Logger instance
     *
     * @var DeedPay_Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct() {
        $this->id = 'deedpay';
        
        // Initialize logger early (needed for icon logging)
        $this->logger = DeedPay_Logger::get_instance();
        
        // Set gateway icon (PNG preferred, fallback to SVG)
        $icon_png = DEEDPAY_PLUGIN_DIR . 'assets/images/deedpay-icon.png';
        $icon_svg = DEEDPAY_PLUGIN_DIR . 'assets/images/icon.svg';
        $icon_url = '';
        if (file_exists($icon_png)) {
            $icon_url = plugins_url('assets/images/deedpay-icon.png', DEEDPAY_PLUGIN_FILE);
        } elseif (file_exists($icon_svg)) {
            $icon_url = plugins_url('assets/images/icon.svg', DEEDPAY_PLUGIN_FILE);
        } else {
            $this->logger->warning('Gateway icon file not found', array('expected_path' => $icon_png));
        }
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- WooCommerce gateway icon filter naming (woocommerce_ + gateway id).
        $this->icon = apply_filters('woocommerce_deedpay_icon', $icon_url);
        
        $this->method_title = __('DEED Pay', 'deed-pay-woo');
        $this->method_description = __(
            'Accept payments with escrow protection. Funds are held securely until delivery confirmation.',
            'deed-pay-woo'
        );

        // Supported features
        $this->supports = array(
            'products',
            'refunds',
        );

        // Load settings
        $this->init_form_fields();
        $this->init_settings();

        // Get settings
        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->enabled = $this->get_option('enabled');
        $this->test_mode = $this->get_option('test_mode') === 'yes';
        $this->api_public_key = $this->get_option('api_public_key');
        $this->api_secret_key = $this->get_option('api_secret_key');
        $this->webhook_secret = $this->get_option('webhook_secret');
        $this->auto_release_days = $this->get_option('auto_release_days', 7);
        $this->product_type = $this->get_option('product_type', 'PHYSICAL');
        $this->debug_mode = $this->get_option('debug_mode') === 'yes';

        // Save settings hook
        add_action(
            'woocommerce_update_options_payment_gateways_' . $this->id,
            array($this, 'process_admin_options')
        );

        // Validate API key field
        add_filter('woocommerce_settings_api_sanitized_fields_' . $this->id, array($this, 'validate_api_secret_key'));

        // Order received page redirect handling
        add_action('woocommerce_thankyou_deedpay', array($this, 'check_payment_status'), 10, 1);
    }

    /**
     * Return the option key used by WooCommerce to store gateway settings.
     * Uses plugin-specific prefix per WordPress.org requirements.
     *
     * @return string
     */
    public function get_option_key() {
        return 'deedpay_settings';
    }

    /**
     * Initialize form fields for admin settings
     */
    public function init_form_fields() {
        $webhook_url = home_url('/?wc-api=wc_gateway_deedpay');

        $this->form_fields = array(
            'enabled' => array(
                'title'       => __('Enable/Disable', 'deed-pay-woo'),
                'label'       => __('Enable DEED Pay', 'deed-pay-woo'),
                'type'        => 'checkbox',
                'description' => __('Enable DEED Pay as a payment method', 'deed-pay-woo'),
                'default'     => 'no',
            ),
            'title' => array(
                'title'       => __('Title', 'deed-pay-woo'),
                'type'        => 'text',
                'description' => __('Payment method title shown to customers during checkout', 'deed-pay-woo'),
                'default'     => __('DEED Pay', 'deed-pay-woo'),
                'desc_tip'    => true,
            ),
            'description' => array(
                'title'       => __('Description', 'deed-pay-woo'),
                'type'        => 'textarea',
                'description' => __('Payment method description shown to customers during checkout', 'deed-pay-woo'),
                'default'     => __('Pay securely with DEED Pay. Your funds are protected until delivery confirmation.', 'deed-pay-woo'),
                'desc_tip'    => true,
            ),
            'test_mode' => array(
                'title'       => __('Test Mode', 'deed-pay-woo'),
                'label'       => __('Enable Test Mode', 'deed-pay-woo'),
                'type'        => 'checkbox',
                'description' => __('Use test API keys for sandbox testing. Disable for live transactions.', 'deed-pay-woo'),
                'default'     => 'yes',
            ),
            'api_base_url' => array(
                'title'       => __('API Base URL (Dev Only)', 'deed-pay-woo'),
                'type'        => 'text',
                'description' => __('Leave as default for production. Enter your tunnel URL (e.g., api.webwox.xyz) for testing.', 'deed-pay-woo'),
                'default'     => 'https://api.usedeed.com',
                'desc_tip'    => true,
            ),
            'api_secret_key' => array(
                'title'       => __('API Secret Key', 'deed-pay-woo'),
                'type'        => 'password',
                'description' => __('Enter your DEED Pay API Secret Key. Format: DDSK_test_xxx or DDSK_live_xxx', 'deed-pay-woo'),
                'default'     => '',
                'desc_tip'    => false,
                'custom_attributes' => array(
                    'autocomplete' => 'new-password',
                ),
            ),
            'api_public_key' => array(
                'title'       => __('API Public Key (Optional)', 'deed-pay-woo'),
                'type'        => 'text',
                'description' => __('Enter your DEED Pay API Public Key. Format: DDPK_xxx (for frontend use if needed)', 'deed-pay-woo'),
                'default'     => '',
                'desc_tip'    => true,
            ),
            'webhook_secret' => array(
                'title'       => __('Webhook Secret', 'deed-pay-woo'),
                'type'        => 'password',
                'description' => sprintf(
                    /* translators: %1$s: Webhook URL, %2$s: Redirect domains note */
                    __('Enter your webhook secret for signature verification. Webhook URL: %1$s %2$s', 'deed-pay-woo'),
                    '<code class="deedpay-webhook-url">' . esc_html($webhook_url) . '</code>',
                    '<br><small>' . __('<strong>Important:</strong> Configure allowed redirect domains in DEED Pay Business Portal for successUrl and cancelUrl validation.', 'deed-pay-woo') . '</small>'
                ),
                'default'     => '',
                'desc_tip'    => false,
                'custom_attributes' => array(
                    'autocomplete' => 'new-password',
                ),
            ),
            'auto_release_days' => array(
                'title'       => __('Auto-Release Days', 'deed-pay-woo'),
                'type'        => 'number',
                'description' => __('Number of days after payment to automatically release funds (3-30 days). Default: 7', 'deed-pay-woo'),
                'default'     => 7,
                'custom_attributes' => array(
                    'min'  => 3,
                    'max'  => 30,
                    'step' => 1,
                ),
                'desc_tip'    => true,
            ),
            'product_type' => array(
                'title'       => __('Default Product Type', 'deed-pay-woo'),
                'type'        => 'select',
                'description' => __('Default product type for escrow auto-release timeline. Will auto-detect based on order items.', 'deed-pay-woo'),
                'default'     => 'PHYSICAL',
                'options'     => array(
                    'PHYSICAL' => __('Physical Products', 'deed-pay-woo'),
                    'DIGITAL'  => __('Digital Products', 'deed-pay-woo'),
                    'SERVICE'  => __('Services', 'deed-pay-woo'),
                ),
                'desc_tip'    => true,
            ),
            'debug_mode' => array(
                'title'       => __('Debug Mode', 'deed-pay-woo'),
                'label'       => __('Enable Debug Logging', 'deed-pay-woo'),
                'type'        => 'checkbox',
                'description' => sprintf(
                    /* translators: %s: Log file path */
                    __('Log detailed debug information. Log files: %s', 'deed-pay-woo'),
                    '<code>wp-content/uploads/wc-logs/deedpay-*.log</code>'
                ),
                'default'     => 'no',
                'desc_tip'    => false,
            ),
        );
    }


    /**
     * Validate API secret key field
     *
     * @param array $settings Settings array
     * @return array
     */
    public function validate_api_secret_key($settings) {
        if (isset($settings['api_secret_key']) && !empty($settings['api_secret_key'])) {
            $value = sanitize_text_field($settings['api_secret_key']);

            // Validate API key format
            if (!preg_match('/^DDSK_(test|live)_[a-zA-Z0-9]+$/', $value)) {
                WC_Admin_Settings::add_error(__('Invalid API Secret Key format. Should start with DDSK_test_ or DDSK_live_', 'deed-pay-woo'));
                $settings['api_secret_key'] = '';
            }
        }

        return $settings;
    }

    /**
     * Process admin options
     */
    public function process_admin_options() {
        parent::process_admin_options();

        // Validate auto_release_days
        $auto_release_days = absint($this->get_option('auto_release_days'));
        if ($auto_release_days < 3 || $auto_release_days > 30) {
            WC_Admin_Settings::add_error(__('Auto-Release Days must be between 3 and 30.', 'deed-pay-woo'));
            $this->settings['auto_release_days'] = 7;
            $this->update_option('auto_release_days', 7);
        }
    }

    /**
     * Process payment
     *
     * @param int $order_id Order ID
     * @return array
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);

        if (!$order) {
            wc_add_notice(__('Order not found.', 'deed-pay-woo'), 'error');
            return array('result' => 'fail');
        }

        // Validate settings
        if (empty($this->api_secret_key)) {
            wc_add_notice(__('DEED Pay API Secret Key is not configured.', 'deed-pay-woo'), 'error');
            $this->logger->error('API Secret Key not configured', array('order_id' => $order_id));
            return array('result' => 'fail');
        }

        // Validate customer email
        $customer_email = $order->get_billing_email();
        if (empty($customer_email) || !is_email($customer_email)) {
            wc_add_notice(__('A valid email address is required to complete payment.', 'deed-pay-woo'), 'error');
            $this->logger->error('Invalid customer email', array('order_id' => $order_id, 'email' => $customer_email));
            return array('result' => 'fail');
        }

        // Initialize API client
        $api_base_url = $this->get_option('api_base_url', 'https://api.usedeed.com');
        $api_client = new DeedPay_Api_Client($this->api_secret_key, $this->test_mode, $api_base_url);

        // Map WooCommerce Order to DEED Pay CreatePaymentDto
        $payment_data = $this->map_order_to_payment_intent($order);

        try {
            // Create payment intent via DEED Pay API
            $response = $api_client->create_payment_intent($payment_data);

            // Validate response structure: { success: true, data: {...} }
            if (!$response || !isset($response['success']) || !$response['success']) {
                $error_message = isset($response['error']['message']) ? $response['error']['message'] : __('Failed to create payment intent.', 'deed-pay-woo');
                throw new Exception($error_message);
            }

            // Ensure data exists in response
            if (!isset($response['data']) || !is_array($response['data'])) {
                throw new Exception(__('Invalid response format from DEED Pay API.', 'deed-pay-woo'));
            }

            $payment_intent = $response['data'];

            // Handle idempotent response (existing payment intent returned)
            if (isset($payment_intent['idempotencyKey']) && $payment_intent['idempotencyKey'] === $payment_data['idempotencyKey']) {
                $this->logger->info('Existing payment intent returned (idempotent)', array(
                    'order_id'       => $order_id,
                    'reference_code' => $payment_intent['referenceCode'],
                    'idempotency_key' => $payment_data['idempotencyKey'],
                ));
            }

            // Validate required fields in response
            if (empty($payment_intent['referenceCode']) || empty($payment_intent['checkoutUrl'])) {
                throw new Exception(__('Invalid payment intent response: missing required fields.', 'deed-pay-woo'));
            }

            // Store reference code and payment details in order meta
            $order->update_meta_data('_deedpay_reference_code', sanitize_text_field($payment_intent['referenceCode']));
            $order->update_meta_data('_deedpay_payment_id', isset($payment_intent['paymentId']) ? sanitize_text_field($payment_intent['paymentId']) : '');
            $order->update_meta_data('_deedpay_checkout_url', esc_url_raw($payment_intent['checkoutUrl']));
            $order->update_meta_data('_deedpay_status', isset($payment_intent['status']) ? sanitize_text_field($payment_intent['status']) : 'PENDING');
            $order->update_meta_data('_deedpay_idempotency_key', sanitize_text_field($payment_data['idempotencyKey']));
            
            // Store additional fields if available (for future use)
            if (isset($payment_intent['deedLink'])) {
                $order->update_meta_data('_deedpay_deed_link', esc_url_raw($payment_intent['deedLink']));
            }
            if (isset($payment_intent['qrCodeUrl'])) {
                $order->update_meta_data('_deedpay_qr_code_url', esc_url_raw($payment_intent['qrCodeUrl']));
            }
            if (isset($payment_intent['expiresAt'])) {
                $order->update_meta_data('_deedpay_expires_at', sanitize_text_field($payment_intent['expiresAt']));
            }
            
            $order->save_meta_data();

            // Update order status to "Pending Payment"
            $order->update_status('pending', __('Awaiting DEED Pay payment', 'deed-pay-woo'));

            // Clear cart
            WC()->cart->empty_cart();

            // Log successful payment intent creation
            $this->logger->info('Payment intent created', array(
                'order_id'       => $order_id,
                'reference_code' => $payment_intent['referenceCode'],
            ));

            // Redirect to DEED Pay checkout page
            return array(
                'result'   => 'success',
                'redirect' => esc_url_raw($payment_intent['checkoutUrl']),
            );

        } catch (Exception $e) {
            // Get error code from exception (if available)
            $error_code = $e->getCode();
            $error_message = $e->getMessage();

            // Log error with full context
            $this->logger->error('Payment intent creation failed: ' . $error_message, array(
                'order_id'    => $order_id,
                'error'       => $error_message,
                'error_code'  => $error_code,
            ));

            // Provide user-friendly error messages based on error code
            $user_message = __('Payment processing error. Please try again or contact support.', 'deed-pay-woo');

            // Handle specific error codes from DEED Pay API
            if (strpos($error_message, 'INVALID_REDIRECT_URL') !== false || strpos($error_message, 'redirect') !== false) {
                $user_message = __('Payment configuration error: Redirect URLs must be whitelisted in DEED Pay Business Portal.', 'deed-pay-woo');
            } elseif (strpos($error_message, 'DUPLICATE_ORDER_ID') !== false) {
                $user_message = __('This order has already been processed. Please create a new order.', 'deed-pay-woo');
            } elseif (strpos($error_message, 'UNAUTHORIZED') !== false || strpos($error_message, 'Invalid API key') !== false) {
                $user_message = __('Payment gateway configuration error. Please contact the store administrator.', 'deed-pay-woo');
                // Don't expose API key issues to customers
            } elseif (strpos($error_message, 'RATE_LIMIT') !== false) {
                $user_message = __('Too many requests. Please wait a moment and try again.', 'deed-pay-woo');
            }

            // Add error notice
            wc_add_notice($user_message, 'error');

            return array('result' => 'fail');
        }
    }

    /**
     * Map WooCommerce Order to DEED Pay CreatePaymentDto
     *
     * @param WC_Order $order Order object
     * @return array Payment data
     */
    private function map_order_to_payment_intent($order) {
        // Get order items
        $items = array();
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            $items[] = array(
                'name'        => sanitize_text_field($item->get_name()),
                'quantity'    => absint($item->get_quantity()),
                'unitPrice'   => round(floatval($item->get_subtotal()) / absint($item->get_quantity()), 2),
                'description' => $product ? sanitize_text_field($product->get_short_description()) : '',
            );
        }

        // Determine currency (convert to DEED Pay currency enum)
        $wc_currency = $order->get_currency();
        // DEED Pay uses ISO currency codes (NGN, USD, GBP, etc.)
        $currency = strtoupper(sanitize_text_field($wc_currency));

        // Build success URL (WooCommerce order received page)
        $success_url = $this->get_return_url($order);

        // Build cancel URL (cart page)
        $cancel_url = wc_get_cart_url();

        // Build webhook URL (DEED Pay will send events to this URL)
        $webhook_url = home_url('/?wc-api=wc_gateway_deedpay');

        // Get customer email (already validated in process_payment)
        $customer_email = sanitize_email($order->get_billing_email());

        // Get customer phone (optional)
        $customer_phone = $order->get_billing_phone();
        if (!empty($customer_phone)) {
            $customer_phone = sanitize_text_field($customer_phone);
        } else {
            $customer_phone = null;
        }

        // Generate idempotency key (prevent duplicate payment intents)
        $idempotency_key = sprintf(
            /* translators: 1: WooCommerce order ID, 2: Unix timestamp, 3: Random string */
            'woo_%d_%d_%s',
            $order->get_id(),
            time(),
            wp_generate_password(12, false)
        );

        // Build metadata (store WooCommerce order info)
        $metadata = array(
            'woocommerce_order_id' => $order->get_id(),
            'woocommerce_order_key' => $order->get_order_key(),
            'store_name' => get_bloginfo('name'),
            'store_url' => home_url(),
        );

        // Determine product type based on order items
        // ESCROW SAFETY: If ANY item is physical, the order is PHYSICAL
        $product_type = $this->detect_product_type($order);

        return array(
            'orderId'             => (string) $order->get_id(),
            'amount'              => round(floatval($order->get_total()), 2),
            'currency'            => $currency,
            'customerEmail'       => $customer_email,
            'customerPhone'       => $customer_phone,
            'items'               => $items,
            'successUrl'          => esc_url_raw($success_url),
            'cancelUrl'           => esc_url_raw($cancel_url),
            'webhookUrl'          => esc_url_raw($webhook_url), // ✅ Required: Webhook URL for event notifications
            'pageType'            => 'checkout',
            'productType'         => $product_type,
            'autoReleaseAfterDays' => absint($this->auto_release_days),
            'idempotencyKey'      => $idempotency_key,
            'metadata'            => $metadata,
        );
    }

    /**
     * Detect product type based on order items
     * ESCROW SAFETY LOGIC: If ANY item is physical, the order is PHYSICAL
     *
     * @param WC_Order $order Order object
     * @return string Product type (PHYSICAL, DIGITAL, SERVICE)
     */
    private function detect_product_type($order) {
        // ESCROW SAFETY: If ANY item is physical, the order is PHYSICAL
        // We only downgrade to DIGITAL if the entire cart is virtual/downloadable
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if ($product) {
                // If we find a single physical item, return PHYSICAL immediately
                if (!$product->is_virtual() && !$product->is_downloadable()) {
                    return 'PHYSICAL';
                }
            }
        }

        // If we get here, it means NO physical items were found
        // Use default from settings (or DIGITAL if all items are virtual)
        return $this->product_type;
    }

    /**
     * Check payment status on order received page
     *
     * @param int $order_id Order ID
     */
    public function check_payment_status($order_id) {
        $order = wc_get_order($order_id);

        if (!$order || $order->get_payment_method() !== 'deedpay') {
            return;
        }

        // Check if order is still pending
        if ($order->get_status() === 'pending') {
            $reference_code = $order->get_meta('_deedpay_reference_code');

            if (!empty($reference_code)) {
                // Optional: Poll DEED Pay API to check status
                // This is a fallback if webhook hasn't arrived yet
                try {
                    $api_base_url = $this->get_option('api_base_url', 'https://api.usedeed.com');
                    $api_client = new DeedPay_Api_Client($this->api_secret_key, $this->test_mode, $api_base_url);
                    $payment_intent_response = $api_client->get_payment_intent($reference_code);

                    if ($payment_intent_response && isset($payment_intent_response['success']) && $payment_intent_response['success']) {
                        if (isset($payment_intent_response['data']['status'])) {
                            $status = sanitize_text_field($payment_intent_response['data']['status']);

                            if ($status === 'PAID' && $order->get_status() === 'pending') {
                                // Update order status if webhook hasn't processed yet
                                $order->update_status('processing', __('Payment verified via status check', 'deed-pay-woo'));
                                $order->update_meta_data('_deedpay_status', 'PAID');
                                $order->save_meta_data();
                            } elseif ($status === 'EXPIRED' && $order->get_status() === 'pending') {
                                // Payment expired
                                $order->update_status('cancelled', __('Payment expired', 'deed-pay-woo'));
                                $order->update_meta_data('_deedpay_status', 'EXPIRED');
                                $order->save_meta_data();
                            }
                        }
                    }
                } catch (Exception $e) {
                    // Silently fail - webhook will handle status updates
                    $this->logger->debug('Status check failed: ' . $e->getMessage(), array('order_id' => $order_id));
                }
            }
        }
    }

    /**
     * Process refund
     *
     * @param int    $order_id Order ID
     * @param float  $amount   Refund amount (null for full refund)
     * @param string $reason   Refund reason
     * @return bool|WP_Error
     */
    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = wc_get_order($order_id);

        if (!$order || $order->get_payment_method() !== 'deedpay') {
            return new WP_Error('invalid_order', __('Invalid order or payment method', 'deed-pay-woo'));
        }

        $reference_code = $order->get_meta('_deedpay_reference_code');

        if (empty($reference_code)) {
            return new WP_Error('no_reference', __('No DEED Pay reference code found', 'deed-pay-woo'));
        }

        // Validate API key
        if (empty($this->api_secret_key)) {
            return new WP_Error('no_api_key', __('DEED Pay API Secret Key is not configured', 'deed-pay-woo'));
        }

        // Call DEED Pay refund API
        $api_base_url = $this->get_option('api_base_url', 'https://api.usedeed.com');
        $api_client = new DeedPay_Api_Client($this->api_secret_key, $this->test_mode, $api_base_url);

        try {
            $refund_data = array(
                'reason' => sanitize_text_field($reason),
            );

            // If amount specified, it's a partial refund
            if ($amount !== null) {
                $refund_data['amount'] = round(floatval($amount), 2);
            }

            $response = $api_client->refund_payment($reference_code, $refund_data);

            if ($response && isset($response['success']) && $response['success'] && isset($response['data'])) {
                $refund_response = $response['data'];
                $refund_amount = isset($refund_response['refundAmount']) ? floatval($refund_response['refundAmount']) : $amount;
                $is_partial = isset($refund_response['isPartial']) ? $refund_response['isPartial'] : ($amount !== null);
                $refund_id = isset($refund_response['refundId']) ? sanitize_text_field($refund_response['refundId']) : '';
                $refund_status = isset($refund_response['status']) ? sanitize_text_field($refund_response['status']) : 'PROCESSED';

                // Store refund details in order meta
                if ($refund_id) {
                    $order->update_meta_data('_deedpay_refund_id', $refund_id);
                }
                $order->update_meta_data('_deedpay_refund_status', $refund_status);

                // Handle refund status
                if ($refund_status === 'PENDING_APPROVAL') {
                    // Large refund requires approval
                    $refund_message = sprintf(
                        /* translators: 1: Refund ID, 2: Refund amount with currency symbol */
                        __('Refund request submitted via DEED Pay (Refund ID: %1$s). Amount: %2$s. Pending approval.', 'deed-pay-woo'),
                        $refund_id ?: 'N/A',
                        wc_price($refund_amount)
                    );
                    $order->update_status('on-hold', $refund_message);
                } elseif ($refund_status === 'PROCESSED' || $refund_status === 'COMPLETED') {
                    // Refund processed successfully
                    if ($is_partial) {
                        $refund_message = sprintf(
                            /* translators: 1: Refund amount with currency symbol, 2: Remaining amount with currency symbol */
                            __('Partial refund processed via DEED Pay. Amount: %1$s. Remaining: %2$s', 'deed-pay-woo'),
                            wc_price($refund_amount),
                            wc_price($order->get_total() - $refund_amount)
                        );
                    } else {
                        $refund_message = sprintf(
                            /* translators: %s: Refund amount with currency symbol */
                            __('Full refund processed via DEED Pay. Amount: %s', 'deed-pay-woo'),
                            wc_price($refund_amount)
                        );
                    }

                    $order->add_order_note($refund_message);

                    // Only update status to refunded if it's a full refund
                    if (!$is_partial) {
                        $order->update_status('refunded', $refund_message);
                    }
                } elseif ($refund_status === 'FAILED') {
                    // Refund failed
                    $refund_message = sprintf(
                        /* translators: %s: Refund amount with currency symbol */
                        __('Refund failed via DEED Pay. Manual bank transfer may be required. Amount: %s', 'deed-pay-woo'),
                        wc_price($refund_amount)
                    );
                    $order->add_order_note($refund_message);
                    $order->update_status('on-hold', $refund_message);
                }

                $order->save_meta_data();

                $this->logger->info('Refund processed', array(
                    'order_id'       => $order_id,
                    'refund_id'      => $refund_id,
                    'amount'         => $refund_amount,
                    'is_partial'     => $is_partial,
                    'status'         => $refund_status,
                    'reference_code' => $reference_code,
                ));

                return true;
            }

            return new WP_Error('refund_failed', __('Refund failed', 'deed-pay-woo'));

        } catch (Exception $e) {
            $this->logger->error('Refund error: ' . $e->getMessage(), array(
                'order_id' => $order_id,
                'error'    => $e->getMessage(),
            ));

            return new WP_Error('refund_error', $e->getMessage());
        }
    }
}
