<?php
/**
 * DEED Pay Webhook Handler
 *
 * Processes incoming webhooks from DEED Pay.
 * Handles signature verification, idempotency, and event processing.
 *
 * @package DEEDPay
 * @since 0.1.1
 */

if (!defined('ABSPATH')) {
    exit;
}

class DeedPay_Webhook {
    /**
     * Logger instance
     *
     * @var DeedPay_Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct() {
        $this->logger = DeedPay_Logger::get_instance();
        // Register webhook endpoint
        add_action('woocommerce_api_wc_gateway_deedpay', array($this, 'handle_webhook'));
    }

    /**
     * Handle incoming webhook
     */
    public function handle_webhook() {
        // Get raw request body (must be raw for signature verification)
        $payload = file_get_contents('php://input');

        if (empty($payload)) {
            $this->logger->error('Empty webhook payload received');
            http_response_code(400);
            exit;
        }

        $payload_json = json_decode($payload, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->logger->error('Invalid JSON payload received', array(
                'json_error' => json_last_error_msg(),
                'payload_preview' => substr($payload, 0, 200),
            ));
            http_response_code(400);
            exit;
        }

        // Get headers - WordPress/SERVER normalizes headers to HTTP_HEADER_NAME format
        // X-Deed-Signature becomes HTTP_X_DEED_SIGNATURE in $_SERVER
        $signature = '';
        $timestamp = '';
        $event_type = '';

        // Try to get headers using getallheaders() if available (more reliable)
        $headers = array();
        if (function_exists('getallheaders')) {
            $all_headers = getallheaders();
            if (is_array($all_headers)) {
                foreach ($all_headers as $key => $value) {
                    $headers[strtolower($key)] = $value;
                }
            }
        }

        // Check getallheaders() first, then fall back to $_SERVER
        if (!empty($headers)) {
            $signature = isset($headers['x-deed-signature']) ? sanitize_text_field($headers['x-deed-signature']) : '';
            $timestamp = isset($headers['x-deed-timestamp']) ? sanitize_text_field($headers['x-deed-timestamp']) : '';
            $event_type = isset($headers['x-deed-event']) ? sanitize_text_field($headers['x-deed-event']) : '';
        }

        // Fallback to $_SERVER (check multiple possible formats)
        if (empty($signature)) {
            $header_variants = array(
                'HTTP_X_DEED_SIGNATURE',
                'HTTP_X_DEED_signature',
                'HTTP_X_DEED_Signature',
            );
            foreach ($header_variants as $variant) {
                if (isset($_SERVER[$variant])) {
                    $signature = sanitize_text_field(wp_unslash($_SERVER[$variant]));
                    break;
                }
            }
        }

        if (empty($timestamp)) {
            $header_variants = array(
                'HTTP_X_DEED_TIMESTAMP',
                'HTTP_X_DEED_timestamp',
                'HTTP_X_DEED_Timestamp',
            );
            foreach ($header_variants as $variant) {
                if (isset($_SERVER[$variant])) {
                    $timestamp = sanitize_text_field(wp_unslash($_SERVER[$variant]));
                    break;
                }
            }
        }

        if (empty($event_type)) {
            $header_variants = array(
                'HTTP_X_DEED_EVENT',
                'HTTP_X_DEED_event',
                'HTTP_X_DEED_Event',
            );
            foreach ($header_variants as $variant) {
                if (isset($_SERVER[$variant])) {
                    $event_type = sanitize_text_field(wp_unslash($_SERVER[$variant]));
                    break;
                }
            }
        }

        // Get gateway settings
        $gateway_settings = get_option('deedpay_settings', array());
        $webhook_secret = isset($gateway_settings['webhook_secret']) ? $gateway_settings['webhook_secret'] : '';
        $api_secret_key = isset($gateway_settings['api_secret_key']) ? $gateway_settings['api_secret_key'] : '';
        $test_mode = isset($gateway_settings['test_mode']) ? $gateway_settings['test_mode'] === 'yes' : false;
        $api_base_url = isset($gateway_settings['api_base_url']) ? $gateway_settings['api_base_url'] : 'https://api.usedeed.com';

        if (empty($webhook_secret)) {
            $this->logger->error('Webhook secret not configured');
            http_response_code(500);
            exit;
        }

        if (empty($api_secret_key)) {
            $this->logger->error('API secret key not configured');
            http_response_code(500);
            exit;
        }

        // Verify signature
        $api_client = new DeedPay_Api_Client($api_secret_key, $test_mode, $api_base_url);

        $signature_valid = $api_client->verify_webhook_signature(
            $payload,
            $signature,
            $timestamp,
            $webhook_secret
        );

        if (!$signature_valid) {
            $this->logger->error('Invalid webhook signature', array(
                'signature_received' => DeedPay_Logger::mask_api_key($signature),
                'timestamp'          => $timestamp,
            ));
            http_response_code(401);
            exit;
        }

        // Verify timestamp (prevent replay attacks)
        // Webhook timestamp is Unix timestamp (seconds) as per API docs
        $current_time = time();
        
        // Handle both Unix timestamp (numeric) and ISO 8601 string formats
        if (is_numeric($timestamp)) {
            $payload_time = intval($timestamp);
        } else {
            $payload_time = strtotime($timestamp);
        }

        if ($payload_time === false || $payload_time <= 0) {
            $this->logger->error('Invalid webhook timestamp format', array('timestamp' => $timestamp));
            http_response_code(400);
            exit;
        }

        $time_difference = abs($current_time - $payload_time);

        // Allow 5 minutes tolerance
        if ($time_difference > 300) {
            $this->logger->error('Webhook timestamp outside tolerance', array(
                'timestamp'       => $timestamp,
                'time_difference' => $time_difference,
                'current_time'    => $current_time,
                'payload_time'    => $payload_time,
            ));
            http_response_code(400);
            exit;
        }

        // Extract event data
        $event_id = isset($payload_json['id']) ? sanitize_text_field($payload_json['id']) : '';
        $event = isset($payload_json['event']) ? sanitize_text_field($payload_json['event']) : $event_type;
        $event_data = isset($payload_json['data']) && is_array($payload_json['data']) ? $payload_json['data'] : array();
        $reference_code = isset($event_data['referenceCode']) ? sanitize_text_field($event_data['referenceCode']) : '';
        $order_id = isset($event_data['orderId']) ? sanitize_text_field($event_data['orderId']) : '';

        // Log webhook received
        $this->logger->info('Webhook received', array(
            'event_id'       => $event_id,
            'event'          => $event,
            'reference_code' => $reference_code,
            'order_id'       => $order_id,
        ));

        // Return 200 OK immediately (before processing)
        http_response_code(200);
        header('Content-Type: application/json');
        echo wp_json_encode(array('status' => 'received'));

        // Flush output if possible (allows processing to continue after response sent)
        if (function_exists('fastcgi_finish_request')) {
            fastcgi_finish_request();
        }
        ignore_user_abort(true);

        // Find order
        $order = $this->find_order($reference_code, $order_id);

        if (!$order) {
            $this->logger->warning('Order not found for webhook', array(
                'reference_code' => $reference_code,
                'order_id'       => $order_id,
            ));
            exit;
        }

        // Check idempotency (prevent duplicate processing)
        $processed_events = $order->get_meta('_deedpay_processed_events', true);
        if (!is_array($processed_events)) {
            $processed_events = array();
        }

        if (in_array($event_id, $processed_events, true)) {
            $this->logger->info('Webhook event already processed', array(
                'event_id' => $event_id,
                'order_id' => $order->get_id(),
            ));
            exit;
        }

        // Process event
        $this->process_webhook_event($order, $event, $event_data, $event_id);

        // Mark event as processed
        $processed_events[] = $event_id;
        $order->update_meta_data('_deedpay_processed_events', $processed_events);
        $order->save_meta_data();

        exit;
    }

    /**
     * Find order by reference code or order ID
     *
     * @param string $reference_code Reference code
     * @param string $order_id       Order ID
     * @return WC_Order|null
     */
    private function find_order($reference_code, $order_id) {
        global $wpdb;
        $order = null;

        // Try to find by reference code first using direct query (faster)
        if (!empty($reference_code)) {
            // Check cache first
            $cache_key = 'deedpay_order_' . md5($reference_code);
            $cached_order_id = wp_cache_get($cache_key, 'deedpay');

            if (false === $cached_order_id) {
                // Cache miss - query database
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $cached_order_id = $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT post_id FROM {$wpdb->postmeta} 
                        WHERE meta_key = '_deedpay_reference_code' 
                        AND meta_value = %s 
                        LIMIT 1",
                        $reference_code
                    )
                );

                // Cache the result for 1 hour (even if null)
                wp_cache_set($cache_key, $cached_order_id, 'deedpay', HOUR_IN_SECONDS);
            }

            if ($cached_order_id) {
                $order = wc_get_order(absint($cached_order_id));
                if ($order && $order->get_payment_method() === 'deedpay') {
                    return $order;
                }
            }
        }

        // Fallback to order ID - try direct lookup first
        if (!$order && !empty($order_id)) {
            $order = wc_get_order(absint($order_id));

            // Verify order uses DEED Pay gateway before continuing
            if ($order && $order->get_payment_method() === 'deedpay') {
                return $order;
            }

            // If order not found and we have API access, try fetching from DEED Pay API
            if (!$order) {
                $gateway_settings = get_option('deedpay_settings', array());
                $api_secret_key = isset($gateway_settings['api_secret_key']) ? $gateway_settings['api_secret_key'] : '';
                $api_base_url = isset($gateway_settings['api_base_url']) ? $gateway_settings['api_base_url'] : 'https://api.usedeed.com';

                if (!empty($api_secret_key)) {
                    try {
                        $api_client = new DeedPay_Api_Client(
                            $api_secret_key,
                            isset($gateway_settings['test_mode']) && $gateway_settings['test_mode'] === 'yes',
                            $api_base_url
                        );

                        $payment_response = $api_client->get_payment_by_order_id($order_id);

                        if ($payment_response && isset($payment_response['success']) && $payment_response['success']) {
                            $payment_data = $payment_response['data'];

                            // Now try to find order by reference code from API response using direct query
                            if (isset($payment_data['referenceCode'])) {
                                $ref_code = sanitize_text_field($payment_data['referenceCode']);
                                $ref_cache_key = 'deedpay_order_' . md5($ref_code);
                                $cached_ref_order_id = wp_cache_get($ref_cache_key, 'deedpay');

                                if (false === $cached_ref_order_id) {
                                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                                    $cached_ref_order_id = $wpdb->get_var(
                                        $wpdb->prepare(
                                            "SELECT post_id FROM {$wpdb->postmeta} 
                                            WHERE meta_key = '_deedpay_reference_code' 
                                            AND meta_value = %s 
                                            LIMIT 1",
                                            $ref_code
                                        )
                                    );

                                    // Cache the result for 1 hour
                                    wp_cache_set($ref_cache_key, $cached_ref_order_id, 'deedpay', HOUR_IN_SECONDS);
                                }

                                if ($cached_ref_order_id) {
                                    $order = wc_get_order(absint($cached_ref_order_id));
                                    if ($order && $order->get_payment_method() === 'deedpay') {
                                        return $order;
                                    }
                                }
                            }
                        }
                    } catch (Exception $e) {
                        // Silently fail - order lookup via API failed
                        $this->logger->debug('Failed to fetch payment by order ID from API: ' . $e->getMessage());
                    }
                }
            }
        }

        // Verify order uses DEED Pay gateway (final check)
        if ($order && $order->get_payment_method() !== 'deedpay') {
            $this->logger->warning('Order found but uses different payment method', array(
                'order_id'         => $order->get_id(),
                'payment_method'   => $order->get_payment_method(),
            ));
            return null;
        }

        return $order;
    }

    /**
     * Process webhook event and update order status
     *
     * @param WC_Order $order      Order object
     * @param string   $event      Event type
     * @param array    $event_data Event data
     * @param string   $event_id   Event ID
     */
    private function process_webhook_event($order, $event, $event_data, $event_id) {
        $reference_code = isset($event_data['referenceCode']) ? sanitize_text_field($event_data['referenceCode']) : '';
        $status = isset($event_data['status']) ? sanitize_text_field($event_data['status']) : '';

        // Get event timestamp for out-of-order event handling
        // Event timestamp from payload can be ISO 8601 string or Unix timestamp
        $event_timestamp = time();
        if (isset($event_data['timestamp'])) {
            $event_ts = $event_data['timestamp'];
            if (is_numeric($event_ts)) {
                $event_timestamp = intval($event_ts);
            } else {
                $parsed_timestamp = strtotime($event_ts);
                if ($parsed_timestamp !== false) {
                    $event_timestamp = $parsed_timestamp;
                }
            }
        }

        // Check if event is older than last processed event (ignore out-of-order events)
        $last_event_timestamp = $order->get_meta('_deedpay_last_event_timestamp', true);
        if (!$last_event_timestamp) {
            $last_event_timestamp = 0;
        }

        if ($event_timestamp < $last_event_timestamp) {
            $this->logger->warning('Ignoring out-of-order event', array(
                'event'               => $event,
                'event_timestamp'     => $event_timestamp,
                'last_event_timestamp' => $last_event_timestamp,
                'order_id'            => $order->get_id(),
            ));
            return;
        }

        // Update last event timestamp
        $order->update_meta_data('_deedpay_last_event_timestamp', $event_timestamp);

        // Process based on event type
        switch ($event) {
            case 'payment.created':
                // Payment intent created (already handled in process_payment)
                $this->logger->info('Payment intent created webhook', array(
                    'order_id' => $order->get_id(),
                ));
                $order->update_meta_data('_deedpay_status', 'PENDING');
                $order->save_meta_data();
                break;

            case 'payment.paid':
            case 'payment.escrowed':
                // Customer paid, funds locked in escrow
                $order->update_status(
                    'processing',
                    sprintf(
                        /* translators: %s: DEED Pay payment reference code */
                        __('Payment received via DEED Pay (Reference: %s). Funds held in escrow.', 'deed-pay-woo'),
                        $reference_code
                    )
                );

                // Update order meta
                $order->update_meta_data('_deedpay_status', 'PAID');
                $order->update_meta_data('_deedpay_paid_at', current_time('mysql'));

                if (isset($event_data['paymentMethod'])) {
                    $order->update_meta_data('_deedpay_payment_method', sanitize_text_field($event_data['paymentMethod']));
                }

                // Reduce stock (if WooCommerce inventory management is enabled)
                wc_reduce_stock_levels($order->get_id());

                // Send order confirmation email (WooCommerce handles this)
                if (function_exists('WC')) {
                    WC()->mailer()->emails['WC_Email_New_Order']->trigger($order->get_id());
                }

                $order->save_meta_data();

                $this->logger->info('Payment received', array(
                    'order_id'       => $order->get_id(),
                    'reference_code' => $reference_code,
                    'status'         => $status,
                ));
                break;

            case 'payment.shipped':
                // Order marked as shipped
                $tracking_number = isset($event_data['trackingNumber']) ? sanitize_text_field($event_data['trackingNumber']) : '';
                $carrier = isset($event_data['carrier']) ? sanitize_text_field($event_data['carrier']) : '';
                
                $order->update_status(
                    'processing',
                    sprintf(
                        /* translators: %1$s: DEED Pay payment reference code, %2$s: Optional tracking information */
                        __('Order shipped via DEED Pay (Reference: %1$s)%2$s.', 'deed-pay-woo'),
                        $reference_code,
                        $tracking_number ? ' - Tracking: ' . $tracking_number : ''
                    )
                );

                if ($tracking_number) {
                    $order->update_meta_data('_deedpay_tracking_number', $tracking_number);
                }
                if ($carrier) {
                    $order->update_meta_data('_deedpay_carrier', $carrier);
                }
                $order->update_meta_data('_deedpay_status', 'SHIPPED');
                $order->save_meta_data();

                $this->logger->info('Order shipped', array(
                    'order_id'        => $order->get_id(),
                    'reference_code'  => $reference_code,
                    'tracking_number' => $tracking_number,
                ));
                break;

            case 'payment.delivered':
                // Order delivered to customer
                $order->update_status(
                    'processing',
                    sprintf(
                        /* translators: %s: DEED Pay payment reference code */
                        __('Order delivered via DEED Pay (Reference: %s). Awaiting customer confirmation.', 'deed-pay-woo'),
                        $reference_code
                    )
                );

                $order->update_meta_data('_deedpay_status', 'DELIVERED');
                $order->update_meta_data('_deedpay_delivered_at', current_time('mysql'));
                $order->save_meta_data();

                $this->logger->info('Order delivered', array(
                    'order_id'       => $order->get_id(),
                    'reference_code' => $reference_code,
                ));
                break;

            case 'payment.completed':
                // Funds released to business (customer confirmed receipt or auto-release)
                $order->update_status(
                    'completed',
                    sprintf(
                        /* translators: %s: DEED Pay payment reference code */
                        __('Payment completed via DEED Pay (Reference: %s). Funds released.', 'deed-pay-woo'),
                        $reference_code
                    )
                );

                $order->update_meta_data('_deedpay_status', 'COMPLETED');
                $order->update_meta_data('_deedpay_completed_at', current_time('mysql'));
                $order->save_meta_data();

                $this->logger->info('Payment completed', array(
                    'order_id'       => $order->get_id(),
                    'reference_code' => $reference_code,
                ));
                break;

            case 'payment.expired':
                // Payment intent expired (24 hours without payment)
                $order->update_status(
                    'cancelled',
                    sprintf(
                        /* translators: %s: DEED Pay payment reference code */
                        __('Payment expired via DEED Pay (Reference: %s).', 'deed-pay-woo'),
                        $reference_code
                    )
                );

                $order->update_meta_data('_deedpay_status', 'EXPIRED');
                $order->save_meta_data();

                $this->logger->warning('Payment expired', array(
                    'order_id'       => $order->get_id(),
                    'reference_code' => $reference_code,
                ));
                break;

            case 'payment.failed':
                // Payment failed
                $order->update_status(
                    'failed',
                    sprintf(
                        /* translators: %s: DEED Pay payment reference code */
                        __('Payment failed via DEED Pay (Reference: %s).', 'deed-pay-woo'),
                        $reference_code
                    )
                );

                $order->update_meta_data('_deedpay_status', 'FAILED');
                $order->save_meta_data();

                $this->logger->error('Payment failed', array(
                    'order_id'       => $order->get_id(),
                    'reference_code' => $reference_code,
                ));
                break;

            case 'payment.refunded':
            case 'payment.partially_refunded':
                // Payment refunded - Handle different refund statuses
                $is_partial = ($event === 'payment.partially_refunded');
                $refund_amount = isset($event_data['refundAmount']) ? floatval($event_data['refundAmount']) : null;
                $refund_status = isset($event_data['refundStatus']) ? sanitize_text_field($event_data['refundStatus']) : 'COMPLETED';
                $refund_id = isset($event_data['refundId']) ? sanitize_text_field($event_data['refundId']) : '';
                $total_refunded = isset($event_data['totalRefunded']) ? floatval($event_data['totalRefunded']) : null;
                $remaining_amount = isset($event_data['remainingAmount']) ? floatval($event_data['remainingAmount']) : null;

                if ($refund_amount === null) {
                    $refund_amount = $order->get_total();
                }

                // Handle refund statuses
                switch ($refund_status) {
                    case 'PENDING_APPROVAL':
                        // Large refund requires admin approval
                        $order->update_status(
                            'on-hold',
                            sprintf(
                                /* translators: 1: DEED Pay payment reference code, 2: Refund amount with currency symbol */
                                __('Refund pending approval via DEED Pay (Reference: %1$s). Amount: %2$s', 'deed-pay-woo'),
                                $reference_code,
                                wc_price($refund_amount)
                            )
                        );
                        $order->update_meta_data('_deedpay_refund_status', 'PENDING_APPROVAL');
                        $this->logger->warning('Refund pending approval', array(
                            'order_id'       => $order->get_id(),
                            'reference_code' => $reference_code,
                            'amount'         => $refund_amount,
                        ));
                        break;

                    case 'APPROVED':
                        // Refund approved, processing
                        $order->add_order_note(
                            sprintf(
                                /* translators: 1: DEED Pay payment reference code, 2: Refund amount with currency symbol */
                                __('Refund approved and processing via DEED Pay (Reference: %1$s). Amount: %2$s', 'deed-pay-woo'),
                                $reference_code,
                                wc_price($refund_amount)
                            )
                        );
                        $order->update_meta_data('_deedpay_refund_status', 'APPROVED');
                        $this->logger->info('Refund approved', array(
                            'order_id'       => $order->get_id(),
                            'reference_code' => $reference_code,
                        ));
                        break;

                    case 'REJECTED':
                        // Refund rejected
                        $order->update_status(
                            'processing',
                            sprintf(
                                /* translators: %s: DEED Pay payment reference code */
                                __('Refund request rejected via DEED Pay (Reference: %s).', 'deed-pay-woo'),
                                $reference_code
                            )
                        );
                        $order->update_meta_data('_deedpay_refund_status', 'REJECTED');
                        $this->logger->warning('Refund rejected', array(
                            'order_id'       => $order->get_id(),
                            'reference_code' => $reference_code,
                        ));
                        break;

                    case 'PROCESSING':
                        // Refund being processed
                        $order->add_order_note(
                            sprintf(
                                /* translators: 1: DEED Pay payment reference code, 2: Refund amount with currency symbol */
                                __('Refund processing via DEED Pay (Reference: %1$s). Amount: %2$s', 'deed-pay-woo'),
                                $reference_code,
                                wc_price($refund_amount)
                            )
                        );
                        $order->update_meta_data('_deedpay_refund_status', 'PROCESSING');
                        break;

                    case 'FAILED':
                        // Refund failed - manual bank transfer required
                        $order->update_status(
                            'on-hold',
                            sprintf(
                                /* translators: 1: DEED Pay payment reference code, 2: Refund amount with currency symbol */
                                __('Refund failed via DEED Pay (Reference: %1$s). Manual bank transfer required. Amount: %2$s', 'deed-pay-woo'),
                                $reference_code,
                                wc_price($refund_amount)
                            )
                        );
                        $order->update_meta_data('_deedpay_refund_status', 'FAILED');
                        $this->logger->error('Refund failed - manual transfer required', array(
                            'order_id'       => $order->get_id(),
                            'reference_code' => $reference_code,
                            'amount'         => $refund_amount,
                        ));
                        break;

                    case 'COMPLETED':
                    default:
                        // Refund completed - create WooCommerce refund
                        if ($is_partial) {
                            /* translators: Refund reason for partial refunds */
                            $refund_reason = __('Partial refund via DEED Pay', 'deed-pay-woo');
                        } else {
                            /* translators: Refund reason for full refunds */
                            $refund_reason = __('Refunded via DEED Pay', 'deed-pay-woo');
                        }
                        $refund = wc_create_refund(array(
                            'amount'   => $refund_amount,
                            'reason'   => $refund_reason,
                            'order_id' => $order->get_id(),
                        ));

                        if (is_wp_error($refund)) {
                            $this->logger->error('Failed to create refund', array(
                                'order_id' => $order->get_id(),
                                'error'    => $refund->get_error_message(),
                            ));
                        } else {
                            // Store refund details
                            if ($refund_id) {
                                $order->update_meta_data('_deedpay_refund_id', $refund_id);
                            }
                            if ($total_refunded !== null) {
                                $order->update_meta_data('_deedpay_total_refunded', $total_refunded);
                            }
                            if ($remaining_amount !== null) {
                                $order->update_meta_data('_deedpay_remaining_amount', $remaining_amount);
                            }

                            // Update order status and notes
                            if ($is_partial) {
                                // Partial refund - keep order in current status or processing
                                $refund_note = sprintf(
                                    /* translators: 1: DEED Pay payment reference code, 2: Refund amount with currency symbol, 3: Total refunded with currency symbol, 4: Remaining amount with currency symbol */
                                    __('Partial refund via DEED Pay (Reference: %1$s). Amount: %2$s. Total refunded: %3$s. Remaining: %4$s', 'deed-pay-woo'),
                                    $reference_code,
                                    wc_price($refund_amount),
                                    wc_price($total_refunded ?: $refund_amount),
                                    wc_price($remaining_amount ?: ($order->get_total() - $refund_amount))
                                );
                                $order->add_order_note($refund_note);
                                
                                // Update status to partial-refund if WooCommerce supports it, otherwise keep current status
                                if ($order->get_status() !== 'refunded') {
                                    // Don't change status for partial refunds
                                }
                            } else {
                                // Full refund
                                $order->update_status(
                                    'refunded',
                                    sprintf(
                                        /* translators: 1: DEED Pay payment reference code, 2: Refund amount with currency symbol */
                                        __('Refunded via DEED Pay (Reference: %1$s). Amount: %2$s', 'deed-pay-woo'),
                                        $reference_code,
                                        wc_price($refund_amount)
                                    )
                                );
                                $order->update_meta_data('_deedpay_status', 'REFUNDED');
                            }

                            $order->update_meta_data('_deedpay_refund_status', 'COMPLETED');
                            $order->save_meta_data();

                            $this->logger->info($is_partial ? 'Partial refund processed' : 'Payment refunded', array(
                                'order_id'       => $order->get_id(),
                                'reference_code' => $reference_code,
                                'amount'         => $refund_amount,
                                'total_refunded' => $total_refunded,
                                'remaining'      => $remaining_amount,
                                'refund_id'      => $refund_id,
                            ));
                        }
                        break;
                }

                $order->save_meta_data();
                break;

            case 'payment.disputed':
                // Dispute raised
                $order->update_status(
                    'on-hold',
                    sprintf(
                        /* translators: %s: DEED Pay payment reference code */
                        __('Dispute raised via DEED Pay (Reference: %s). Please review.', 'deed-pay-woo'),
                        $reference_code
                    )
                );

                $order->update_meta_data('_deedpay_status', 'DISPUTED');
                $order->save_meta_data();

                $this->logger->warning('Payment disputed', array(
                    'order_id'       => $order->get_id(),
                    'reference_code' => $reference_code,
                ));
                break;

            case 'payment.cancelled':
                // Payment cancelled
                $order->update_status(
                    'cancelled',
                    sprintf(
                        /* translators: %s: DEED Pay payment reference code */
                        __('Payment cancelled via DEED Pay (Reference: %s).', 'deed-pay-woo'),
                        $reference_code
                    )
                );

                $order->update_meta_data('_deedpay_status', 'CANCELLED');
                $order->save_meta_data();

                $this->logger->info('Payment cancelled', array(
                    'order_id'       => $order->get_id(),
                    'reference_code' => $reference_code,
                ));
                break;

            default:
                $this->logger->warning('Unknown webhook event', array(
                    'event'    => $event,
                    'order_id' => $order->get_id(),
                ));
        }
    }
}

// Initialize webhook handler
new DeedPay_Webhook();
