<?php
/**
 * DEED Pay Logger
 *
 * Centralized logging for debugging and monitoring.
 * Uses WooCommerce's built-in logger (wc_get_logger).
 *
 * @package DEEDPay
 * @since 0.1.1
 */

if (!defined('ABSPATH')) {
    exit;
}

class DeedPay_Logger {
    /**
     * Singleton instance
     *
     * @var DeedPay_Logger|null
     */
    private static $instance = null;

    /**
     * Debug mode enabled
     *
     * @var bool
     */
    private $enabled = false;

    /**
     * Get singleton instance
     *
     * @return DeedPay_Logger
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Check if debug mode is enabled in gateway settings
        $gateway_settings = get_option('deedpay_settings', array());
        $this->enabled = isset($gateway_settings['debug_mode']) && $gateway_settings['debug_mode'] === 'yes';
    }

    /**
     * Log a message
     *
     * @param string $level   Log level (debug, info, warning, error)
     * @param string $message Log message
     * @param array  $context Additional context data
     */
    public function log($level, $message, $context = array()) {
        // Always log errors, only log other levels if debug mode enabled
        if (!$this->enabled && $level !== 'error') {
            return;
        }

        // Ensure WooCommerce logger is available
        if (!function_exists('wc_get_logger')) {
            return;
        }

        $logger = wc_get_logger();

        // Format message
        if (is_string($message)) {
            $log_message = $message;
        } else {
            // Use wc_print_r if available, otherwise use JSON encoding
            if (function_exists('wc_print_r')) {
                $log_message = wc_print_r($message, true);
            } else {
                // Use wp_json_encode for safe array/object serialization
                $log_message = wp_json_encode($message, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
                if (false === $log_message) {
                    // Fallback if JSON encoding fails
                    $log_message = '(Unable to serialize message)';
                }
            }
        }

        // Add context if provided
        if (!empty($context)) {
            // Mask sensitive data in context
            $safe_context = $this->mask_sensitive_data($context);
            $log_message .= ' | Context: ' . wp_json_encode($safe_context);
        }

        // Log to WooCommerce logger
        $logger->log($level, $log_message, array('source' => 'deedpay'));
    }

    /**
     * Log info message
     *
     * @param string $message Log message
     * @param array  $context Additional context data
     */
    public function info($message, $context = array()) {
        $this->log('info', $message, $context);
    }

    /**
     * Log error message
     *
     * @param string $message Log message
     * @param array  $context Additional context data
     */
    public function error($message, $context = array()) {
        $this->log('error', $message, $context);
    }

    /**
     * Log warning message
     *
     * @param string $message Log message
     * @param array  $context Additional context data
     */
    public function warning($message, $context = array()) {
        $this->log('warning', $message, $context);
    }

    /**
     * Log debug message
     *
     * @param string $message Log message
     * @param array  $context Additional context data
     */
    public function debug($message, $context = array()) {
        $this->log('debug', $message, $context);
    }

    /**
     * Mask sensitive data in context
     *
     * @param array $context Context data
     * @return array Masked context
     */
    private function mask_sensitive_data($context) {
        $sensitive_keys = array('api_key', 'api_secret', 'secret_key', 'webhook_secret', 'password', 'token', 'authorization');

        foreach ($context as $key => $value) {
            if (!is_string($key) || !is_string($value)) {
                continue;
            }

            foreach ($sensitive_keys as $sensitive_key) {
                if (stripos($key, $sensitive_key) !== false) {
                    $context[$key] = self::mask_api_key($value);
                    break;
                }
            }
        }

        return $context;
    }

    /**
     * Mask API key (show first 4 and last 4 characters)
     *
     * @param string $key API key
     * @return string Masked key
     */
    public static function mask_api_key($key) {
        if (empty($key) || !is_string($key) || strlen($key) < 12) {
            return '****';
        }
        return substr($key, 0, 4) . '****' . substr($key, -4);
    }
}
