<?php
/**
 * DEED Pay API Client
 *
 * Handles all communication with DEED Pay API.
 * Implements retry logic, error handling, and idempotency.
 *
 * @package DEEDPay
 * @since 0.1.1
 */

if (!defined('ABSPATH')) {
    exit;
}

class DeedPay_Api_Client {
    /**
     * API base URL
     *
     * @var string
     */
    private $base_url = 'https://api.usedeed.com';

    /**
     * API secret key
     *
     * @var string
     */
    private $api_secret_key;

    /**
     * Test mode
     *
     * @var bool
     */
    private $test_mode;

    /**
     * Max retries
     *
     * @var int
     */
    private $max_retries = 3;

    /**
     * Retry delays (seconds)
     *
     * @var array
     */
    private $retry_delays = array(1, 2, 4);

    /**
     * Request timeout (seconds)
     *
     * @var int
     */
    private $timeout = 30;

    /**
     * Connection timeout (seconds)
     *
     * @var int
     */
    private $connection_timeout = 10;

    /**
     * Logger instance
     *
     * @var DeedPay_Logger
     */
    private $logger;

    /**
     * Constructor
     *
     * @param string $api_secret_key API secret key
     * @param bool   $test_mode      Test mode enabled
     * @param string $base_url       API base URL (optional, defaults to production)
     */
    public function __construct($api_secret_key, $test_mode = false, $base_url = null) {
        $this->api_secret_key = $api_secret_key;
        $this->test_mode = $test_mode;
        $this->logger = DeedPay_Logger::get_instance();
        
        // Set base URL if provided, otherwise use default
        if ($base_url !== null && !empty($base_url)) {
            // Sanitize URL
            $base_url = sanitize_text_field($base_url);
            
            // Add protocol if missing (assume https)
            if (!preg_match('#^https?://#i', $base_url)) {
                $base_url = 'https://' . $base_url;
            }
            
            // Ensure it doesn't end with a slash
            $this->base_url = rtrim($base_url, '/');
        }
    }

    /**
     * Create payment intent
     *
     * @param array $payment_data Payment data
     * @return array API response
     * @throws Exception On error
     */
    public function create_payment_intent($payment_data) {
        $endpoint = '/deed-pay/payment/create';

        // Add idempotency key to headers if provided
        $headers = array();
        if (isset($payment_data['idempotencyKey'])) {
            $headers['Idempotency-Key'] = $payment_data['idempotencyKey'];
            // Remove from body (it's in headers)
            $request_data = $payment_data;
            unset($request_data['idempotencyKey']);
        } else {
            $request_data = $payment_data;
        }

        return $this->make_request('POST', $endpoint, $request_data, $headers);
    }

    /**
     * Get payment intent details by reference code
     *
     * @param string $reference_code Payment reference code
     * @return array API response
     * @throws Exception On error
     */
    public function get_payment_intent($reference_code) {
        $endpoint = '/deed-pay/payment/' . rawurlencode($reference_code);
        return $this->make_request('GET', $endpoint);
    }

    /**
     * Get payment intent details by order ID
     *
     * @param string $order_id Business order ID
     * @return array API response
     * @throws Exception On error
     */
    public function get_payment_by_order_id($order_id) {
        $endpoint = '/deed-pay/payment/by-order/' . rawurlencode($order_id);
        return $this->make_request('GET', $endpoint);
    }

    /**
     * Refund payment
     *
     * @param string $reference_code Payment reference code
     * @param array  $refund_data    Refund data
     * @return array API response
     * @throws Exception On error
     */
    public function refund_payment($reference_code, $refund_data) {
        $endpoint = '/deed-pay/payment/' . rawurlencode($reference_code) . '/refund';
        return $this->make_request('POST', $endpoint, $refund_data);
    }

    /**
     * Verify webhook signature
     *
     * @param string $payload        Raw webhook payload
     * @param string $signature      Webhook signature
     * @param string $timestamp      Webhook timestamp
     * @param string $webhook_secret Webhook secret
     * @return bool Signature valid
     */
    public function verify_webhook_signature($payload, $signature, $timestamp, $webhook_secret) {
        if (empty($payload) || empty($signature) || empty($timestamp) || empty($webhook_secret)) {
            return false;
        }

        // Remove 'sha256=' prefix if present
        $signature = str_replace('sha256=', '', $signature);

        // Construct signed payload: timestamp + "." + raw_body
        // Note: $payload should be the raw JSON string from php://input (not parsed object)
        // This matches DEED's server-side signature computation: timestamp + "." + JSON.stringify(payload)
        $signed_payload = $timestamp . '.' . $payload;

        // Compute expected signature using HMAC-SHA256
        $expected_signature = hash_hmac('sha256', $signed_payload, $webhook_secret);

        // Constant-time comparison to prevent timing attacks
        return hash_equals($expected_signature, $signature);
    }

    /**
     * Make API request with retry logic
     *
     * @param string $method       HTTP method
     * @param string $endpoint     API endpoint
     * @param array  $data         Request data
     * @param array  $extra_headers Extra headers
     * @return array API response
     * @throws Exception On error
     */
    private function make_request($method, $endpoint, $data = array(), $extra_headers = array()) {
        $url = $this->base_url . $endpoint;

        // Build headers
        $headers = array(
            'Authorization' => 'Bearer ' . $this->api_secret_key,
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json',
        );

        // Merge extra headers
        $headers = array_merge($headers, $extra_headers);

        // Build request args
        $args = array(
            'method'      => $method,
            'headers'     => $headers,
            'timeout'     => $this->timeout,
            'sslverify'   => true,
            'httpversion' => '1.1',
        );

        // Add body for POST/PATCH/PUT
        if (in_array($method, array('POST', 'PATCH', 'PUT'), true)) {
            $args['body'] = wp_json_encode($data);
        }

        // Add query string for GET requests
        if ($method === 'GET' && !empty($data)) {
            $url = add_query_arg($data, $url);
        }

        // Log request (mask sensitive data)
        $this->logger->debug('API Request', array(
            'method'  => $method,
            'url'     => $url,
            'api_key' => DeedPay_Logger::mask_api_key($this->api_secret_key),
        ));

        // Retry loop
        $last_error = null;
        $last_response = null;

        for ($attempt = 0; $attempt <= $this->max_retries; $attempt++) {
            try {
                // Make request
                $response = wp_remote_request($url, $args);

                // Check for WP error (network issues, DNS failures, etc.)
                if (is_wp_error($response)) {
                    $error_message = $response->get_error_message();
                    $error_code = $response->get_error_code();

                    $this->logger->warning('API Request WP Error', array(
                        'attempt'      => $attempt + 1,
                        'error_code'   => $error_code,
                        'error_message' => $error_message,
                    ));

                    // Retry on network errors
                    $last_error = new Exception('Network error: ' . $error_message);
                    if ($attempt < $this->max_retries) {
                        sleep($this->retry_delays[$attempt]);
                        continue;
                    }
                    throw $last_error;
                }

                // Get status code
                $status_code = wp_remote_retrieve_response_code($response);
                $body = wp_remote_retrieve_body($response);

                // Log response
                $this->logger->debug('API Response', array(
                    'status_code' => $status_code,
                    'body'        => $body,
                ));

                // Success (2xx)
                if ($status_code >= 200 && $status_code < 300) {
                    $response_data = json_decode($body, true);

                    if (json_last_error() !== JSON_ERROR_NONE) {
                        throw new Exception('Invalid JSON response: ' . json_last_error_msg());
                    }

                    return $response_data;
                }

                // Client error (4xx except 429) - don't retry
                if ($status_code >= 400 && $status_code < 500 && $status_code !== 429) {
                    $this->handle_error_response($response, $url);
                }

                // Server error (5xx) or rate limit (429) - retry
                $this->logger->warning('API Request Retry', array(
                    'attempt'     => $attempt + 1,
                    'status_code' => $status_code,
                ));

                $last_response = $response;
                $last_error = new Exception('HTTP ' . $status_code . ': ' . $body);

            } catch (Exception $e) {
                $last_error = $e;
                $this->logger->error('API Request Exception', array(
                    'attempt'      => $attempt + 1,
                    'error'        => $e->getMessage(),
                    'status_code'  => isset($status_code) ? $status_code : null,
                ));
            }

            // Retry with exponential backoff (except on last attempt)
            if ($attempt < $this->max_retries) {
                sleep($this->retry_delays[$attempt]);
            }
        }

        // All retries failed
        if ($last_response) {
            $this->handle_error_response($last_response, $url);
        }

        if ($last_error) {
            throw $last_error;
        }

        throw new Exception('Unknown error occurred');
    }

    /**
     * Handle error response
     *
     * @param array  $response WP HTTP response
     * @param string $url      Request URL
     * @throws Exception
     */
    private function handle_error_response($response, $url) {
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $response_data = json_decode($body, true);

        // Parse error structure
        $error_code = 'UNKNOWN_ERROR';
        $error_message = 'Unknown error occurred';

        if (is_array($response_data)) {
            if (isset($response_data['error']['code'])) {
                $error_code = $response_data['error']['code'];
            }
            if (isset($response_data['error']['message'])) {
                $error_message = $response_data['error']['message'];
            } elseif (isset($response_data['message'])) {
                $error_message = $response_data['message'];
            }
        } elseif (is_string($body) && !empty($body)) {
            $error_message = $body;
        }

        // Log error
        $this->logger->error(
            sprintf('API Error: %s - %s', esc_html($error_code), esc_html($error_message)),
            array(
                'url'         => $url,
                'status_code' => $status_code,
                'error_code'  => $error_code,
            )
        );

        // Throw exception with appropriate message (sanitized for safety)
        throw new Exception(esc_html($error_message), absint($status_code));
    }
}
