<?php
/**
 * Plugin Name: DEED Pay for WooCommerce
 * Plugin URI: https://usedeed.com/woocommerce
 * Description: Accept payments with escrow protection via DEED Pay. Funds are held securely until delivery confirmation.
 * Version: 0.1.1
 * Author: DEED
 * Author URI: https://about.usedeed.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: deed-pay-woo
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * WC requires at least: 7.0
 * WC tested up to: 10.4.3
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('DEEDPAY_VERSION', '0.1.1');
define('DEEDPAY_PLUGIN_FILE', __FILE__);
define('DEEDPAY_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('DEEDPAY_PLUGIN_URL', plugin_dir_url(__FILE__));
define('DEEDPAY_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Check if WooCommerce is active
 *
 * @return bool
 */
function deedpay_is_woocommerce_active() {
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Using core filter name.
    return in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins'))) ||
           (is_multisite() && array_key_exists('woocommerce/woocommerce.php', get_site_option('active_sitewide_plugins', array())));
}

/**
 * Check WordPress version
 *
 * @return bool
 */
function deedpay_check_wordpress_version() {
    if (version_compare(get_bloginfo('version'), '6.0', '<')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p><strong>' . esc_html__('DEED Pay requires WordPress 6.0 or higher. Please update WordPress.', 'deed-pay-woo') . '</strong></p></div>';
        });
        return false;
    }
    return true;
}

/**
 * Check WooCommerce version
 *
 * @return bool
 */
function deedpay_check_woocommerce_version() {
    if (defined('WC_VERSION') && version_compare(WC_VERSION, '7.0', '<')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p><strong>' . esc_html__('DEED Pay requires WooCommerce 7.0 or higher. Please update WooCommerce.', 'deed-pay-woo') . '</strong></p></div>';
        });
        return false;
    }
    return true;
}

/**
 * Check PHP version
 *
 * @return bool
 */
function deedpay_check_php_version() {
    if (version_compare(PHP_VERSION, '7.4', '<')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p><strong>' . esc_html__('DEED Pay requires PHP 7.4 or higher. Please update PHP.', 'deed-pay-woo') . '</strong></p></div>';
        });
        return false;
    }
    return true;
}

/**
 * Initialize the gateway
 */
function deedpay_init_gateway() {
    // Version checks
    if (!deedpay_check_php_version() || !deedpay_check_wordpress_version()) {
        return;
    }

    if (!deedpay_is_woocommerce_active()) {
        add_action('admin_notices', 'deedpay_woocommerce_missing_notice');
        return;
    }

    if (!deedpay_check_woocommerce_version()) {
        return;
    }

    // Load plugin classes
    require_once DEEDPAY_PLUGIN_DIR . 'includes/class-deedpay-logger.php';
    require_once DEEDPAY_PLUGIN_DIR . 'includes/class-deedpay-api-client.php';
    require_once DEEDPAY_PLUGIN_DIR . 'includes/class-deedpay-webhook.php';
    require_once DEEDPAY_PLUGIN_DIR . 'includes/class-wc-gateway-deedpay.php';

    // Add gateway to WooCommerce
    add_filter('woocommerce_payment_gateways', 'deedpay_add_gateway');

    // Enqueue admin styles
    add_action('admin_enqueue_scripts', 'deedpay_enqueue_admin_styles');
}
add_action('plugins_loaded', 'deedpay_init_gateway', 11);

/**
 * Declare WooCommerce HPOS compatibility
 */
function deedpay_declare_hpos_compatibility() {
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
            'custom_order_tables',
            DEEDPAY_PLUGIN_FILE,
            true
        );
    }
}
add_action('before_woocommerce_init', 'deedpay_declare_hpos_compatibility');

/**
 * Add DEED Pay gateway to WooCommerce
 *
 * @param array $gateways Existing gateways
 * @return array
 */
function deedpay_add_gateway($gateways) {
    $gateways[] = 'WC_Gateway_DeedPay';
    return $gateways;
}

/**
 * Add settings link on plugin page
 *
 * @param array $links Existing links
 * @return array
 */
function deedpay_add_settings_link($links) {
    $settings_link = '<a href="' . esc_url(admin_url('admin.php?page=wc-settings&tab=checkout&section=deedpay')) . '">' . esc_html__('Settings', 'deed-pay-woo') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . DEEDPAY_PLUGIN_BASENAME, 'deedpay_add_settings_link');


/**
 * Enqueue admin styles
 */
function deedpay_enqueue_admin_styles() {
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only section check for enqueue; no form submission.
    if (is_admin() && isset($_GET['section']) && sanitize_text_field(wp_unslash($_GET['section'])) === 'deedpay') {
        wp_enqueue_style(
            'deedpay-admin',
            DEEDPAY_PLUGIN_URL . 'assets/css/deedpay-admin.css',
            array(),
            DEEDPAY_VERSION
        );
    }
}

/**
 * Enqueue checkout scripts
 */
function deedpay_enqueue_checkout_scripts() {
    if (is_checkout()) {
        wp_enqueue_script(
            'deedpay-checkout',
            DEEDPAY_PLUGIN_URL . 'assets/js/deedpay-checkout.js',
            array('jquery'),
            DEEDPAY_VERSION,
            true
        );
    }
}
add_action('wp_enqueue_scripts', 'deedpay_enqueue_checkout_scripts');

/**
 * Admin notice if WooCommerce is not active
 */
function deedpay_woocommerce_missing_notice() {
    if (!deedpay_is_woocommerce_active()) {
        echo '<div class="error"><p><strong>' . esc_html__('DEED Pay requires WooCommerce to be installed and active.', 'deed-pay-woo') . '</strong> <a href="' . esc_url(admin_url('plugin-install.php?s=woocommerce&tab=search&type=term')) . '">' . esc_html__('Install WooCommerce', 'deed-pay-woo') . '</a></p></div>';
    }
}

/**
 * Plugin activation
 */
register_activation_hook(__FILE__, 'deedpay_activate');
function deedpay_activate() {
    global $wpdb;

    // Only set default options if they haven't been set before
    $existing_settings = get_option('deedpay_settings');

    if (false === $existing_settings) {
        $default_settings = array(
            'enabled'           => 'no',
            'title'             => __('DEED Pay (Escrow)', 'deed-pay-woo'),
            'description'       => __('Secure escrow payment. Funds held safely until delivery.', 'deed-pay-woo'),
            'test_mode'         => 'yes',
            'api_secret_key'    => '',
            'api_public_key'    => '',
            'webhook_secret'    => '',
            'auto_release_days' => 7,
            'product_type'      => 'PHYSICAL',
            'debug_mode'        => 'no',
        );
        add_option('deedpay_settings', $default_settings);
    }

    // Add database index for better query performance (intentional one-time schema change on activation).
    $wpdb->query( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
        "CREATE INDEX IF NOT EXISTS deedpay_reference_code_idx ON {$wpdb->postmeta} (meta_key(191), meta_value(191))"
    );

    // Flush rewrite rules
    flush_rewrite_rules();
}

/**
 * Plugin deactivation
 */
register_deactivation_hook(__FILE__, 'deedpay_deactivate');
function deedpay_deactivate() {
    global $wpdb;

    // Clear scheduled hooks
    wp_clear_scheduled_hook('deedpay_daily_cleanup');

    // Remove database index on deactivation (intentional schema change).
    $wpdb->query( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
        "DROP INDEX IF EXISTS deedpay_reference_code_idx ON {$wpdb->postmeta}"
    );

    // Flush rewrite rules
    flush_rewrite_rules();
}