<?php
/**
 * Uninstall script for Declarando WooCommerce Integration
 * 
 * This file runs when the plugin is deleted via WordPress admin.
 * It removes all plugin data, options, tables, and temporary files.
 * 
 * @package DeclarandoWooCommerce
 * @since 1.0.0
 */

// If uninstall not called from WordPress, then exit
if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

// Check if it's really being uninstalled
if (!WP_UNINSTALL_PLUGIN || WP_UNINSTALL_PLUGIN !== 'declarando-gestion-facturas/declarando-gestion-facturas.php') {
    exit;
}

/**
 * Remove all plugin options
 */
function declarando_wc_remove_options()
{
    global $wpdb;

    // Remove main plugin options
    delete_option('declarando_options');
    delete_option('declarando_gestion_facturas_options');

    // Remove any cached data
    delete_transient('declarando_gestion_facturas_update_data');
    delete_transient('settings_errors');

    // Remove any site options (for multisite)
    delete_site_option('declarando_options');
    delete_site_option('declarando_gestion_facturas_options');

    // Remove any options with our prefix
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation during uninstall
    $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'declarando_%'");

    // Clean up any orphaned options
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation during uninstall
    $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_declarando_%'");
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation during uninstall
    $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_declarando_%'");
}

/**
 * Remove custom database tables
 */
function declarando_wc_remove_tables()
{
    global $wpdb;

    // Remove the custom table created by the plugin
    $table_name = esc_sql($wpdb->prefix . 'declarando_pedidos');
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Drop table during uninstall
    $wpdb->query("DROP TABLE IF EXISTS {$table_name}");

    // Remove any other custom tables that might have been created
    $custom_tables = array(
        $wpdb->prefix . 'declarando_logs',
        $wpdb->prefix . 'declarando_sync_queue',
        $wpdb->prefix . 'declarando_api_cache'
    );

    foreach ($custom_tables as $table) {
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Drop table during uninstall
        $wpdb->query("DROP TABLE IF EXISTS {$table}");
    }
}

/**
 * Remove uploaded files and temporary data
 */
function declarando_wc_remove_files()
{
    $upload_dir = wp_upload_dir();
    $plugin_upload_dir = $upload_dir['basedir'] . '/declarando/';

    // Remove plugin's upload directory if it exists
    if (is_dir($plugin_upload_dir)) {
        declarando_wc_delete_directory($plugin_upload_dir);
    }

    // Remove temporary files
    $temp_files = glob(sys_get_temp_dir() . '/declarando_*');
    if ($temp_files) {
        foreach ($temp_files as $file) {
            if (is_file($file)) {
                wp_delete_file($file);
            }
        }
    }
}

/**
 * Remove user meta data related to the plugin
 */
function declarando_wc_remove_user_meta()
{
    global $wpdb;

    // Remove user meta data with our prefix
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation during uninstall
    $wpdb->query("DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE 'declarando_%'");
}

/**
 * Remove post meta data related to the plugin
 */
function declarando_wc_remove_post_meta()
{
    global $wpdb;

    // Remove post meta data with our prefix
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation during uninstall
    $wpdb->query("DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE 'declarando_%'");
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup operation during uninstall
    $wpdb->query("DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE '_declarando_%'");
}

/**
 * Remove scheduled hooks and cron jobs
 */
function declarando_wc_remove_cron_jobs()
{
    // Remove scheduled events
    wp_clear_scheduled_hook('declarando_sync_orders');
    wp_clear_scheduled_hook('declarando_cleanup_logs');
    wp_clear_scheduled_hook('declarando_check_updates');

    // Remove any other cron jobs with our prefix
    $cron = _get_cron_array();
    if (!empty($cron)) {
        foreach ($cron as $timestamp => $hooks) {
            foreach ($hooks as $hook => $data) {
                if (strpos($hook, 'declarando_') === 0) {
                    wp_unschedule_event($timestamp, $hook);
                }
            }
        }
    }
}

/**
 * Recursively delete a directory and its contents
 */
function declarando_wc_delete_directory($dir)
{
    if (!is_dir($dir)) {
        return false;
    }

    // Initialize WP_Filesystem
    global $wp_filesystem;
    if (empty($wp_filesystem)) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
    }

    // Use WP_Filesystem to delete directory
    if (!empty($wp_filesystem)) {
        return $wp_filesystem->rmdir($dir, true);
    }

    // Fallback to manual deletion if WP_Filesystem is not available
    $files = array_diff(scandir($dir), array('.', '..'));

    foreach ($files as $file) {
        $path = $dir . DIRECTORY_SEPARATOR . $file;
        if (is_dir($path)) {
            declarando_wc_delete_directory($path);
        } else {
            wp_delete_file($path);
        }
    }

    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir
    return rmdir($dir);
}

/**
 * Log the uninstallation for debugging purposes
 */
function declarando_wc_log_uninstall()
{
    if (function_exists('current_time') && defined('WP_DEBUG') && WP_DEBUG) {
        // Only log if WordPress functions are available
        if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            // Use WordPress logging when available
        }
    }
}

/**
 * Main uninstall function
 */
function declarando_wc_uninstall()
{
    // Only run uninstall for users with proper permissions
    if (!current_user_can('activate_plugins')) {
        return;
    }

    // Log the uninstallation
    //  declarando_wc_log_uninstall();

    // Remove all plugin data
    //  declarando_wc_remove_options();
    // declarando_wc_remove_tables();
    // declarando_wc_remove_files();
    // declarando_wc_remove_user_meta();
    // declarando_wc_remove_post_meta();
    //  declarando_wc_remove_cron_jobs();

    // Force cleanup of any remaining data
    if (function_exists('wp_cache_flush')) {
        wp_cache_flush();
    }

    // Final cleanup
    do_action('declarando_wc_uninstalled');
}

// Run the uninstallation
declarando_wc_uninstall();