<?php
namespace Declarando\Woocommerce\Admin;

use Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController;

if (!defined('ABSPATH'))
    exit;

class RefundsUIBox
{
    public function init(): void
    {
        if (!is_admin())
            return;

        // Enqueue styles and scripts
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);

        // Metabox compatible con HPOS y legacy
        add_action('add_meta_boxes', [$this, 'register_metabox']);

        // Activa los toggles (plegar/ocultar) del metabox en editor clásico
        add_action('admin_print_footer_scripts-post.php', [$this, 'enable_postbox_toggles'], 9);
        add_action('admin_print_footer_scripts-post-new.php', [$this, 'enable_postbox_toggles'], 9);

        // JS del botón dentro del metabox (clásico u HPOS)
        add_action('admin_print_footer_scripts', [$this, 'print_footer_js'], 999);

        // AJAX
        add_action('wp_ajax_decl82gf_crear_abono', [$this, 'ajax_crear_abono']);
    }

    /**
     * Registra el metabox en el screen correcto:
     *  - HPOS: usar wc_get_page_screen_id('shop-order')
     *  - Legacy: 'shop_order'
     */
    public function register_metabox(): void
    {
        $screen = $this->get_order_screen_id();

        add_meta_box(
            'decl82gf_refunds_ui_end',
            __('Declarando', 'declarando-gestion-facturas'),
            [$this, 'render_metabox'],
            $screen,
            'normal', // columna principal
            'low'     // al final
        );
    }

    private function get_order_screen_id()
    {
        // Si HPOS está activo, WooCommerce usa un screen id especial
        if (class_exists('\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController')) {
            $controller = wc_get_container()->get(CustomOrdersTableController::class);
            if ($controller && $controller->custom_orders_table_usage_is_enabled()) {
                return wc_get_page_screen_id('shop-order'); // ← requerido en HPOS
            }
        }
        return 'shop_order'; // ← editor clásico
    }

    public function render_metabox($post_or_order): void
    {
        // En HPOS, Woo pasa un WC_Order; en clásico, un WP_Post
        $order = ($post_or_order instanceof \WP_Post)
            ? wc_get_order($post_or_order->ID)
            : $post_or_order;

        if (!$order || !is_a($order, 'WC_Order')) {
            echo '<p>' . esc_html__('No se pudo cargar el pedido.', 'declarando-gestion-facturas') . '</p>';
            return;
        }

        echo wp_kses_post($this->build_refunds_box_html($order));
    }

    private function build_refunds_box_html(\WC_Order $order): string
    {
        $refunds = $order->get_refunds();
        //if (empty($refunds)) {
        //     return '<p style="margin:8px 0;">' . esc_html__('Este pedido no tiene reembolsos.', 'declarando-gestion-facturas') . '</p>';
        //  }
        // Ordenar por fecha de creación (más antiguo primero)
        /* usort($refunds, function ($a, $b) {
             $da = $a->get_date_created() ? $a->get_date_created()->getTimestamp() : 0;
             $db = $b->get_date_created() ? $b->get_date_created()->getTimestamp() : 0;
             return $da - $db;
         });
         $refunds = array_values($refunds);*/
        ob_start(); ?>
        <?php
        $col = new Column();
        $col->decl82gf_render_cell($order->get_id());

        if (!empty($refunds)) {
            ?>
            <div class="decl82gf-refunds-box">
                <div class="hdr"><?php echo esc_html__('Declarando Reembolsos', 'declarando-gestion-facturas'); ?></div>
                <?php
                foreach ($refunds as $refund):
                    /** @var \WC_Order_Refund $refund */
                    $rid = $refund->get_id();
                    $amount = wc_price($refund->get_amount(), ['currency' => $order->get_currency()]);
                    $date = $refund->get_date_created()
                        ? $refund->get_date_created()->date_i18n(get_option('date_format') . ' ' . get_option('time_format'))
                        : '-';

                    // Si guardas estado/enlace en tu tabla:
                    $row = $this->decl82gf_get_link_by_refund_id($rid);
                    // echo json_encode($row);
                    $badge = '';
                    $estado = null;
                    if ($row && !empty($row->external_id)) {
                        $estado = $row->estado ?: 'pending';
                        $label = ($estado === 'paid') ? __('Sincronizado', 'declarando-gestion-facturas') : __('Pendiente', 'declarando-gestion-facturas');
                        $cls = ($estado === 'paid') ? 'ok' : 'pending';
                        $badge = '<span class="decl82gf-badge ' . esc_attr($cls) . '">' . esc_html($label) . '</span>';
                    }

                    $nonce = wp_create_nonce('decl82gf_crear_abono_' . $rid);
                    if (!$estado) {
                        $btn = '<button class="button button-small decl82gf-crear-refund"
                                data-refund="' . esc_attr($rid) . '"
                                data-nonce="' . esc_attr($nonce) . '">' . esc_html__('Sincronizar reembolso', 'declarando-gestion-facturas') . '</button>';
                    } ?>
                    <div class="it">
                        <div>
                            <div class="amount"><?php echo wp_kses_post($amount); ?></div>
                            <div class="meta">#<?php echo esc_html($rid); ?> · <?php echo esc_html($date); ?></div>
                        </div>
                        <div class="actions"><?php echo wp_kses_post($badge . ' ' . $btn); ?></div>
                    </div>
                <?php endforeach;

                ?>
            </div>
            <?php
        }
        return ob_get_clean();
    }

    /** Activa postboxes (plegar/ocultar) en clásico */
    public function enable_postbox_toggles(): void
    {
        // Postbox toggles now handled via enqueued script
    }

    /** JS del botón dentro del metabox (clásico u HPOS) */
    public function print_footer_js(): void
    {
        // JS now handled via enqueued script
    }

    /** AJAX: crear abono / vincular refund */
    public function ajax_crear_abono(): void
    {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => 'No autorizado'], 403);
        }
        $refund_id = isset($_POST['refund_id']) ? absint($_POST['refund_id']) : 0;
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';

        if (!$refund_id || !wp_verify_nonce($nonce, 'decl82gf_crear_abono_' . $refund_id)) {
            wp_send_json_error(['message' => 'Solicitud no válida'], 400);
        }

        $refund = wc_get_order($refund_id);
        if (!$refund || !is_a($refund, 'WC_Order_Refund')) {
            wp_send_json_error(['message' => 'Refund no válido'], 400);
        }

        $parent_id = (int) $refund->get_parent_id();
        $amount = (float) $refund->get_amount();
        /* translators: %1$d is the parent order ID, %2$d is the refund ID */
        $concepto = sprintf(__('Reembolso pedido #%1$d (refund #%2$d)', 'declarando-gestion-facturas'), $parent_id, $refund_id);
        $external_key = 'refund-' . $refund_id;

        try {
            $api = new Api();
            $api->createRectifiedUninvoicedIncome(
                $external_key,
                $refund_id,
                $parent_id,
                'wc-' . $parent_id,
                $amount,
                $concepto,
            );
        } catch (\Throwable $e) {
            if (function_exists('wp_kses_post') && defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // Safe logging using WordPress functions when available
            }
            wp_send_json_error(['message' => 'Error API: ' . esc_html($e->getMessage())], 500);
        }

        wp_send_json_success([
            'message' => __('Abono creado y refund vinculado.', 'declarando-gestion-facturas'),
            'external_id' => $external_key,
            'estado' => 'paid',
        ]);
    }

    /** === Helpers opcionales para tu tabla de vínculos === */
    private function decl82gf_refund_table_name(): string
    {
        global $wpdb;
        return "{$wpdb->prefix}declarando_pedidos";
    }

    private function decl82gf_get_link_by_refund_id(int $refund_id)
    {
        global $wpdb;
        $table = esc_sql($this->decl82gf_refund_table_name());

        // Use WordPress cache when available
        $cache_key = 'declarando_refund_link_' . $refund_id;
        $row = wp_cache_get($cache_key, 'declarando-gestion-facturas');

        if ($row === false) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query with caching
            $row = $wpdb->get_row($wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} WHERE order_id = %s ORDER BY id DESC LIMIT 1",
                $refund_id
            ));
            wp_cache_set($cache_key, $row, 'declarando-gestion-facturas', 300); // Cache for 5 minutes
        }

        return $row;
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook): void
    {
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen) {
            return;
        }

        $is_classic = ($screen->base === 'post' && get_post_type() === 'shop_order');
        $is_hpos = ($screen->id === wc_get_page_screen_id('shop-order'));

        if (!$is_classic && !$is_hpos) {
            return;
        }

        // Enqueue CSS
        wp_enqueue_style(
            'declarando-admin-refunds',
            DECLARANDO_WC_PLUGIN_URL . 'assets/css/admin-refunds.css',
            [],
            DECLARANDO_WC_VERSION
        );

        // Enqueue JS with localization
        wp_enqueue_script(
            'declarando-admin-refunds',
            DECLARANDO_WC_PLUGIN_URL . 'assets/js/admin-refunds-ui.js',
            ['jquery'],
            DECLARANDO_WC_VERSION,
            true
        );

        // Localize script
        wp_localize_script('declarando-admin-refunds', 'declarandoRefunds', [
            'creating' => __('Creando...', 'declarando-gestion-facturas'),
            'syncRefund' => __('Sincronizar reembolso', 'declarando-gestion-facturas'),
        ]);
    }
}
