<?php

namespace Declarando\Woocommerce\Admin;

use Declarando_Plugin;

/**
 * ------------------------------
 *  Integración WooCommerce
 * ------------------------------
 *
 * Configuración (elige una):
 *  - Definir constantes en wp-config.php:
 *      define('DECLARANDO_API_HOST', 'https://declarando-api-test.declarando.es');
 *      define('DECLARANDO_API_TOKEN', 'eyJ...'); // JWT sin "Bearer "
 *  - O usar filtros:
 *      add_filter('declarando_api_host', function(){ return 'https://api.declarando.es'; });
 *      add_filter('declarando_api_token', function(){ return 'eyJ...'; });
 *
 *  - Opcional: activar logs
 *      define('DECLARANDO_WC_LOG', true);
 *
 * Estados que disparan envío de ingreso:
 *  - Por defecto: processing, completed
 *  - Filtrable con: declarando_wc_push_statuses
 */

class Declarando_WC_Integrator
{
    /** @var Api */
    protected $api;
    protected $push_statuses;
    protected $rembolso_statuses;

    public function __construct()
    {
        $this->api = new Api();

        // Ajustes opcionales por filtros
        if (function_exists('apply_filters')) {
            $timeout = apply_filters('declarando_api_timeout', 25);
            $retries = apply_filters('declarando_api_retries', 2);
            $ua = apply_filters('declarando_api_user_agent', 'DeclarandoAPI-Client/1.1 (+WooCommerce)');
            $this->api->setTimeout($timeout);
            $this->api->setRetries($retries);
            $this->api->setUserAgent($ua);
        }

        $allowed_statuses = Declarando_Plugin::get('allowed_statuses', ['wc-processing', 'wc-completed']);
        $allowed_statuses = array_map(function ($s) {
            return str_replace('wc-', '', $s);
        }, $allowed_statuses);
        // Estados que disparan envío
        $this->push_statuses = $allowed_statuses;
        if (function_exists('apply_filters')) {
            $this->push_statuses = (array) apply_filters('declarando_wc_push_statuses', $this->push_statuses);
        }
        $rembolso_statuses = Declarando_Plugin::get('rembolso_statuses', ['wc-cancelled', 'wc-refunded']);
        $rembolso_statuses = array_map(function ($s) {
            return str_replace('wc-', '', $s);
        }, $rembolso_statuses);
        // Estados que disparan envío
        $this->rembolso_statuses = $rembolso_statuses;
        if (function_exists('apply_filters')) {
            $this->rembolso_statuses = (array) apply_filters('declarando_wc_push_statuses', $this->rembolso_statuses);
        }

        // Hooks
        add_action('woocommerce_order_status_changed', array($this, 'maybe_push_uninvoiced_income'), 10, 4);
        add_action('woocommerce_order_status_changed', array($this, 'maybe_push_refund'), 10, 4);
        //  add_action('woocommerce_order_refunded', array($this, 'on_order_refunded'), 10, 2);

        // Robustez adicional (algunas instalaciones usan otro hook)
        //  if (has_action('woocommerce_refund_created') === false) {
        // En algunas versiones el hook es 'woocommerce_refund_created' con ($refund_id, $args)
        add_action('woocommerce_created_refund', array($this, 'on_created_refund'), 10, 2);
        add_action('woocommerce_refund_created', array($this, 'on_created_refund'), 10, 2);
        //  }
        // Evita que WooCommerce ponga el pedido en refunded automáticamente
        // SOLO si el estado anterior era "cancelled"
        /**
         * 1) Guardar en memoria (solo durante la petición) el estado anterior de cada pedido.
         *    No escribe metadatos, no persiste en DB.
         */
        add_action('woocommerce_order_status_changed', function ($order_id, $from, $to, $order) {
            // Usamos un contenedor global simple para esta petición
            if (!isset($GLOBALS['declarando_wc_prev_status'])) {
                $GLOBALS['declarando_wc_prev_status'] = [];
            }
            $GLOBALS['declarando_wc_prev_status'][$order_id] = $from;
        }, 1, 4);

        /**
         * 2) Evitar que WooCommerce cambie a "refunded" SOLO si venía de "cancelled".
         */
        add_filter('woocommerce_order_fully_refunded_status', function ($status, $order_id, $order) {
            // Asegurar objeto pedido por si llega un int/null
            if (!$order instanceof \WC_Order) {
                $order = wc_get_order($order_id);
                if (!$order) {
                    return $status; // no podemos decidir, seguimos por defecto
                }
            }

            // Recuperar el estado previo que guardamos en el hook anterior
            $prev = $GLOBALS['declarando_wc_prev_status'][$order_id] ?? null;

            // Si venía de "cancelled", forzamos que se quede en "cancelled"
            if ($prev === 'cancelled') {
                return 'cancelled';
            }

            // Comportamiento normal en cualquier otro caso
            return $status;
        }, 10, 3);
    }

    function maybe_push_refund($order_id, $old_status, $new_status, $order)
    {
        try {
            if (!in_array($old_status, (array) $this->push_statuses, true)) {
                return;
            }
            // 1) ¿Este cambio de estado nos interesa?
            if (!in_array($new_status, (array) $this->rembolso_statuses, true)) {
                return;
            }

            if (!is_object($order)) {
                $order = wc_get_order($order_id);
                if (!$order)
                    return;
            }

            // 1) Calcular delta pendiente a reembolsar
            $total_order = (float) wc_format_decimal($order->get_total(), 2);
            $total_refunded_wc = (float) wc_format_decimal($order->get_total_refunded(), 2);
            $delta = (float) wc_format_decimal($total_order - $total_refunded_wc, 2);

            if ($delta <= 0) {
                // Nada pendiente: ya está reembolsado total/igual al target
                _declarando_log("[refund] Sin delta (total={$total_order}, refunded={$total_refunded_wc}). No se crea refund.");
                return;
            }

            // 2) Crear el REFUND en WooCommerce (solo registro, sin pasarela)
            if (!function_exists('wc_create_refund')) {
                _declarando_log('[refund] wc_create_refund no disponible.');
                return;
            }

            $reason = 'Reembolso automático por cambio a estado: ' . $new_status;
            $refund_args = [
                'amount' => $delta,            // importe POSITIVO
                'reason' => $reason,
                'order_id' => $order->get_id(),
                'line_items' => [],                // vacío = reembolso por importe
                'refund_payment' => false,             // true si quieres lanzar también el reembolso en la pasarela
                'restock_items' => false,             // true si quieres reponer stock
            ];
            $refund_id = wc_create_refund($refund_args);

            if (is_wp_error($refund_id) || !$refund_id) {
                _declarando_log('[refund] Error creando refund: ' . (is_wp_error($refund_id) ? $refund_id->get_error_message() : 'desconocido'));
                return;
            }

            $refund = wc_get_order($refund_id);
            if (!$refund || !($refund instanceof \WC_Order_Refund)) {
                _declarando_log("[refund] No se pudo cargar el refund {$refund_id}");
                return;
            }

            $refund_amount = (float) wc_format_decimal($refund->get_amount(), 2); // positivo
            _declarando_log("[refund] Creado refund ID {$refund_id} por {$refund_amount}");

            // 3) (Opcional) Rectificar en Declarando, SOLO si existe el ingreso original
            $orderExternalId = $this->build_order_external_id($order);
            $ddd = new DeclarandoApiDbStorage();
            $original_uuid = $ddd->getOrderId(null, $order->get_id());

            if (!$original_uuid) {
                _declarando_log("[rectify] No existe ingreso original para {$orderExternalId}. Se omite rectificación API.");
                return;
            }

            // ExternalId único de rectificación basado en refund_id
            $rectExternalId = $orderExternalId . ':refund:' . $refund_id;
            // Idempotencia natural: si vuelven a disparar, como ya no habrá delta, no se creará otro refund.
            // Aun así, evitamos doble push a la API comprobando si el cliente ya guardó este externalId.
            $already_uuid = $ddd->getOrderId(null, $refund_id);
            if ($already_uuid) {
                _declarando_log("[rectify] Ya existe rectificación {$rectExternalId} => {$already_uuid}. No repetir.");
                return;
            }

            $concept = 'Rectificación (reembolso) Pedido #' . $this->get_order_number($order);

            // Atento al signo que espera tu API (muchas esperan POSITIVO)
            $amount_to_send = $refund_amount;

            $res = $this->api->createRectifiedUninvoicedIncome(
                $rectExternalId,
                $refund_id,
                $order_id,
                $original_uuid,
                $amount_to_send,
                $concept,
            );

            _declarando_log([
                'createRectifiedUninvoicedIncome' => $res,
                'refund_id' => $refund_id,
                'amount' => $amount_to_send,
                'original_uuid' => $original_uuid,
                'rectExternalId' => $rectExternalId
            ]);


        } catch (\Exception $e) {
            _declarando_log('[refund] Error push refund: ' . $e->getMessage());
        }
    }

    protected function getHost()
    {
        $host = defined('DECLARANDO_API_HOST') ? DECLARANDO_API_HOST : 'https://declarando-api-test.declarando.es';
        if (function_exists('apply_filters')) {
            $host = apply_filters('declarando_api_host', $host);
        }
        return rtrim($host, '/');
    }

    protected function getToken()
    {
        $token = defined('DECLARANDO_API_TOKEN') ? DECLARANDO_API_TOKEN : '';
        if (function_exists('apply_filters')) {
            $token = apply_filters('declarando_api_token', $token);
        }
        return $token;
    }

    protected function getStorePath()
    {
        $base = dirname(__FILE__);
        if (function_exists('wp_upload_dir')) {
            $up = wp_upload_dir();
            if (is_array($up) && !empty($up['basedir'])) {
                $base = $up['basedir'];
            }
        }
        $dir = rtrim($base, '/') . '/declarando';
        if (!is_dir($dir)) {
            if (function_exists('wp_mkdir_p')) {
                wp_mkdir_p($dir);
            } else {
                // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir
                mkdir($dir, 0755, true);
            }
        }
        return $dir . '/uuid_store.json';
    }

    /**
     * Dispara creación de ingreso sin factura si el estado nuevo está en push_statuses
     */
    public function maybe_push_uninvoiced_income($order_id, $old_status, $new_status, $order)
    {
        try {
            if (!in_array($new_status, $this->push_statuses)) {
                return;
            }

            if (!is_object($order)) {
                $order = wc_get_order($order_id);
                if (!$order)
                    return;
            }

            // Construir externalId estable del pedido
            $orderExternalId = $this->build_order_external_id($order);
            $concept = 'Pedido #' . $this->get_order_number($order);

            // Pago: si estado final es "processing|completed" => paid
            $paymentStatus = 'paid';

            // Total del pedido
            $total = (float) $order->get_total();

            // Evitar duplicados por idempotencia (el cliente también lo evita)
            $already = $this->api->getSavedUuidFor($orderExternalId);
            if ($already) {
                _declarando_log("Ingreso ya existente para {$orderExternalId}: {$already}");
                return;
            }

            $res = $this->api->createUninvoicedIncome($order_id, $orderExternalId, $concept, $total, $paymentStatus);
            _declarando_log(array('createUninvoicedIncome' => $res));
        } catch (\Exception $e) {
            _declarando_log('Error push income: ' . $e->getMessage());
        }
    }

    /**
     * Dispara creación de rectificación al crear un reembolso
     */
    public function on_order_refunded($order_id, $refund_id)
    {
        try {
            $order = wc_get_order($order_id);
            if (!$order)
                return;

            $refund = wc_get_order($refund_id);
            if (!$refund || !is_a($refund, 'WC_Order_Refund'))
                return;

            $amount = (float) $refund->get_amount();
            /*if ($amount <= 0) {
                // Algunas versiones devuelven negativo; normalizamos
                $amount = (float) abs($amount);
            }*/

            $refundExternalId = $this->api->buildRefundExternalId($order_id, $refund_id, null);
            $orderExternalId = $this->build_order_external_id($order);
            $concept = 'Reembolso pedido #' . $this->get_order_number($order);

            // Idempotencia por refundExternalId
            $already = $this->api->getSavedUuidFor($refundExternalId);
            if ($already) {
                _declarando_log("Rectificación ya existente para {$refundExternalId}: {$already}");
                return;
            }

            $res = $this->api->createRectifiedUninvoicedIncome(
                $refundExternalId,
                $refund_id,
                $order_id,
                $orderExternalId,
                $amount,
                $concept,
            );

            _declarando_log(array('createRectifiedUninvoicedIncome' => $res));
        } catch (\Exception $e) {
            _declarando_log('Error refund rectify: ' . $e->getMessage());
        }
    }

    /**
     * Hook alternativo por compatibilidad (algunas instalaciones)
     * @param int|string $refund_id
     * @param array|object $args
     */
    public function on_created_refund($refund_id, $args = null)
    {
        try {
            $refund = wc_get_order($refund_id);
            if (!$refund || !is_a($refund, 'WC_Order_Refund'))
                return;
            $order_id = method_exists($refund, 'get_parent_id') ? $refund->get_parent_id() : (int) $refund->get_order_id();
            if ($order_id) {
                $this->on_order_refunded($order_id, $refund_id);
            }
        } catch (\Exception $e) {
            _declarando_log('Error on_created_refund: ' . $e->getMessage());
        }
    }

    // ---- Helpers Woo ----

    protected function get_order_number($order)
    {
        if (is_object($order) && method_exists($order, 'get_order_number')) {
            $n = $order->get_order_number();
            if ($n)
                return $n;
        }
        if (is_object($order) && method_exists($order, 'get_id')) {
            return (string) $order->get_id();
        }
        return (string) $order; // fallback
    }

    protected function build_order_external_id($order)
    {
        // Debe ser el mismo valor que uses siempre para el pedido original
        // Ej.: prefijo 'wc-' + número de pedido visible
        $num = $this->get_order_number($order);
        $externalId = 'wc-' . $num;

        // Permite personalizarlo con filtro
        if (function_exists('apply_filters')) {
            $externalId = apply_filters('declarando_wc_order_external_id', $externalId, $order);
        }
        return $externalId;
    }
}

// Inicializa integración al cargar plugins
/*add_action('plugins_loaded', function () {
    // Solo si WooCommerce está activo
    if (class_exists('WooCommerce')) {
        new Declarando_WC_Integrator();
    }
}, 20);*/