<?php
namespace Declarando\Woocommerce\Admin;

use Declarando_Plugin;

class Column
{
    function init()
    {
        /**
         * Enqueue styles and scripts
         */
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);

        /**
         * 1) Añadir columna (modo clásico - listado WP)
         */
        add_filter('manage_edit-shop_order_columns', function ($columns) {
            // Inserta la columna antes de "total" si existe
            $new = [];
            foreach ($columns as $key => $label) {
                if ($key === 'order_total') {
                    $new['decl82gf_declarando'] = __('Declarando', 'declarando-gestion-facturas');
                }
                $new[$key] = $label;
            }
            if (!isset($new['decl82gf_declarando'])) {
                $new['decl82gf_declarando'] = __('Declarando', 'declarando-gestion-facturas');
            }
            return $new;
        }, 20);

        /**
         * Pintar contenido de columna (modo clásico)
         */
        add_action('manage_shop_order_posts_custom_column', function ($column, $post_id) {
            if ($column !== 'decl82gf_declarando')
                return;
            $this->decl82gf_render_cell($post_id);
        }, 10, 2);

        /**
         * 2) Añadir columna (HPOS – Orders screen nuevo)
         */
        add_filter('woocommerce_shop_order_list_table_columns', function ($columns) {
            // Inserta la columna hacia el final
            $columns['decl82gf_declarando'] = __('Declarando', 'declarando-gestion-facturas');
            return $columns;
        });

        /**
         * Pintar contenido de columna (HPOS)
         */
        add_action('woocommerce_shop_order_list_table_custom_column', function ($column, $order) {
            if ($column !== 'decl82gf_declarando')
                return;
            if (is_a($order, 'WC_Order')) {
                $this->decl82gf_render_cell($order->get_id());
            }
        }, 10, 2);

        /**
         * 3) AJAX: Crear factura / vincular
         */
        add_action('wp_ajax_decl82gf_crear_factura', function () {
            if (!current_user_can('manage_woocommerce')) {
                wp_send_json_error(['message' => 'No autorizado'], 403);
            }
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce is verified below before processing
            $order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : 0;
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce is verified on next line
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';

            if (!$order_id || !wp_verify_nonce($nonce, 'decl82gf_crear_factura_' . $order_id)) {
                wp_send_json_error(['message' => 'Solicitud no válida'], 400);
            }

            // Datos mínimos para insertar/actualizar
            $data = [
                'order_id' => $order_id,
                'concepto' => 'Pedido #' . $order_id,
                'monto' => (float) (wc_get_order($order_id) ? wc_get_order($order_id)->get_total() : 0.0),
                'estado' => 'pending',
                'config' => null,
                'updated_at' => current_time('mysql', 1),
            ];
            $api = new Api();
            $api->createUninvoicedIncome(
                $data['order_id'],
                'wc-' . $data['order_id'],
                $data['concepto'],
                $data['monto'],
                'paid'
            );

            wp_send_json_success([
                'message' => 'Factura creada y pedido vinculado.',
                'external_id' => $data['order_id'],
                'estado' => $data['estado']
            ]);
        });

        add_action('before_woocommerce_init', function () {
            if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
                \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
            }
        });
        $rub = new RefundsUIBox();
        $rub->init();
    }

    /**
     * Helpers
     */
    function decl82gf_table_name()
    {
        global $wpdb;
        return "{$wpdb->prefix}declarando_pedidos";
    }

    function decl82gf_get_link_by_order_id($order_id)
    {
        global $wpdb;
        $table = esc_sql($this->decl82gf_table_name());

        // Use WordPress cache when available
        $cache_key = 'declarando_order_link_' . $order_id;
        $row = wp_cache_get($cache_key, 'declarando-gestion-facturas');

        if ($row === false) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query with caching
            $row = $wpdb->get_row($wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM {$table} WHERE order_id = %d ORDER BY id DESC LIMIT 1",
                $order_id
            ));
            wp_cache_set($cache_key, $row, 'declarando-gestion-facturas', 300);
        }

        return $row;
    }

    function decl82gf_render_cell($order_id)
    {
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        // Solo mostrar en pedidos "processing" o "completed"
        $allowed_statuses = array_merge(
            Declarando_Plugin::get('allowed_statuses', ['wc-processing', 'wc-completed']),
            Declarando_Plugin::get('rembolso_statuses', ['wc-cancelled', 'wc-refunded']),
        );
        $allowed_statuses = array_map(function ($s) {
            return str_replace('wc-', '', $s);
        }, $allowed_statuses);
        if (!in_array(str_replace('wc-', '', $order->get_status()), $allowed_statuses, true)) {
            echo '';
            return;
        }
        $row = $this->decl82gf_get_link_by_order_id($order_id);
        $nonce = wp_create_nonce('decl82gf_crear_factura_' . $order_id);
        $btn = '<button class="button button-small decl82gf-crear" data-order="' . esc_attr($order_id) . '" data-nonce="' . esc_attr($nonce) . '">' . esc_html__('Sincronizar', 'declarando-gestion-facturas') . '</button>';

        echo '<div class="decl82gf-acciones">';

        if ($row && !empty($row->external_id)) {
            // Vinculado
            $estado = $row->estado ?: 'pending';
            $label = $estado === 'paid' ? __('Sincronizado', 'declarando-gestion-facturas') : __('Pendiente', 'declarando-gestion-facturas');
            $cls = $estado === 'paid' ? 'ok' : 'pending';

            echo '<span class="decl82gf-badge ' . esc_attr($cls) . '">' . esc_html($label) . '</span>';

        } elseif ($row && empty($row->external_id)) {
            // Existe registro pero aún sin ID externo (quizá en proceso)
            echo '<span class="decl82gf-badge pending">' . esc_html__('En proceso', 'declarando-gestion-facturas') . '</span>';
            echo wp_kses_post($btn);
        } else {
            // No vinculado aún
            echo wp_kses_post($btn);
        }

        echo '</div>';
    }

    /**
     * Enqueue admin assets
     */
    function enqueue_admin_assets($hook)
    {
        // Solo en listados de pedidos
        if ($hook !== 'edit.php' && $hook !== 'woocommerce_page_wc-orders') {
            return;
        }

        // Enqueue CSS
        wp_enqueue_style(
            'declarando-admin-column',
            DECLARANDO_WC_PLUGIN_URL . 'assets/css/admin-column.css',
            [],
            DECLARANDO_WC_VERSION
        );

        // Enqueue JS
        wp_enqueue_script(
            'declarando-admin-column',
            DECLARANDO_WC_PLUGIN_URL . 'assets/js/admin-column.js',
            ['jquery'],
            DECLARANDO_WC_VERSION,
            true
        );
    }

}
