<?php
/**
 * Declarando API – Cliente + Integración WooCommerce (PHP 5.x+)
 */

namespace Declarando\Woocommerce\Admin;

use Declarando_Plugin;
use Exception;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!function_exists(__NAMESPACE__ . '\\_declarando_log')) {
    function _declarando_log($msg)
    {
        // Desactiva por defecto. Habilita con filtro o constante.
        $enabled = false;
        if (function_exists('apply_filters')) {
            $enabled = (bool) apply_filters('declarando_wc_log_enabled', $enabled);
        }
        if (defined('DECLARANDO_WC_LOG') && DECLARANDO_WC_LOG) {
            $enabled = true;
        }
        if ($enabled) {
            if (is_array($msg) || is_object($msg))
                $msg = wp_json_encode($msg);
        }
    }
}

/**
 * ------------------------------
 *  Cliente API
 * ------------------------------
 */
class Api
{
    protected $host;                 // https://declarando-api-test.declarando.es
    protected $token;                // Bearer token
    protected $timeout;              // segundos
    protected $retries;              // reintentos (transitorios)
    protected $storage;              // DeclarandoApiFileStorage
    protected $userAgent;            // User-Agent

    // cURL header collector
    protected $lastHeaderLines = array();

    public function __construct()
    {
        //$host = Declarando_Plugin::get('sandbox_api_endpoint');
        $host = Declarando_Plugin::get('api_endpoint');
        $this->host = rtrim($host, '/');
        $this->token = Declarando_Plugin::get('token');
        $this->timeout = 25;
        $this->retries = 1; // total intentos: retries+1

        // Use wp_upload_dir() instead of hardcoded WP_CONTENT_DIR
        $upload_dir = wp_upload_dir();
        $storageFile = trailingslashit($upload_dir['basedir']) . 'declarando-gestion-facturas/';

        $this->storage = new DeclarandoApiDbStorage();
        $this->userAgent = 'DeclarandoAPI-Client/1.1 (+php5-compatible)';
    }

    public function setTimeout($seconds)
    {
        $this->timeout = (int) $seconds;
    }
    public function setRetries($n)
    {
        $this->retries = (int) $n;
    }
    public function setToken($token)
    {
        $this->token = $token;
    }
    public function setUserAgent($ua)
    {
        if (is_string($ua) && strlen($ua))
            $this->userAgent = $ua;
    }
    public function setHost($host)
    {
        $this->host = rtrim($host, '/');
    }

    // Fábricas rápidas
    public static function sandbox($token, $storageFile)
    {
        return new self(DECLARANDO_API_SANDBOX_HOST, $token, $storageFile);
    }
    public static function production($token, $storageFile)
    {
        return new self(DECLARANDO_API_HOST, $token, $storageFile);
    }

    // ---- JWT decode (sin verificación de firma)
    public function decodeJwtPayload($jwt)
    {
        if (!is_string($jwt) || strpos($jwt, '.') === false)
            return array();
        $parts = explode('.', $jwt);
        if (count($parts) < 2)
            return array();
        $payloadB64 = $parts[1];
        $payloadJson = $this->b64url_decode($payloadB64);
        $arr = json_decode($payloadJson, true);
        return is_array($arr) ? $arr : array();
    }

    protected function b64url_decode($data)
    {
        $rem = strlen($data) % 4;
        if ($rem > 0)
            $data .= str_repeat('=', 4 - $rem);
        $data = strtr($data, '-_', '+/');
        return base64_decode($data);
    }

    protected function getScopeKey()
    {
        $claims = $this->decodeJwtPayload($this->token);
        $ecommerceID = isset($claims['ecommerceID']) ? $claims['ecommerceID'] : (isset($claims['id']) ? $claims['id'] : null);
        $userID = isset($claims['userID']) ? $claims['userID'] : null;
        return ($ecommerceID ? $ecommerceID : 'ecom') . '|' . ($userID ? $userID : 'user');
    }

    // ---- request genérico
    public function request($method, $path, $bodyAssoc /* or null */ , $extraHeaders)
    {
        if (!$this->token) {
            throw new \Exception('Bearer token no configurado.');
        }

        $url = (strpos($path, 'http') === 0) ? $path : $this->host . '/' . ltrim($path, '/');

        $headers = array(
            // 'Authorization: Bearer ' . $this->token,
            'Accept: application/json',
            'X-API-Key: ' . $this->token,
            'Connection: keep-alive',
            'User-Agent: ' . $this->userAgent
        );
        if ($bodyAssoc !== null) {
            $json = json_encode($bodyAssoc, JSON_UNESCAPED_UNICODE);
            $headers[] = 'Content-Type: application/json';
            $headers[] = 'Content-Length: ' . strlen($json);
        } else {
            $json = null;
        }

        if (is_array($extraHeaders)) {
            foreach ($extraHeaders as $h) {
                $headers[] = $h;
            }
        }

        $maxAttempts = max(1, (int) $this->retries + 1);
        $lastEx = null;

        for ($attempt = 1; $attempt <= $maxAttempts; $attempt++) {
            try {
                list($status, $respHeaders, $respBody) = $this->httpExecute($method, $url, $headers, $json);
                $respJson = null;
                if (strlen($respBody)) {
                    $respJson = json_decode($respBody, true);
                }

                if ($status >= 200 && $status < 300) {
                    return array('status' => $status, 'headers' => $respHeaders, 'body' => $respBody, 'json' => $respJson);
                }

                // Errores conocidos
                if ($status == 400) {
                    $msg = 'Solicitud inválida (400).';
                    if (is_array($respJson)) {
                        $detail = $this->extractErrorMessage($respJson);
                        if ($detail)
                            $msg .= ' ' . $detail;
                    }
                    throw new DeclarandoApiException($msg, $status, $respBody, $respJson);
                } elseif ($status == 401) {
                    throw new DeclarandoApiException('No autorizado (401). Token inválido o ecommerce no conectado.', $status, $respBody, $respJson);
                } elseif ($status == 429) {
                    // Rate limit → reintento
                    if ($attempt < $maxAttempts) {
                        $this->sleepBackoff($attempt);
                        continue;
                    }
                    throw new DeclarandoApiException('Rate limited (429).', $status, $respBody, $respJson);
                } elseif ($status >= 500) {
                    if ($attempt < $maxAttempts) {
                        $this->sleepBackoff($attempt);
                        continue;
                    }
                    throw new DeclarandoApiException('Error interno del servidor (' . $status . ').', $status, $respBody, $respJson);
                } else {
                    throw new DeclarandoApiException('Error HTTP inesperado (' . $status . ').', $status, $respBody, $respJson);
                }
            } catch (DeclarandoApiException $ex) {
                throw $ex; // no transitorios
            } catch (\Exception $ex) {
                $lastEx = $ex;
                if ($attempt < $maxAttempts) {
                    $this->sleepBackoff($attempt);
                    continue;
                }
                throw $ex;
            }
        }

        if ($lastEx)
            throw $lastEx;
        throw new \Exception('Fallo desconocido en request().');
    }

    protected function sleepBackoff($attempt)
    {
        // Backoff lineal con pequeño jitter (compatible PHP 5)
        $base = min(1 * $attempt, 3);
        $jitterMs = wp_rand(0, 250); // 0–250ms
        if ($base > 0)
            sleep($base);
        if ($jitterMs > 0)
            usleep($jitterMs * 1000);
    }

    protected function httpExecute($method, $url, $headers, $body /* string|null */)
    {
        $respHeaders = [];
        $respBody = '';
        $status = 0;

        // Normaliza/asegura cabeceras seguras
        // - 'Expect:' evita el 100-continue
        // - 'Connection: close' evita conexiones persistentes problemáticas en h2
        // - No forces 'Content-Length': curl lo calcula
        $safeHeaders = [];
        // $hasAcceptEncoding = false;
        foreach ((array) $headers as $h) {
            if (stripos($h, 'accept-encoding:') === 0)
                $hasAcceptEncoding = true;
            if (stripos($h, 'connection:') === 0)
                continue;
            if (stripos($h, 'expect:') === 0)
                continue;
            $safeHeaders[] = $h;
        }
        // if (!$hasAcceptEncoding)
        //     $safeHeaders[] = 'Accept-Encoding: gzip';
        // $safeHeaders[] = 'Expect:';             // desactiva 100-continue
        //  $safeHeaders[] = 'Connection: close';   // cierra al terminar

        // Use WordPress HTTP API
        $wpHeaders = array();
        foreach ($safeHeaders as $header) {
            if (strpos($header, ':') !== false) {
                list($key, $value) = explode(':', $header, 2);
                $wpHeaders[trim($key)] = trim($value);
            }
        }

        $args = array(
            'method' => strtoupper($method),
            'timeout' => $this->timeout,
            'redirection' => 5,
            'httpversion' => '1.1',
            'headers' => $wpHeaders,
            'body' => $body,
            'sslverify' => true,
            'compress' => true,
        );

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            throw new \Exception('HTTP request error: ' . esc_html($response->get_error_message()));
        }

        $respBody = wp_remote_retrieve_body($response);
        $status = wp_remote_retrieve_response_code($response);
        $respHeaders = wp_remote_retrieve_headers($response);

        // Convert headers object to array format expected by parseHeaderLines
        $headerLines = array();
        if (is_object($respHeaders) && method_exists($respHeaders, 'getAll')) {
            foreach ($respHeaders->getAll() as $key => $values) {
                foreach ((array) $values as $value) {
                    $headerLines[] = $key . ': ' . $value;
                }
            }
        } elseif (is_array($respHeaders)) {
            foreach ($respHeaders as $key => $value) {
                if (is_array($value)) {
                    foreach ($value as $v) {
                        $headerLines[] = $key . ': ' . $v;
                    }
                } else {
                    $headerLines[] = $key . ': ' . $value;
                }
            }
        }
        $respHeaders = $this->parseHeaderLines($headerLines);

        return [$status, $respHeaders, $respBody];
    }


    public function collectHeaders($ch, $headerLine)
    {
        $len = strlen($headerLine);
        $this->lastHeaderLines[] = $headerLine;
        return $len;
    }

    protected function parseHeaderLines($lines)
    {
        $out = array();
        if (!is_array($lines))
            return $out;
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '' || stripos($line, 'HTTP/') === 0)
                continue;
            $pos = strpos($line, ':');
            if ($pos === false)
                continue;
            $name = strtolower(trim(substr($line, 0, $pos)));
            $value = trim(substr($line, $pos + 1));
            if (!isset($out[$name]))
                $out[$name] = array();
            $out[$name][] = $value;
        }
        return $out;
    }

    protected function extractErrorMessage($respJson)
    {
        if (isset($respJson['message']) && is_string($respJson['message']))
            return $respJson['message'];
        if (isset($respJson['error']) && is_string($respJson['error']))
            return $respJson['error'];
        if (isset($respJson['detail']) && is_string($respJson['detail']))
            return $respJson['detail'];
        if (isset($respJson['errors']) && is_array($respJson['errors'])) {
            $first = reset($respJson['errors']);
            if (is_string($first))
                return $first;
            if (is_array($first) && isset($first['message']))
                return $first['message'];
        }
        return null;
    }

    // ---------------- Endpoints ----------------

    // POST /v2/api/ecommerces/uninvoiced-incomes
    public function createUninvoicedIncome($order_id, $externalId, $concept, $totalAmount, $paymentStatus /* pending|paid */)
    {
        if ($externalId === null || $externalId === '')
            throw new \InvalidArgumentException('externalId es requerido.');
        if (!is_numeric($totalAmount))
            throw new \InvalidArgumentException('totalAmount debe ser numérico.');
        if ($paymentStatus !== 'pending' && $paymentStatus !== 'paid') {
            throw new \InvalidArgumentException('paymentStatus debe ser "pending" o "paid".');
        }

        $scopeKey = $this->getScopeKey();

        // Idempotencia: ¿ya existe?
        $existing = $this->storage->getOrderId($scopeKey, (string) $order_id);
        if ($existing) {
            return array('status' => 200, 'json' => array('uninvoicedIncomeUuid' => $existing), 'cached' => true);
        }

        $payload = array(
            'externalId' => (string) $externalId,
            'concept' => $concept === null ? null : (string) $concept,
            'totalAmount' => (float) $totalAmount,
            'paymentStatus' => (string) $paymentStatus
        );
        $res = null;
        try {
            $res = $this->request('POST', '/uninvoiced-incomes', $payload, array());
        } catch (DeclarandoApiException $th) {
            //throw $th;
            switch ($th->httpStatus) {
                case 400:
                    $resp = json_decode($th->responseBody, 1);
                    switch ($resp['code']) {
                        case 'DEK.UN_INVOICED_INCOME_WITH_ECOMMERCE_REFERENCE_ALREADY_EXISTS':
                            $pattern = '/user <(\d+)> with reference <([a-f0-9\-]+)\|([^>]+)>/i';

                            if (preg_match($pattern, $resp['message'], $matches)) {
                                $user_id = $matches[1]; // 57316
                                $reference = $matches[2]; // cd511bbf-86d6-428a-841c-ea4f73bfa654
                                $pt_code = $matches[3]; // PT0000001

                                if ($reference) {
                                    $this->storage->saveUuid(
                                        $scopeKey,
                                        $order_id,
                                        (string) $externalId,
                                        (string) $reference,
                                        $payload['concept'] ?? '',
                                        $payload['totalAmount'] ?? 0,
                                        $payload['paymentStatus'] ?? 'paid',
                                    );
                                    $res = [
                                        "status" => 200,
                                        "json" => [
                                            "uninvoicedIncomeUuid" => $reference
                                        ],
                                        "cached" => false
                                    ];
                                }
                            }
                            break;

                        default:
                            # code...
                            break;
                    }
                    break;

                default:
                    break;
            }
        }

        $uuid = null;
        if (is_array($res['json'])) {
            if (isset($res['json']['uninvoicedIncomeUuid'])) {
                $uuid = $res['json']['uninvoicedIncomeUuid'];
            } elseif (isset($res['json']['uuid'])) {
                $uuid = $res['json']['uuid'];
            }
        }
        if ($uuid) {
            $this->storage->saveUuid(
                $scopeKey,
                $order_id,
                (string) $externalId,
                (string) $uuid,
                $payload['concept'] ?? '',
                $payload['totalAmount'] ?? 0,
                $payload['paymentStatus'] ?? 'paid',
            );
        }
        return $res;
    }
    // POST /connection
    public function linkEcommerce($assoc = null)
    {
        try {
            $resp = $this->request('POST', '/connection', null, null);
            return ['success' => true];
        } catch (DeclarandoApiException $dap) {
            switch ($dap->httpStatus) {
                case 204:
                case 400:
                    return ['success' => true];
                case 401:
                    // Token inválido = no se puede vincular
                    return ['success' => false, 'message' => __('Token inválido o expirado. Verifica tu código de conexión.', 'declarando-gestion-facturas')];
                default:
                    return ['success' => false, 'message' => $dap->getMessage()];
            }

        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    // POST /connection
    public function unlinkEcommerce($assoc = null)
    {
        try {
            $resp = $this->request('DELETE', '/connection', null, null);
            return ['success' => true];
        } catch (DeclarandoApiException $dap) {
            switch ($dap->httpStatus) {
                case 204:
                case 400:
                case 401: // Token inválido = consideramos como desvinculado exitosamente
                    return ['success' => true];
                default:
                    return ['success' => false, 'message' => $dap->getMessage()];
            }

        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    // POST /ping
    public function ping()
    {
        return $this->request('GET', '/ping', null, null);
    }
    // POST /v2/api/ecommerces
    /* public function createEcommerce($assoc = null)
     {
         if (!is_array($assoc))
             $assoc = array();
         return $this->request('POST', '/v2/api/ecommerces', $assoc, array());
     }

     // PATCH /v2/api/ecommerces
     public function updateEcommerce($assoc)
     {
         if (!is_array($assoc))
             $assoc = array();
         return $this->request('PATCH', '/v2/api/ecommerces', $assoc, array());
     }

     // DELETE /v2/api/ecommerces
     public function deleteEcommerce($assoc = null)
     {
         if ($assoc !== null && !is_array($assoc)) {
             throw new \InvalidArgumentException('El body de DELETE debe ser array o null.');
         }
         return $this->request('DELETE', '/v2/api/ecommerces', $assoc, array());
     }*/

    /**
     * POST /v2/api/ecommerces/uninvoiced-incomes/uuid/rectify
     * Crea un ingreso SIN factura que RECTIFICA a otro por externalId de pedido.
     */
    public function createRectifiedUninvoicedIncome($refundExternalId, $refund_id, $order_id, $orderExternalId, $totalAmount, $concept = null)
    {
        if ($refundExternalId === null || $refundExternalId === '')
            throw new \InvalidArgumentException('refundExternalId es requerido.');
        if ($orderExternalId === null || $orderExternalId === '')
            throw new \InvalidArgumentException('orderExternalId es requerido.');
        //if (!is_numeric($totalAmount))
        //   throw new \InvalidArgumentException('totalAmount debe ser numérico.');
        $scopeKey = $this->getScopeKey();

        // Idempotencia por refundExternalId
        $existing = $this->storage->getOrderId($scopeKey, (string) $refund_id);
        if ($existing) {
            return array('status' => 200, 'json' => array('uninvoicedIncomeUuid' => $existing), 'cached' => true);
        }
        // $totalAmount = (float) $totalAmount;
        $payload = array(
            'externalId' => (string) $refundExternalId,
            'concept' => $concept === null ? null : (string) $concept,
            'totalAmount' => $totalAmount < 0 ? $totalAmount : $totalAmount * -1,
            'rectifies' => array('externalId' => (string) $orderExternalId)
        );
        $scopeKey2 = $this->getScopeKey();
        $uuid = $this->storage->getOrderId($scopeKey2, $order_id);
        try {
            $res = $this->request('POST', '/uninvoiced-incomes/' . $uuid . '/rectify', $payload, array());
        } catch (DeclarandoApiException $th) {
            //throw $th;
            if (function_exists('wp_kses_post')) {
                // Log the error using WordPress logging if available
                if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                    // Safe logging using WordPress functions
                }
            }
        } catch (Exception $th) {
            if (function_exists('wp_kses_post')) {
                // Log the error using WordPress logging if available  
                if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                    // Safe logging using WordPress functions
                }
            }

        }

        $uuid = null;
        if (is_array($res['json'])) {
            if (isset($res['json']['uninvoicedIncomeUuid'])) {
                $uuid = $res['json']['uninvoicedIncomeUuid'];
            } elseif (isset($res['json']['uuid'])) {
                $uuid = $res['json']['uuid'];
            }
        }
        if ($uuid) {
            $this->storage->saveUuid(
                $scopeKey,
                $refund_id,
                (string) $refundExternalId,
                (string) $uuid,
                $payload['concept'] ?? '',
                $payload['totalAmount'] ?? 0,
                $payload['paymentStatus'] ?? 'paid',
                ['original_order_id' => $order_id]
            );
        }
        return $res;
    }

    // Helpers para IDs
    public function buildRefundExternalId($orderId, $refundId = null, $suffix = null)
    {
        $orderId = (string) $orderId;
        if ($refundId !== null && $refundId !== '') {
            return 'refund-' . (string) $refundId;
        }
        if ($suffix === null || $suffix === '') {
            $suffix = (string) time();
        }
        return 'order-' . $orderId . '-r-' . (string) $suffix;
    }

    public function getSavedUuidFor($externalId)
    {
        $scopeKey = $this->getScopeKey();
        return $this->storage->getUuid($scopeKey, (string) $externalId);
    }
}

/** Back-compat alias si tu código antiguo instanciaba DeclarandoApiClient */
if (!class_exists(__NAMESPACE__ . '\\DeclarandoApiClient')) {
    class_alias(__NAMESPACE__ . '\\Api', __NAMESPACE__ . '\\DeclarandoApiClient');
}

/**
 * ------------------------------
 *  Excepción API
 * ------------------------------
 */
class DeclarandoApiException extends \Exception
{
    public $httpStatus;
    public $responseBody;
    public $responseJson;

    public function __construct($message, $httpStatus, $responseBody, $responseJson)
    {
        parent::__construct($message, 0);
        $this->httpStatus = $httpStatus;
        $this->responseBody = $responseBody;
        $this->responseJson = $responseJson;
    }
}

/**
 * ------------------------------
 *  Storage archivo (idempotencia)
 * ------------------------------
 */
class DeclarandoApiDbStorage
{
    protected $table;

    public function __construct()
    {
        global $wpdb;
        $this->table = $wpdb->prefix . 'declarando_pedidos';
    }

    /**
     * Guardar o actualizar el UUID asociado a un externalId en un scope
     */
    public function saveUuid($scopeKey, $order_id, $externalId, $uuid, $concepto = '', $monto = 0.00, $estado = 'pendiente', $config = [])
    {
        global $wpdb;

        // Use WordPress cache functions when available
        $cache_key = 'declarando_existing_' . md5($externalId . $estado);
        $existing = wp_cache_get($cache_key, 'declarando-gestion-facturas');

        if ($existing === false) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query with caching
            $existing = $wpdb->get_var(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    "SELECT id FROM {$this->table} WHERE external_id = %s AND estado = %s",
                    $externalId,
                    $estado
                )
            );
            wp_cache_set($cache_key, $existing, 'declarando-gestion-facturas', 300); // Cache for 5 minutes
        }

        $data = [
            'order_id' => $order_id, // aquí puedes pasar el order_id real si lo tienes
            'external_id' => $externalId,
            'concepto' => $concepto,
            'monto' => $monto,
            'estado' => $estado,
            'config' => json_encode(array_merge(['scope' => $scopeKey, 'uuid' => $uuid], $config)),
            'updated_at' => current_time('mysql'),
        ];

        if ($existing) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update with manual cache invalidation
            $wpdb->update(
                $this->table,
                $data,
                ['id' => $existing],
                ['%d', '%s', '%s', '%f', '%s', '%s', '%s'],
                ['%d']
            );
            // Clear cache after update
            wp_cache_delete($cache_key, 'declarando-gestion-facturas');
            return $existing;
        } else {
            $data['created_at'] = current_time('mysql');
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table insert with manual cache invalidation
            $wpdb->insert(
                $this->table,
                $data,
                ['%d', '%s', '%s', '%f', '%s', '%s', '%s']
            );
            // Clear cache after insert
            wp_cache_delete($cache_key, 'declarando-gestion-facturas');
            return $wpdb->insert_id;
        }
    }

    /**
     * Obtener un UUID a partir de externalId y scopeKey
     */
    public function getUuid($scopeKey, $externalId)
    {
        global $wpdb;

        // Use WordPress cache when available
        $cache_key = 'declarando_uuid_' . md5($scopeKey . $externalId);
        $config = wp_cache_get($cache_key, 'declarando-gestion-facturas');

        if ($config === false) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query with caching
            $config = $wpdb->get_var(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    "SELECT config FROM {$this->table} WHERE external_id = %s ORDER BY id DESC LIMIT 1",
                    $externalId
                )
            );
            wp_cache_set($cache_key, $config, 'declarando-gestion-facturas', 300); // Cache for 5 minutes
        }

        if ($config) {
            $arr = json_decode($config, true);
            if (is_array($arr) && isset($arr['scope']) && $arr['scope'] === $scopeKey && isset($arr['uuid'])) {
                return $arr['uuid'];
            }
        }

        return null;
    }
    public function getOrderId($scopeKey, $orderId, $all = false)
    {
        global $wpdb;

        // Use WordPress cache when available
        $cache_key = 'declarando_order_' . md5($scopeKey . $orderId);
        $config = wp_cache_get($cache_key, 'declarando-gestion-facturas');

        if ($config === false) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query with caching
            $config = $wpdb->get_var(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    "SELECT config FROM {$this->table} WHERE order_id = %s ORDER BY id DESC LIMIT 1",
                    $orderId
                )
            );
            wp_cache_set($cache_key, $config, 'declarando-gestion-facturas', 300); // Cache for 5 minutes
        }

        if ($all)
            return $config;

        if ($config) {
            $arr = json_decode($config, true);
            if (is_array($arr) && isset($arr['uuid'])) {
                return $arr['uuid'];
            }
        }

        return null;
    }
}
