<?php
/**
 * Plugin Name: Declarando - Invoice Management
 * Plugin URI:  https://wordpress.org/plugins/declarando-gestion-facturas/
 * Description: Integrate your online store with Declarando to automate invoice management, sync orders, and keep your accounting up to date. Compatible with test and production environments.
 * Version:     1.0.0
 * Author:      Declarando
 * Author URI:  https://declarando.es
 * License:     GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: declarando-gestion-facturas
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.7
 * Requires PHP: 7.4
 * WC requires at least: 3.0
 * WC tested up to: 9.0
 * 
 * @package DeclarandoParaWooCommerce
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Check if WooCommerce is active
// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Using WordPress core filter
if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins'))) && !class_exists('WooCommerce')) {
    add_action('admin_notices', function () {
        echo '<div class="notice notice-error"><p>';
        echo esc_html__('Declarando para WooCommerce requires WooCommerce to be installed and active.', 'declarando-gestion-facturas');
        echo '</p></div>';
    });
    return;
}

// Define plugin constants
define('DECLARANDO_WC_VERSION', '1.0.0');
define('DECLARANDO_WC_PLUGIN_FILE', __FILE__);
define('DECLARANDO_WC_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('DECLARANDO_WC_PLUGIN_URL', plugin_dir_url(__FILE__));
define('DECLARANDO_WC_PLUGIN_BASENAME', plugin_basename(__FILE__));

spl_autoload_register(function ($class) {
    $prefix = 'Declarando\\Woocommerce\\';
    $base_dir = __DIR__ . '/src/';

    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }

    $relative_class = substr($class, $len);
    $file = $base_dir . str_replace('\\', '/', $relative_class) . '.php';

    if (file_exists($file)) {
        require $file;
    }
});


use Declarando\Woocommerce\Admin\Api;
use Declarando\Woocommerce\Admin\Declarando_WC_Integrator;
use Declarando\Woocommerce\Admin\DeclarandoApiException;
use Declarando\Woocommerce\Admin\Init;


//define('DECLARANDO_API_SANDBOX_HOST', 'https://declarando-api-test.declarando.es'); // o sandbox
define('DECLARANDO_API_SANDBOX_HOST', 'https://declarando-api-ecommerce-828031991052.europe-west4.run.app'); // o sandbox
define('DECLARANDO_API_HOST', 'https://declarando-api-ecommerce-prod.declarando.es'); // o producción
define('DECLARANDO_WC_LOG', true);
register_activation_hook(__FILE__, function () {
    global $wpdb;

    $table_name = $wpdb->prefix . 'declarando_pedidos';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        order_id BIGINT(20) UNSIGNED NOT NULL,          -- ID del pedido de WooCommerce
        external_id VARCHAR(255) DEFAULT NULL,          -- ID externo en Declarando
        concepto TEXT DEFAULT NULL,                     -- Concepto del pedido
        monto DECIMAL(15,2) DEFAULT 0.00,               -- Monto asociado
        estado VARCHAR(50) DEFAULT 'pending',         -- Estado actual pending|paid
        config LONGTEXT DEFAULT NULL,                   -- JSON con configuraciones
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,  -- Fecha de creación
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP, -- Fecha actualización
        PRIMARY KEY  (id),
        KEY order_id (order_id),
        KEY external_id (external_id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
});



final class Declarando_Plugin
{
    const OPTION_KEY = 'declarando_options';
    const CAPABILITY = 'manage_options';
    const MENU_SLUG = 'declarando';
    const PAGE_SLUG = 'declarando-facturas';

    function init()
    {
        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('admin_post_guardar_declarando_vincular', [$this, 'guardar_declarando_vincular']);
        add_action('admin_post_guardar_declarando_datos', [$this, 'guardar_declarando_datos']);
        add_action('admin_post_declarando_desvincular', [$this, 'declarando_desvincular']);
        add_action('plugins_loaded', function () {
            // Solo si WooCommerce está activo
            if (class_exists('WooCommerce')) {
                new Declarando_WC_Integrator();
            }
        }, 20);
        $init = new Init();
        $init->init();
        add_action('admin_enqueue_scripts', function ($hook) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce not required for reading page parameter
            $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
            if ($page !== self::PAGE_SLUG)
                return;

            $use_wc_assets = class_exists('WooCommerce') && function_exists('WC') && WC();

            if ($use_wc_assets) {
                // Rutas de WooCommerce
                $wc_base = trailingslashit(WC()->plugin_url()) . 'assets/';
                $css_url = $wc_base . 'css/select2.css'; // En algunas versiones es select2.min.css, probamos ambos:
                $css_min = $wc_base . 'css/select2.min.css';
                $js_url = $wc_base . 'js/selectWoo/selectWoo.full.min.js';
                $init_url = $wc_base . 'js/admin/wc-enhanced-select.js';

                // Estilos: usa el handle 'select2' si existe; si no, registra el tuyo
                if (!wp_style_is('select2', 'registered')) {
                    // Comprueba cuál existe en tu instalación usando WordPress HTTP API
                    $response = wp_remote_head($css_url, array('timeout' => 5));
                    $exists = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
                    $final_css = $exists ? $css_url : $css_min;

                    wp_register_style('decl82gf-wc-select2', $final_css, [], WC()->version);
                    wp_enqueue_style('decl82gf-wc-select2');
                    $select2_style_handle = 'decl82gf-wc-select2';
                } else {
                    wp_enqueue_style('select2');
                    $select2_style_handle = 'select2';
                }

                // Script selectWoo: si no está registrado, cárgalo
                if (!wp_script_is('selectWoo', 'registered')) {
                    wp_register_script('selectWoo', $js_url, ['jquery'], WC()->version, true);
                }
                wp_enqueue_script('selectWoo');

                // Inicializador de Woo (opcional; si no está, inicializamos nosotros)
                if (!wp_script_is('wc-enhanced-select', 'registered')) {
                    wp_register_script('wc-enhanced-select', $init_url, ['jquery', 'selectWoo'], WC()->version, true);
                }
                wp_enqueue_script('wc-enhanced-select');

                // CSS de afinado para WP-Admin (ancho, z-index, alturas)
                wp_add_inline_style($select2_style_handle, '
            .form-table .select2-container{width:100% !important; max-width:560px;}
            .select2-dropdown{z-index:100000;}
            .select2-container--default .select2-selection--multiple{
                min-height:34px; border-color:#8c8f94; border-radius:2px;
            }
        ');

                // Init explícito por si el de Woo no “engancha” en tu página
                wp_add_inline_script('wc-enhanced-select', '
            jQuery(function($){
                $("select.wc-enhanced-select").each(function(){
                    var $s = $(this);
                    if (!$s.hasClass("select2-hidden-accessible")) {
                        if ($.fn.selectWoo) {
                            $s.selectWoo({
                                width: "100%",
                                placeholder: $s.data("placeholder") || "Selecciona estados…",
                                allowClear: true
                            });
                        }
                    }
                });
            });
        ');

            }
        });
        add_action('admin_head', function () {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce not required for reading page parameter
            $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
            if ($page === 'declarando-facturas') {
                remove_all_actions('admin_notices');
                remove_all_actions('all_admin_notices');
            }
        });

    }
    public function declarando_desvincular()
    {

        $api = new Api();
        $resp = $api->unlinkEcommerce();
        $has_error = false;
        if (!($resp['success'] ?? false)) {
            add_settings_error(
                'token',
                'token_error',
                $resp['message'],
                'error'
            );
            $has_error = true;
        }
        // Si hay errores: persístelos y vuelve sin guardar
        if ($has_error) {
            set_transient('settings_errors', get_settings_errors(), 30);
            $back = wp_get_referer() ?: admin_url('admin.php?page=' . self::PAGE_SLUG);
            $back = add_query_arg('settings-updated', 'false', $back); // para que settings_errors() los muestre
            wp_safe_redirect($back, 303);
            exit;
        }
        add_settings_error(
            'token',
            'token_ok',
            __('Desvinculamos con éxito.', 'declarando-gestion-facturas'),
            'updated'
        );
        $opts = $this->get_options();
        $opts['token'] = null;
        delete_option($this::OPTION_KEY);
        add_option($this::OPTION_KEY, $opts);
        // Declarando_Plugin::set('token', null);
        set_transient('settings_errors', get_settings_errors(), 30);
        $back = wp_get_referer() ?: admin_url('admin.php?page=' . self::PAGE_SLUG);
        $back = add_query_arg('settings-updated', 'false', $back); // para que settings_errors() los muestre
        wp_safe_redirect($back, 303);
        exit;
    }
    function guardar_declarando_datos()
    {
        if (!current_user_can(self::CAPABILITY)) {
            wp_die('No autorizado');
        }

        check_admin_referer('declarando_guardar', 'declarando_nonce');
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Data is sanitized on next line
        $posted = isset($_POST['declarando_options']) ? (array) wp_unslash($_POST['declarando_options']) : [];
        $sandbox = isset($posted['sandbox']) ? sanitize_text_field($posted['sandbox']) : '';
        $has_error = false;

        $this::set('sandbox', $sandbox);
        /*$opts = get_option($this::OPTION_KEY, []);
        $opts['sandbox'] = $sandbox;

        update_option($this::OPTION_KEY, $opts);*/

        // Mensaje de éxito persistente tras el redirect
        add_settings_error(
            'token',
            'token_ok',
            __('Vinculamos cuenta con éxito.', 'declarando-gestion-facturas'),
            'updated'
        );
        set_transient('settings_errors', get_settings_errors(), 30);

        $back = wp_get_referer() ?: admin_url('admin.php?page=' . self::PAGE_SLUG);
        $back = add_query_arg('settings-updated', 'true', $back);
        wp_safe_redirect($back, 303);
        exit;
    }
    public function guardar_declarando_vincular()
    {
        if (!current_user_can(self::CAPABILITY)) {
            wp_die('No autorizado');
        }

        check_admin_referer('declarando_guardar', 'declarando_nonce');
        // Si tu input es declarando_options[token]
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Data is sanitized on next line
        $posted = isset($_POST['declarando_options']) ? (array) wp_unslash($_POST['declarando_options']) : [];
        $token = isset($posted['token']) ? sanitize_text_field($posted['token']) : '';

        $has_error = false;

        if (empty($token)) {
            add_settings_error(
                'token',
                'token_empty',
                __('El código de conexión no puede estar vacío.', 'declarando-gestion-facturas'),
                'error'
            );
            $has_error = true;
        } elseif (strlen($token) < 8) {
            add_settings_error(
                'token',
                'token_short',
                __('El código debe tener al menos 8 caracteres.', 'declarando-gestion-facturas'),
                'error'
            );
            $has_error = true;
        }
        if ($token && !empty($token)) {
            $api = new Api();
            $api->setToken($token);
            $resp = $api->linkEcommerce();
            if (!($resp['success'] ?? false)) {
                add_settings_error(
                    'token',
                    'token_error',
                    $resp['message'],
                    'error'
                );
                $has_error = true;
            }
        }

        // Si hay errores: persístelos y vuelve sin guardar
        if ($has_error) {
            set_transient('settings_errors', get_settings_errors(), 30);
            $back = wp_get_referer() ?: admin_url('admin.php?page=' . self::PAGE_SLUG);
            $back = add_query_arg('settings-updated', 'false', $back); // para que settings_errors() los muestre
            wp_safe_redirect($back, 303);
            exit;
        }

        // Guardado (array declarando_options)
        $this::set('token', $token);
        /*  $opts = get_option($this::OPTION_KEY, []);
          $opts['token'] = $token;

          update_option($this::OPTION_KEY, $opts);*/

        // Mensaje de éxito persistente tras el redirect
        add_settings_error(
            'token',
            'token_ok',
            __('Vinculamos cuenta con éxito.', 'declarando-gestion-facturas'),
            'updated'
        );
        set_transient('settings_errors', get_settings_errors(), 30);

        $back = wp_get_referer() ?: admin_url('admin.php?page=' . self::PAGE_SLUG);
        $back = add_query_arg('settings-updated', 'true', $back);
        wp_safe_redirect($back, 303);
        exit;
    }
    /** Menú principal Declarando + submenú Facturas */
    public function register_admin_menu()
    {
        // Use WordPress filesystem functions instead of direct file_get_contents
        $icon_path = __DIR__ . '/assets/icono.svg';
        $icon_svg = '';

        if (file_exists($icon_path)) {
            global $wp_filesystem;
            if (empty($wp_filesystem)) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
                WP_Filesystem();
            }

            if (!empty($wp_filesystem) && $wp_filesystem->exists($icon_path)) {
                $svg_content = $wp_filesystem->get_contents($icon_path);
                if ($svg_content !== false) {
                    $icon_svg = 'data:image/svg+xml;base64,' . base64_encode($svg_content);
                }
            }
        }


        add_submenu_page(
            'woocommerce', // 👈 Slug del menú padre (WooCommerce)
            __('Declarando', 'declarando-gestion-facturas'), // Título de la página
            __('Declarando', 'declarando-gestion-facturas'), // Título del submenú
            'manage_woocommerce', // Capacidad necesaria
            'declarando-facturas', // Slug único de tu página
            [$this, 'render_facturas_page'] // Callback que renderiza la página
        );
    }

    /** Render de la página de Facturas */
    public function render_facturas_page()
    {

        if (!current_user_can(self::CAPABILITY))
            return;
        ?>
        <div class="wrap">
            <?php
            if (empty($this->get('token'))) {
                ?>
                <h1><img src="<?php echo esc_url(plugin_dir_url(__FILE__) . '/assets/icono-blue.svg'); ?>">
                    <?php echo esc_html__('Accede a Declarando', 'declarando-gestion-facturas'); ?>
                </h1>
                <?php
            } else {
                ?>
                <h1><img src="<?php echo esc_url(plugin_dir_url(__FILE__) . '/assets/icono-blue.svg'); ?>">
                    <?php echo esc_html__('Conexión completada 🚀', 'declarando-gestion-facturas'); ?>
                </h1>
                <?php
            }
            ?>
        </div>
        <?php settings_errors(); ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" id="real_form_account">
            <?php wp_nonce_field('declarando_guardar', 'declarando_nonce'); ?>
            <?php
            settings_fields('declarando_settings_group');
            do_settings_sections(self::PAGE_SLUG);
            if (empty($this->get('token'))) {
                submit_button(__('Vincular tienda', 'declarando-gestion-facturas'));
                ?>
                <input type="hidden" name="action" value="guardar_declarando_vincular">
                <?php
            } else {
                ?>
                <p>
                    Tus pedidos de WooCommerce ya se guardarán como ingresos en Declarando, para que ahorres tiempo y lleves tu
                    negocio siempre al día.
                </p>
                <?php
                //   echo '<p><strong style="color:red;">Importante:</strong> Por favor, asegúrate de que se ha seleccionado la misma moneda tanto en Woocommerce como en Contasimple apra evitar cualquier posible problema de conversión en el momento de sincronizar.</p>';
                echo '<br><div class="inlined-buttons">';
                //submit_button(__('Guardar cambios', 'declarando'));
                echo '<a href="' . esc_url(admin_url('admin-post.php')) . '?action=declarando_desvincular" class="button button-primary">Desvincular tienda</a>';
                echo '</div>';
                ?>
                <input type="hidden" name="action" value="guardar_declarando_datos">
                <?php
            }
            ?>

        </form>
        <?php
    }
    /** Página principal (puedes usarla de índice o enlaces rápidos) */

    /** Registro de opciones y campos (Settings API) */
    public function register_settings()
    {
        register_setting(
            'declarando_settings_group',
            self::OPTION_KEY,
            ['type' => 'array', 'sanitize_callback' => [$this, 'sanitize_options']]
        );
        if (empty($this->get('token'))) {
            add_settings_section(
                'declarando_section_general',
                '',
                function () {
                    echo esc_html__('Acceder a Declarando a través de WooCommerce', 'declarando-gestion-facturas');
                },
                self::PAGE_SLUG
            );

            add_settings_field(
                'token',
                __('Código de conexión de Declarando', 'declarando-gestion-facturas'),
                [$this, 'field_text'],
                self::PAGE_SLUG,
                'declarando_section_general',
                [
                    'key' => 'token',
                    'placeholder' => 'xxxxxxx',
                    'type' => 'text',
                    'description' => 'Necesitas un código de conexión para empezar a sincronizar tu tienda con Declarando. Puedes generarlo desde tu Perfil > Tienda online de Declarando. '
                ]
            );
            return;
        }

        //====================================================================
        add_settings_section(
            'declarando_section_general',
            '',
            function () {
            },
            self::PAGE_SLUG
        );

    }

    /** Sanitiza todas las opciones */
    public function sanitize_options($input)
    {
        $out = $this->get_options(); // base actual
        $in = is_array($input) ? $input : [];
        if (isset($in['token']))
            $out['token'] = isset($in['token']) ? sanitize_text_field($in['token']) : '';
        if (isset($in['sandbox']))
            $out['sandbox'] = !empty($in['sandbox']) ? 1 : 0;
        return $out;
    }

    /** Helpers de campos */
    private function get_options()
    {
        $defaults = [
            'token' => '',
            'api_endpoint' => DECLARANDO_API_HOST,
            'sandbox_api_endpoint' => DECLARANDO_API_SANDBOX_HOST,
            'sandbox' => 1,
            'allowed_statuses' => ['wc-processing', 'wc-completed'],
            'rembolso_statuses' => ['wc-cancelled', 'wc-refunded']
        ];
        $opts = get_option(self::OPTION_KEY, []);
        $opts['api_endpoint'] = DECLARANDO_API_HOST;
        $opts['sandbox_api_endpoint'] = DECLARANDO_API_SANDBOX_HOST;
        return wp_parse_args(is_array($opts) ? $opts : [], $defaults);
    }

    public function field_text($args)
    {
        $options = get_option(self::OPTION_KEY, []);

        $key = isset($args['key']) ? (string) $args['key'] : '';
        $type = isset($args['type']) ? (string) $args['type'] : 'text';
        $val = isset($options[$key]) ? (string) $options[$key] : '';
        $ph = isset($args['placeholder']) ? (string) $args['placeholder'] : '';

        printf(
            '<input type="%1$s" class="regular-text" name="%2$s[%3$s]" value="%4$s" placeholder="%5$s" />',
            esc_attr($type),              // %1$s
            esc_attr(self::OPTION_KEY),   // %2$s
            esc_attr($key),               // %3$s
            esc_attr($val),               // %4$s
            esc_attr($ph)                 // %5$s
        );
        if (!empty($args['description'])) {
            echo '<p>';
            echo wp_kses_post($args['description'] ?? '');
            echo '</p>';
        }

    }
    public function field_statuses_multiselect($args)
    {
        $key = $args['key'] ?? 'allowed_statuses';
        $description = $args['description'] ?? '';
        $default = $args['default'] ?? [];

        $opts = self::get_options();
        $current = isset($opts[$key]) && is_array($opts[$key]) ? $opts[$key] : (array) $default;

        if (!function_exists('wc_get_order_statuses')) {
            echo '<p style="color:#b00">' . esc_html__('WooCommerce no está activo.', 'declarando-gestion-facturas') . '</p>';
            return;
        }
        $statuses = wc_get_order_statuses();

        echo '<select multiple
                 class="wc-enhanced-select decl82gf-select2"
                 style="width:100%"
                 data-placeholder="' . esc_attr__('Selecciona estados…', 'declarando-gestion-facturas') . '"
                 name="declarando_options[' . esc_attr($key) . '][]">';

        foreach ($statuses as $status_key => $status_label) {
            $selected = in_array($status_key, $current, true) ? 'selected' : '';
            echo '<option value="' . esc_attr($status_key) . '" ' . esc_attr($selected) . '>' .
                esc_html($status_label) . ' (' . esc_html($status_key) . ')' .
                '</option>';
        }

        echo '</select>';

        if ($description) {
            echo '<p class="description">' . wp_kses_post($description) . '</p>';
        }
    }

    public function field_number($args)
    {
        $opts = $this->get_options();

        $key = isset($args['key']) ? (string) $args['key'] : '';
        $val = isset($opts[$key]) ? $opts[$key] : '';
        $min = isset($args['min']) ? $args['min'] : '0';
        $max = isset($args['max']) ? $args['max'] : '100';
        $step = isset($args['step']) ? $args['step'] : '1';
        $ph = isset($args['placeholder']) ? (string) $args['placeholder'] : '';
        $id = 'opt_' . $key;

        printf(
            '<input type="number" id="%1$s" class="small-text" name="%2$s[%3$s]" value="%4$s" min="%5$s" max="%6$s" step="%7$s" placeholder="%8$s" />',
            esc_attr($id),                // %1$s
            esc_attr(self::OPTION_KEY),   // %2$s
            esc_attr($key),               // %3$s
            esc_attr($val),               // %4$s
            esc_attr($min),               // %5$s
            esc_attr($max),               // %6$s
            esc_attr($step),              // %7$s
            esc_attr($ph)                 // %8$s
        );
    }

    public function field_checkbox($args)
    {
        $opts = $this->get_options();

        $key = isset($args['key']) ? (string) $args['key'] : '';
        $label = isset($args['label']) ? (string) $args['label'] : '';
        $id = 'opt_' . $key;
        $is_on = !empty($opts[$key]); // cualquier valor truthy cuenta como marcado

        printf(
            // hidden para asegurar que se envía 0 cuando no está marcado
            '<input type="hidden" name="%1$s[%2$s]" value="0" />
         <label for="%5$s">
            <input type="checkbox" id="%5$s" name="%1$s[%2$s]" value="1" %3$s />
            %4$s
         </label>',
            esc_attr(self::OPTION_KEY),     // %1$s
            esc_attr($key),                 // %2$s
            checked($is_on, true, false),   // %3$s
            esc_html($label),               // %4$s
            esc_attr($id)                   // %5$s
        );
    }

    public function field_textarea($args)
    {
        $opts = $this->get_options();

        $key = isset($args['key']) ? (string) $args['key'] : '';
        $val = isset($opts[$key]) ? (string) $opts[$key] : '';
        $rows = isset($args['rows']) ? max(1, (int) $args['rows']) : 5;
        $ph = isset($args['placeholder']) ? (string) $args['placeholder'] : '';
        $id = 'opt_' . $key;

        printf(
            '<textarea id="%1$s" name="%2$s[%3$s]" rows="%4$d" class="large-text code" placeholder="%5$s">%6$s</textarea>',
            esc_attr($id),                // %1$s
            esc_attr(self::OPTION_KEY),   // %2$s
            esc_attr($key),               // %3$s
            (int) $rows,                  // %4$d
            esc_attr($ph),                // %5$s
            esc_textarea($val)            // %6$s
        );
    }



    /** CSS leve opcional */
    public function enqueue_admin_assets($hook)
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce not required for reading page parameter
        $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';

        if ($page === self::PAGE_SLUG) {
            // Enqueue settings page assets
            wp_enqueue_style(
                'declarando-admin-settings',
                DECLARANDO_WC_PLUGIN_URL . 'assets/css/admin-settings.css',
                [],
                DECLARANDO_WC_VERSION
            );

            wp_enqueue_script(
                'declarando-admin-settings',
                DECLARANDO_WC_PLUGIN_URL . 'assets/js/admin-settings.js',
                ['jquery'],
                DECLARANDO_WC_VERSION,
                true
            );

            wp_add_inline_style('declarando-admin-settings', '
                #declarando-form .form-table th { width: 220px; }
            ');
        }
    }

    /** Helper público estático para obtener opciones en tu tema/otros plugins */
    public static function get($key = null, $default = null)
    {
        $self = new self(); // sólo para usar get_options()
        global $declarando_gestion_facturas_opts;
        if (empty($declarando_gestion_facturas_opts))
            $declarando_gestion_facturas_opts = $opts = $self->get_options();
        else
            $opts = $declarando_gestion_facturas_opts;
        if ($key === null)
            return $opts;
        return array_key_exists($key, $opts) ? $opts[$key] : $default;
    }
    /** Helper público estático para obtener opciones en tu tema/otros plugins */
    public static function set($key = null, $value = null)
    {
        global $declarando_gestion_facturas_opts;
        $self = new self(); // sólo para usar get_options()
        if (empty($declarando_gestion_facturas_opts))
            $opts = $self->get_options();
        else
            $opts = $declarando_gestion_facturas_opts;
        $opts[$key] = $value;
        $declarando_gestion_facturas_opts = $opts;
        update_option($self::OPTION_KEY, $opts);
        return $opts[$key];
    }
}

$declarando_wc_plugin = new Declarando_Plugin();
$declarando_wc_plugin->init();