# Declarando WooCommerce Integration - Documentación para Desarrolladores

## Instalación y Configuración

### Requisitos del sistema
- WordPress 5.0+
- WooCommerce 3.0+
- PHP 7.4+
- Cuenta activa en Declarando

### Instalación manual
1. Descargar el plugin desde WordPress.org o GitHub
2. Subir al directorio `/wp-content/plugins/`
3. Activar desde el panel de WordPress
4. Configurar en **WooCommerce > Declarando**

## Hooks y Filters

### Actions (Acciones)

#### `declarando_wc_before_sync_order`
Se ejecuta antes de sincronizar un pedido con Declarando.

```php
add_action('declarando_wc_before_sync_order', function($order_id, $order_data) {
    // Tu código aquí
    error_log("Sincronizando pedido: " . $order_id);
}, 10, 2);
```

#### `declarando_wc_after_sync_order`
Se ejecuta después de sincronizar un pedido exitosamente.

```php
add_action('declarando_wc_after_sync_order', function($order_id, $response) {
    // Tu código aquí
    update_post_meta($order_id, '_declarando_sync_time', current_time('timestamp'));
}, 10, 2);
```

#### `declarando_wc_sync_failed`
Se ejecuta cuando falla la sincronización de un pedido.

```php
add_action('declarando_wc_sync_failed', function($order_id, $error_message) {
    // Tu código aquí
    error_log("Error sincronizando pedido {$order_id}: {$error_message}");
}, 10, 2);
```

#### `declarando_wc_uninstalled`
Se ejecuta cuando el plugin se desinstala completamente.

```php
add_action('declarando_wc_uninstalled', function() {
    // Limpieza adicional si es necesario
    delete_option('mi_plugin_declarando_config');
});
```

### Filters (Filtros)

#### `declarando_wc_order_data`
Permite modificar los datos del pedido antes de enviarlos a Declarando.

```php
add_filter('declarando_wc_order_data', function($order_data, $order) {
    // Añadir datos personalizados
    $order_data['custom_field'] = get_post_meta($order->get_id(), '_custom_field', true);
    return $order_data;
}, 10, 2);
```

#### `declarando_wc_api_timeout`
Permite modificar el timeout de las llamadas API.

```php
add_filter('declarando_wc_api_timeout', function($timeout) {
    return 30; // 30 segundos
});
```

#### `declarando_wc_log_enabled`
Controla si los logs están activos.

```php
add_filter('declarando_wc_log_enabled', function($enabled) {
    return (defined('WP_DEBUG') && WP_DEBUG);
});
```

#### `declarando_wc_syncable_statuses`
Permite modificar qué estados de pedido se sincronizan.

```php
add_filter('declarando_wc_syncable_statuses', function($statuses) {
    // Añadir estado personalizado
    $statuses[] = 'wc-custom-status';
    return $statuses;
});
```

## API del Plugin

### Clase Principal: `Declarando_Plugin`

#### Métodos estáticos disponibles:

##### `Declarando_Plugin::get($key, $default)`
Obtiene opciones de configuración del plugin.

```php
$api_key = Declarando_Plugin::get('token');
$sandbox_mode = Declarando_Plugin::get('sandbox', false);
$all_options = Declarando_Plugin::get();
```

##### `Declarando_Plugin::set($key, $value)`
Establece opciones de configuración del plugin.

```php
Declarando_Plugin::set('sandbox', true);
Declarando_Plugin::set('custom_setting', 'mi_valor');
```

### Clase API: `Declarando\Woocommerce\Admin\Api`

#### Métodos principales:

##### `ping()`
Verifica la conectividad con la API de Declarando.

```php
$api = new \Declarando\Woocommerce\Admin\Api();
$response = $api->ping();
if ($response['success']) {
    echo "Conexión exitosa";
}
```

##### `linkEcommerce()`
Vincula la tienda con Declarando.

```php
$api = new \Declarando\Woocommerce\Admin\Api();
$api->setToken('tu_token_aqui');
$response = $api->linkEcommerce();
```

##### `createUninvoicedIncome($order_id, $external_id, $date, $amount, $status)`
Crea un ingreso no facturado en Declarando.

```php
$api = new \Declarando\Woocommerce\Admin\Api();
$response = $api->createUninvoicedIncome(
    123,                    // ID interno
    'P-123',               // ID externo
    '2024-11-07',          // Fecha
    100.50,                // Cantidad
    'paid'                 // Estado
);
```

## Constantes del Plugin

### Constantes principales:
```php
DECLARANDO_WC_VERSION           // Versión del plugin
DECLARANDO_WC_PLUGIN_FILE       // Archivo principal del plugin
DECLARANDO_WC_PLUGIN_DIR        // Directorio del plugin
DECLARANDO_WC_PLUGIN_URL        // URL del plugin
DECLARANDO_WC_PLUGIN_BASENAME   // Basename del plugin
```

### Constantes de configuración:
```php
DECLARANDO_API_HOST             // Host de producción
DECLARANDO_API_SANDBOX_HOST     // Host de sandbox
DECLARANDO_WC_LOG               // Activar logs
```

## Base de Datos

### Tabla: `wp_declarando_pedidos`

Estructura de la tabla personalizada creada por el plugin:

```sql
CREATE TABLE wp_declarando_pedidos (
    id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
    order_id BIGINT(20) UNSIGNED NOT NULL,
    external_id VARCHAR(100),
    declarando_id VARCHAR(100),
    status VARCHAR(50),
    sync_date DATETIME,
    error_message TEXT,
    retry_count INT DEFAULT 0,
    PRIMARY KEY (id),
    INDEX idx_order_id (order_id),
    INDEX idx_external_id (external_id),
    INDEX idx_status (status)
);
```

## Personalización Avanzada

### Modificar comportamiento de sincronización

```php
// Sincronizar solo pedidos de cierto valor
add_filter('declarando_wc_should_sync_order', function($should_sync, $order) {
    return $order->get_total() >= 50; // Solo pedidos >= 50€
}, 10, 2);

// Modificar datos antes del envío
add_filter('declarando_wc_order_data', function($data, $order) {
    // Añadir metadatos personalizados
    $data['store_location'] = get_option('woocommerce_store_city');
    return $data;
}, 10, 2);
```

### Integración con otros plugins

```php
// Integración con WooCommerce Subscriptions
add_action('declarando_wc_before_sync_order', function($order_id) {
    if (function_exists('wcs_order_contains_subscription')) {
        if (wcs_order_contains_subscription($order_id)) {
            // Lógica especial para suscripciones
        }
    }
});
```

## Troubleshooting

### Logs del plugin
Los logs se almacenan usando el sistema de logs de WordPress cuando `DECLARANDO_WC_LOG` está activo.

```php
// Activar logs en wp-config.php
define('DECLARANDO_WC_LOG', true);
define('WP_DEBUG', true);
define('WP_DEBUG_LOG', true);
```

### Problemas comunes

#### 1. Pedidos no se sincronizan
- Verificar que WooCommerce está activo
- Comprobar la configuración del token API
- Revisar los estados de sincronización configurados
- Verificar conectividad con Declarando

#### 2. Errores de timeout
- Aumentar el timeout de PHP
- Verificar la conectividad del servidor
- Usar el filtro `declarando_wc_api_timeout`

#### 3. Problemas de permisos
- Verificar que el usuario tiene capability `manage_woocommerce`
- Comprobar nonces en formularios personalizados

### Debug avanzado

```php
// Activar debug completo
add_filter('declarando_wc_log_enabled', '__return_true');

// Ver todas las llamadas API
add_action('declarando_wc_before_api_call', function($endpoint, $data) {
    error_log("API Call: {$endpoint} - " . json_encode($data));
});
```

## Actualizaciones

### Desde WordPress.org
Las actualizaciones se gestionan automáticamente por WordPress cuando el plugin está instalado desde el repositorio oficial.

### Desde fuente privada
El plugin incluye un sistema de auto-actualización que se activa solo cuando:
- No está instalado desde WordPress.org
- El archivo `autoupdate.php` está presente
- La función `declarando_wc_is_from_wordpress_org()` retorna `false`

## Contribuir

### Reportar bugs
- Usar el [foro de soporte de WordPress.org](https://wordpress.org/support/plugin/declarando-woocommerce/)
- Incluir versión de WordPress, WooCommerce y PHP
- Proporcionar logs si es posible

### Proponer mejoras
- Crear issues en el repositorio del proyecto
- Seguir las guías de contribución
- Incluir tests cuando sea apropiado

## Licencia

Este plugin está licenciado bajo GPL v2 o posterior, compatible con la licencia de WordPress.