(function ($) {
  'use strict';

  // Security check
  if (typeof window.DCXSiteBackup !== 'object') {
    console.error('DCXSiteBackup not defined. Did wp_localize_script run?');
    return;
  }

  var API = window.DCXSiteBackup;
  var polling = null;

  function setStatus(msg, isError) {
    $('#backup-status').html(
      isError ? '<span style="color:red;">' + String(msg) + '</span>' : String(msg)
    );
  }

  function toggleStartButton(disabled, label) {
    var $btn = $('#start-backup');
    $btn.prop('disabled', !!disabled);
    if (label) $btn.text(label);
  }

  function check_status() {
    $.post(API.ajax_url, {
      action: API.actions.status,
      nonce: API.nonce
    })
      .done(function (response) {
        if (!response || !response.success) {
          setStatus((response && response.data) || 'Unknown error', true);
          return;
        }

        setStatus(response.data.message || '');

        if (!response.data.done) {
          // continue polling
          polling = setTimeout(check_status, 3000);
        } else {
          toggleStartButton(false, 'Start Backup');
          refresh_backup_list();
        }
      })
      .fail(function (xhr) {
        setStatus('Network error (' + xhr.status + ')', true);
        toggleStartButton(false, 'Start Backup');
      });
  }

  function refresh_backup_list() {
    $.post(API.ajax_url, {
      action: API.actions.list,
      nonce: API.nonce
    })
      .done(function (resp) {
        if (!resp || !resp.success) return;

        // Simple escape function
        function escHtml(s) {
          return String(s)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
        }

        var html = '<h2>Existing Backups</h2><ul>';
        (resp.data || []).forEach(function (file) {
          var url  = escHtml(file.url || '');
          var name = escHtml(file.name || '');
          html +=
            '<li><a href="' + url + '" target="_blank" rel="noopener noreferrer">' +
            name +
            '</a> ' +
            '<button class="button delete-backup-file" data-filename="' + name + '">Delete</button></li>';
        });
        html += '</ul>';

        $('.wrap h2, .wrap ul').remove();
        $('#backup-status').before(html);
      })
      .fail(function () {
        // silently ignore
      });
  }


  $(document).ready(function () {
    // initial state
    check_status();
    refresh_backup_list();

    $('#start-backup').on('click', function (e) {
      e.preventDefault();
      if (polling) {
        clearTimeout(polling);
        polling = null;
      }

      toggleStartButton(true, 'Working…');
      setStatus('Starting backup…');

      $.post(API.ajax_url, {
        action: API.actions.start,
        nonce: API.nonce
      })
        .done(function (response) {
          if (!response || !response.success) {
            setStatus((response && response.data) || 'Unknown error', true);
            toggleStartButton(false, 'Start Backup');
            return;
          }
          // start polling
          check_status();
        })
        .fail(function (xhr) {
          setStatus('Network error (' + xhr.status + ')', true);
          toggleStartButton(false, 'Start Backup');
        });
    });

    // deleting file
    $(document).on('click', '.delete-backup-file', function () {
      var $btn = $(this);
      var filename = $btn.data('filename');

      if (!filename) return;
      if (!window.confirm('Are you sure you want to delete this backup file?')) return;

      $btn.prop('disabled', true).text('Deleting…');

      $.post(API.ajax_url, {
        action: API.actions.delete,
        file: filename,
        nonce: API.nonce
      })
        .done(function (res) {
          if (res && res.success) {
            $btn.closest('li').fadeOut(200, function () {
              $(this).remove();
              // if empty, refresh
              if ($('.wrap ul li').length === 0) {
                refresh_backup_list();
              }
            });
          } else {
            alert('Failed to delete file: ' + ((res && res.data) || 'Unknown error'));
            $btn.prop('disabled', false).text('Delete');
          }
        })
        .fail(function (xhr) {
          alert('Network error (' + xhr.status + ')');
          $btn.prop('disabled', false).text('Delete');
        });
    });
  });
})(jQuery);
