<?php
/*
Plugin Name: DCX Site Backup
Description: Make full backup of your website. When ran it will save docroot + database (.sql.gz) into wp-content/backups/
Version:     6.3
Author:      darktwen
Author URI:  https://dcx.rs/
Text Domain: dcx-site-backup
License:     GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if (!defined('ABSPATH')) exit;

// Unique constants/prefix
define('DCXSITEBACKUP_SLUG', 'dcxsitebackup');
define('DCXSITEBACKUP_VER',  '6.3');
define('DCXSITEBACKUP_DIR',  WP_CONTENT_DIR . '/backups');

// Ensure backup dir exists on activation
register_activation_hook(__FILE__, function () {
    wp_mkdir_p(DCXSITEBACKUP_DIR);
});

add_action('admin_menu', function () {
    add_management_page(
        'DCX Site Backup',
        'DCX Site Backup',
        'manage_options',
        DCXSITEBACKUP_SLUG,
        'dcxsitebackup_page'
    );
});

add_action('admin_enqueue_scripts', function ($hook) {
    if ($hook !== 'tools_page_' . DCXSITEBACKUP_SLUG) {
        return;
    }

    wp_register_script(
        DCXSITEBACKUP_SLUG . '-admin',
        plugin_dir_url(__FILE__) . 'site-backup.js',
        ['jquery'],
        DCXSITEBACKUP_VER,
        true
    );

    if (function_exists('wp_script_add_data')) {
        wp_script_add_data(DCXSITEBACKUP_SLUG . '-admin', 'defer', true);
    }

    wp_localize_script(
        DCXSITEBACKUP_SLUG . '-admin',
        'DCXSiteBackup',
        [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('dcxsitebackup_wp_migration'),
            'nonceDismiss' => wp_create_nonce('dcxsitebackup_dismiss_notice'),
            'actions'  => [
                'start'   => 'dcxsitebackup_wp_migration_start',
                'status'  => 'dcxsitebackup_wp_migration_status',
                'list'    => 'dcxsitebackup_wp_migration_list_files',
                'delete'  => 'dcxsitebackup_wp_migration_delete_backup',
                'dismiss' => 'dcxsitebackup_dismiss_notice',
            ],
        ]
    );

    wp_enqueue_script(DCXSITEBACKUP_SLUG . '-admin');
});

/** Admin page */
function dcxsitebackup_page() {
    echo '<div class="wrap"><h1>DCX Site Backup</h1>';

    if (is_dir(DCXSITEBACKUP_DIR)) {
        $files = array_merge(
            glob(DCXSITEBACKUP_DIR . '/*.zip') ?: [],
            glob(DCXSITEBACKUP_DIR . '/*.sql.gz') ?: []
        );
        if ($files) {
            echo '<h2>Existing Backups</h2><ul>';
            foreach ($files as $f) {
                $url = content_url('backups/' . basename($f));
                echo '<li>' .
                    '<a href="' . esc_url($url) . '" target="_blank">' . esc_html(basename($f)) . '</a> ' .
                    '<button class="button delete-backup-file" data-filename="' . esc_attr(basename($f)) . '">Delete</button>' .
                 '</li>';
            }
            echo '</ul>';
        }
    }

    echo '<button id="start-backup" class="button button-primary">Start Backup</button>';
    echo '<div id="backup-status" style="margin-top:1em;"></div></div>';
}

/** AJAX: start */
add_action('wp_ajax_dcxsitebackup_wp_migration_start', function () {
    check_ajax_referer('dcxsitebackup_wp_migration', 'nonce');
    if (!current_user_can('manage_options')) wp_send_json_error('forbidden');

    wp_mkdir_p(DCXSITEBACKUP_DIR);

    $paths_to_include = [
        WP_CONTENT_DIR . '/uploads',
        WP_CONTENT_DIR . '/themes',
        WP_CONTENT_DIR . '/plugins',
    ];

    $all_files = [];
    foreach ($paths_to_include as $path) {
        if (!is_dir($path)) continue;
        $rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($path, FilesystemIterator::SKIP_DOTS));
        foreach ($rii as $file) {
            if ($file->isFile()) $all_files[] = $file->getRealPath();
        }
    }

    dcxsitebackup_dump_db();

    global $wp_version;
    $wp_version_file = DCXSITEBACKUP_DIR . '/wp-version.txt';
    file_put_contents($wp_version_file, 'WordPress version: ' . $wp_version . "\n");
    $all_files[] = $wp_version_file;

    $state = [
        'zip'     => DCXSITEBACKUP_DIR . '/site-backup-' . gmdate('Ymd-His') . '.zip',
        'current' => 0,
        'files'   => $all_files,
        'done'    => false
    ];
    file_put_contents(DCXSITEBACKUP_DIR . '/backup-' . DCXSITEBACKUP_SLUG . '-state.json', json_encode($state));
    wp_schedule_single_event(time(), 'dcxsitebackup_wp_migration_cron');

    wp_send_json_success(['message' => 'Backup scheduled.']);
});

/** AJAX: status */
add_action('wp_ajax_dcxsitebackup_wp_migration_status', function () {
    check_ajax_referer('dcxsitebackup_wp_migration', 'nonce');
    if (!current_user_can('manage_options')) wp_send_json_error('forbidden');

    $state_file = DCXSITEBACKUP_DIR . '/backup-' . DCXSITEBACKUP_SLUG . '-state.json';

    if (!file_exists($state_file)) {
        wp_send_json_success(['done' => true, 'message' => 'No backup in progress']);
    }

    $state = json_decode(file_get_contents($state_file), true);

    if (!empty($state['done'])) {
        wp_delete_file($state_file);
        $zip_url = content_url('backups/' . basename($state['zip']));
        wp_send_json_success(['done' => true, 'message' => 'Backup complete!', 'zip' => $zip_url]);
    }

    $progress = round(($state['current'] / max(1, count($state['files']))) * 100);
    wp_send_json_success(['done' => false, 'message' => 'Backup in progress: ' . $progress . '%']);
});

/** AJAX: list files */
add_action('wp_ajax_dcxsitebackup_wp_migration_list_files', function () {
    check_ajax_referer('dcxsitebackup_wp_migration', 'nonce');
    if (!current_user_can('manage_options')) wp_send_json_error('forbidden');

    $files = array_merge(
        glob(DCXSITEBACKUP_DIR . '/*.zip') ?: [],
        glob(DCXSITEBACKUP_DIR . '/*.sql.gz') ?: []
    );
    $out = [];
    foreach ($files as $f) {
        $out[] = [
            'url'  => content_url('backups/' . basename($f)),
            'name' => basename($f)
        ];
    }
    wp_send_json_success($out);
});

/** Cron worker */
add_action('dcxsitebackup_wp_migration_cron', function () {
    $state_file = DCXSITEBACKUP_DIR . '/backup-' . DCXSITEBACKUP_SLUG . '-state.json';
    if (!file_exists($state_file)) return;

    $state = json_decode(file_get_contents($state_file), true);
    $zip_path = $state['zip'];

    $zip = new ZipArchive();
    $zip->open($zip_path, ZipArchive::CREATE);
    $batch = 2000; $count = 0;

    while ($state['current'] < count($state['files']) && $count < $batch) {
        $file = $state['files'][$state['current']];
        if (is_readable($file)) {
            $relative = ltrim(str_replace(ABSPATH, '', $file), '/');
            $zip->addFile($file, ltrim($relative, '/'));
        }
        $state['current']++; $count++;
    }

    if ($state['current'] >= count($state['files'])) {
        $zip->close();
        $state['done'] = true;
        file_put_contents($state_file, json_encode($state));

        $wp_version_txt = DCXSITEBACKUP_DIR . '/wp-version.txt';
        if (file_exists($wp_version_txt)) {
			wp_delete_file($wp_version_txt);
		}
        return;
    }

    $zip->close();
    file_put_contents($state_file, json_encode($state));
    wp_schedule_single_event(time() + 1, 'dcxsitebackup_wp_migration_cron');
});

/** DB dump */
function dcxsitebackup_dump_db() {
    global $wpdb;
    wp_mkdir_p(DCXSITEBACKUP_DIR);

    // 1) Temporary .sql file
    $sql_path = DCXSITEBACKUP_DIR . '/db-backup-' . gmdate('Ymd-His') . '.sql';
    $f = fopen($sql_path, 'w');
    if (!$f) return false;

    fwrite($f, "-- WordPress Database Backup\n");
    fwrite($f, "-- Generated: " . gmdate("Y-m-d H:i:s") . " UTC\n\n");
    fwrite($f, "SET NAMES utf8mb4;\n");
    fwrite($f, "SET FOREIGN_KEY_CHECKS=0;\n");
    fwrite($f, "SET SQL_MODE='NO_AUTO_VALUE_ON_ZERO';\n");
    fwrite($f, "SET time_zone = '+00:00';\n\n");

    $tables = $wpdb->get_col($wpdb->prepare(
        "SHOW TABLES LIKE %s",
        $wpdb->esc_like($wpdb->prefix) . '%'
    ));

    foreach ($tables as $table) {
        if (!preg_match('/^[a-zA-Z0-9_]+$/', $table)) continue;

        $table_safe = str_replace('`', '', $table);

        $create_row = $wpdb->get_row("SHOW CREATE TABLE `$table_safe`", ARRAY_N);
        if (!isset($create_row[1])) continue;

        $create_sql = $create_row[1];
        fwrite($f, "\n-- Table structure for `$table`\n");
        fwrite($f, "DROP TABLE IF EXISTS `$table_safe`;\n");
        fwrite($f, "$create_sql;\n\n");

        $total_rows = (int) $wpdb->get_var("SELECT COUNT(*) FROM `$table_safe`");
        if ($total_rows > 0) {
            fwrite($f, "-- Dumping data for table `$table`\n");
            $offset = 0;
            $limit  = 1000;

            while ($offset < $total_rows) {
                $sql  = "SELECT * FROM `$table_safe` LIMIT $offset, $limit";
                $rows = $wpdb->get_results($sql, ARRAY_A);
                if (empty($rows)) break;

                $insert_values = [];
                $columns = null;

                foreach ($rows as $row) {
                    if ($columns === null) {
                        $columns = array_map(function($col) {
                            return '`' . str_replace('`', '', $col) . '`';
                        }, array_keys($row));
                    }

                    $values = array_map(function($val) {
                        if (is_null($val)) return 'NULL';
                        return "'" . addslashes($val) . "'";
                    }, array_values($row));

                    $insert_values[] = '(' . implode(',', $values) . ')';
                }

                if (!empty($insert_values)) {
                    $stmt = sprintf(
                        "INSERT INTO `%s` (%s) VALUES %s;\n",
                        $table_safe,
                        implode(',', $columns),
                        implode(',', $insert_values)
                    );
                    fwrite($f, $stmt);
                }

                $offset += $limit;
            }
        }
    }

    fwrite($f, "\nSET FOREIGN_KEY_CHECKS=1;\n");
    fwrite($f, "-- Dump completed\n");
    fclose($f);

    // 2) Add .sql in .zip and delete .sql
    $zip_path = DCXSITEBACKUP_DIR . '/db-backup-' . gmdate('Ymd-His') . '.zip';
    $zip = new ZipArchive();
    if ($zip->open($zip_path, ZipArchive::CREATE) !== true) {
        // If can't open zip, return false
        return false;
    }
    // In zip we put sql file
    $zip->addFile($sql_path, basename($sql_path));
    $zip->close();

    if (function_exists('wp_delete_file')) {
        wp_delete_file($sql_path);
    } else {
        @unlink($sql_path);
    }

    return $zip_path;
}


/** Admin promo notice */
add_action('admin_notices', function () {
    if ( ! current_user_can('manage_options') ) {
        return;
    }

    // Ako je korisnik već dismissovao — ne prikazuj
    if ( get_user_meta( get_current_user_id(), 'dcxsitebackup_notice_dismissed', true ) ) {
        return;
    }

    // Vrati korisnika na istu stranicu posle dismiss-a
    $base = wp_get_referer();
    if ( ! $base ) {
        $base = admin_url();
    }

    $base = wp_validate_redirect( $base, admin_url() );

    $dismiss_url = wp_nonce_url(
        add_query_arg( 'dcxsitebackup-dismiss', '1', $base ),
        'dcxsitebackup_dismiss_notice',
        '_dcxsb_nonce'
    );

    echo '<div class="notice notice-info is-dismissible dcxsitebackup-notice" style="padding:15px;">';
    echo '<h2 style="margin-top:0;">' . esc_html__( '🙏 Thank You for Using DCX Site Backup!', 'dcx-site-backup' ) . '</h2>';
    echo '<p>';
    echo esc_html__( 'We appreciate you using our free WordPress backup plugin.', 'dcx-site-backup' ) . ' ';
    echo wp_kses(
        __(
            'Did you know our <a href="https://dcx.rs/en/vps-eu" target="_blank" rel="noopener"><strong>DCX.RS - EU VPS hosting</strong></a> scored <strong>3154</strong> in Geekbench 6 single-core benchmark — compared to the industry average of just <strong>750</strong>? That means your site can run over <strong>4× faster</strong> with us!',
            'dcx-site-backup'
        ),
        array(
            'a'      => array( 'href' => array(), 'target' => array(), 'rel' => array() ),
            'strong' => array(),
        )
    );
    echo '</p>';

    echo '<p>';
    echo '<a href="https://dcx.rs/en/vps-eu" class="button button-primary" target="_blank" rel="noopener">' . esc_html__( '🚀 Explore Our VPS Plans', 'dcx-site-backup' ) . '</a> ';
    echo '<a href="' . esc_url( $dismiss_url ) . '" class="button button-link">' . esc_html__( 'Dismiss', 'dcx-site-backup' ) . '</a>';
    echo '</p>';

    echo '</div>';
});

/** AJAX: delete backup */
add_action('wp_ajax_dcxsitebackup_wp_migration_delete_backup', function () {
    check_ajax_referer('dcxsitebackup_wp_migration', 'nonce');
    if (!current_user_can('manage_options')) {
        wp_send_json_error('forbidden');
    }

    // SANITIZE EARLY
    $raw = isset($_POST['file']) ? sanitize_text_field( wp_unslash($_POST['file']) ) : '';
    $filename = basename($raw);

    // VALIDATE
    if ($filename === '' || !preg_match('/^[A-Za-z0-9._-]+\z/', $filename)) {
        wp_send_json_error('Invalid file name');
    }

    // Allowed extensions (.zip ili .sql.gz)
    if (!preg_match('/\.(?:zip|sql\.gz)\z/i', $filename)) {
        wp_send_json_error('Invalid file name');
    }

    $backup_dir = DCXSITEBACKUP_DIR;
    $path       = $backup_dir . '/' . $filename;

    // Path traversal protection
    $real_backup = realpath($backup_dir);
    $real_path   = realpath($path);
    if (!$real_backup || !$real_path || strpos($real_path, $real_backup) !== 0) {
        wp_send_json_error('Invalid file path');
    }

    if (is_file($real_path)) {
        if (function_exists('wp_delete_file')) {
            wp_delete_file($real_path);
        }
        wp_send_json_success(['message' => 'File deleted']);
    }

    wp_send_json_error('File not found');
});


add_action('admin_init', function () {
    if ( ! is_user_logged_in() ) {
        return;
    }

    if ( empty($_GET['dcxsitebackup-dismiss']) ) {
        return;
    }

    // SANITIZE + VERIFY NONCE (GET)
    $nonce = isset($_GET['_dcxsb_nonce']) ? sanitize_text_field( wp_unslash($_GET['_dcxsb_nonce']) ) : '';
    if ( ! wp_verify_nonce( $nonce, 'dcxsitebackup_dismiss_notice' ) ) {
        return;
    }

    update_user_meta( get_current_user_id(), 'dcxsitebackup_notice_dismissed', true );

    $redirect = wp_get_referer();
    if ( ! $redirect ) {
        $redirect = admin_url();
    }
    $redirect = wp_validate_redirect( $redirect, admin_url() );

    wp_safe_redirect( remove_query_arg( array('dcxsitebackup-dismiss','_dcxsb_nonce'), $redirect ) );
    exit;
});

