<?php
/**
 * Plugin Name: Dashboard Menu Visibility
 * Plugin URI:  https://wordpress.org/plugins/dashboard-menu-visibility/
 * Description: Choose which admin menu sections are visible in the WordPress dashboard.
 * Version: 1.1.1
 * Author:      Studio JKN
 * Author URI:  https://studio.jakubkanna.com
 * Text Domain: dashboard-menu-visibility
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit;
}

const DASHMEVI_OPTION_KEY = 'dashmevi_visible_admin_menu_items';
const DASHMEVI_CATALOG_OPTION_KEY = 'dashmevi_known_admin_menu_items';
const DASHMEVI_SETTINGS_SLUG = 'dashmevi-dashboard-menu-visibility';

/**
 * Default visible menu items.
 *
 * @return string[]
 */
function dashmevi_default_visible_items()
{
    return array('edit.php', 'upload.php', 'edit.php?post_type=page', 'options-general.php');
}

/**
 * Core menu choices that may exist depending on site configuration.
 *
 * @return array<string, string>
 */
function dashmevi_core_menu_choices()
{
    return array(
        'index.php' => 'Dashboard',
        'edit.php' => 'Posts',
        'upload.php' => 'Media',
        'edit.php?post_type=page' => 'Pages',
        'edit-comments.php' => 'Comments',
        'themes.php' => 'Appearance',
        'plugins.php' => 'Plugins',
        'users.php' => 'Users',
        'tools.php' => 'Tools',
        'options-general.php' => 'Settings',
        'site-editor.php' => 'Editor',
        'link-manager.php' => 'Links',
        'my-sites.php' => 'My Sites',
        'sites.php' => 'Sites',
        'users.php?page=users' => 'Network Users',
        'themes.php?page=themes' => 'Network Themes',
        'plugins.php?page=plugins' => 'Network Plugins',
        'settings.php' => 'Network Settings',
    );
}

/**
 * Extract menu choices from menu entries.
 *
 * @param mixed $menu_entries Global admin menu value.
 *
 * @return array<string, string>
 */
function dashmevi_extract_menu_choices($menu_entries)
{
    $choices = array();

    if (!is_array($menu_entries)) {
        return $choices;
    }

    foreach ($menu_entries as $item) {
        if (!isset($item[2])) {
            continue;
        }

        $slug = (string) $item[2];

        if ($slug === '' || strpos($slug, 'separator') === 0) {
            continue;
        }

        if ($slug === DASHMEVI_SETTINGS_SLUG) {
            continue;
        }

        $label = isset($item[0]) ? wp_strip_all_tags((string) $item[0]) : $slug;
        if ($label === '') {
            $label = $slug;
        }

        $choices[$slug] = $label;
    }

    return $choices;
}

/**
 * Gets every known menu choice (core + discovered dynamic items).
 *
 * @return array<string, string>
 */
function dashmevi_get_menu_choices()
{
    global $menu;

    $core_choices = dashmevi_core_menu_choices();
    $saved_choices = get_option(DASHMEVI_CATALOG_OPTION_KEY, array());
    $live_choices = dashmevi_extract_menu_choices($menu);

    if (!is_array($saved_choices)) {
        $saved_choices = array();
    }

    $choices = array_merge($core_choices, $saved_choices, $live_choices);
    asort($choices, SORT_NATURAL | SORT_FLAG_CASE);

    return $choices;
}

/**
 * Persist discovered menu items so hidden items remain selectable.
 */
function dashmevi_capture_menu_catalog()
{
    global $menu;

    $saved_choices = get_option(DASHMEVI_CATALOG_OPTION_KEY, array());
    if (!is_array($saved_choices)) {
        $saved_choices = array();
    }

    $live_choices = dashmevi_extract_menu_choices($menu);
    $merged = array_merge($saved_choices, $live_choices);

    if ($merged !== $saved_choices) {
        update_option(DASHMEVI_CATALOG_OPTION_KEY, $merged, false);
    }
}
add_action('admin_menu', 'dashmevi_capture_menu_catalog', 100);

/**
 * Returns saved visible items with defaults.
 *
 * @return string[]
 */
function dashmevi_get_visible_items()
{
    $saved = get_option(DASHMEVI_OPTION_KEY, dashmevi_default_visible_items());

    if (!is_array($saved)) {
        return dashmevi_default_visible_items();
    }

    $sanitized = dashmevi_sanitize_visible_items($saved);

    if (empty($sanitized)) {
        return dashmevi_default_visible_items();
    }

    return $sanitized;
}

/**
 * Returns all allowed menu slugs.
 *
 * @return string[]
 */
function dashmevi_allowed_menu_slugs()
{
    $choices = dashmevi_get_menu_choices();
    $allowed = array_keys($choices);

    $allowed[] = DASHMEVI_SETTINGS_SLUG;

    return array_values(array_unique(array_merge($allowed, dashmevi_default_visible_items())));
}

/**
 * Sanitize setting value.
 *
 * @param mixed $value Raw value.
 * @return string[]
 */
function dashmevi_sanitize_visible_items($value)
{
    if (!is_array($value)) {
        return dashmevi_default_visible_items();
    }

    $allowed = dashmevi_allowed_menu_slugs();
    $sanitized = array();

    foreach ($value as $slug) {
        if (!is_scalar($slug)) {
            continue;
        }

        $clean_slug = sanitize_text_field((string) $slug);

        if ($clean_slug === '' || !in_array($clean_slug, $allowed, true)) {
            continue;
        }

        $sanitized[] = $clean_slug;
    }

    $sanitized = array_values(array_unique($sanitized));

    return empty($sanitized) ? dashmevi_default_visible_items() : $sanitized;
}

/**
 * Register plugin setting.
 */
function dashmevi_register_settings()
{
    register_setting(
        'dashmevi_settings_group',
        DASHMEVI_OPTION_KEY,
        array(
            'type' => 'array',
            'sanitize_callback' => 'dashmevi_sanitize_visible_items',
            'default' => dashmevi_default_visible_items(),
        )
    );
}
add_action('admin_init', 'dashmevi_register_settings');

/**
 * Add options page under Settings.
 */
function dashmevi_add_options_page()
{
    add_options_page(
        'Dashboard Menu Visibility',
        'Dashboard Menu Visibility',
        'manage_options',
        DASHMEVI_SETTINGS_SLUG,
        'dashmevi_render_settings_page'
    );
}
add_action('admin_menu', 'dashmevi_add_options_page');

/**
 * Add settings shortcut in plugins list.
 *
 * @param string[] $links Existing links.
 * @return string[]
 */
function dashmevi_plugin_action_links($links)
{
    $url = admin_url('options-general.php?page=' . DASHMEVI_SETTINGS_SLUG);
    $links[] = '<a href="' . esc_url($url) . '">Settings</a>';

    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'dashmevi_plugin_action_links');

/**
 * Hide unselected top-level admin menu pages.
 */
function dashmevi_filter_admin_menu()
{
    global $menu;

    if (!is_array($menu)) {
        return;
    }

    $visible = dashmevi_get_visible_items();

    foreach ($menu as $item) {
        if (!isset($item[2])) {
            continue;
        }

        $slug = (string) $item[2];

        if ($slug === '' || strpos($slug, 'separator') === 0) {
            continue;
        }

        if (in_array($slug, $visible, true)) {
            continue;
        }

        remove_menu_page($slug);
    }
}
add_action('admin_menu', 'dashmevi_filter_admin_menu', 999);

/**
 * Render settings page.
 */
function dashmevi_render_settings_page()
{
    if (!current_user_can('manage_options')) {
        return;
    }

    $visible = dashmevi_get_visible_items();
    $choices = dashmevi_get_menu_choices();
    ?>
    <div class="wrap">
        <h1>Dashboard Menu Visibility</h1>
        <p>Select which top-level dashboard menu items should remain visible.</p>

        <form method="post" action="options.php">
            <?php settings_fields('dashmevi_settings_group'); ?>

            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row">Visible menu items</th>
                        <td>
                            <?php if (empty($choices)) : ?>
                                <p>No menu items available.</p>
                            <?php else : ?>
                                <?php foreach ($choices as $slug => $label) : ?>
                                    <label style="display: block; margin-bottom: 6px;">
                                        <input
                                            type="checkbox"
                                            name="<?php echo esc_attr(DASHMEVI_OPTION_KEY); ?>[]"
                                            value="<?php echo esc_attr($slug); ?>"
                                            <?php checked(in_array($slug, $visible, true)); ?>
                                        />
                                        <?php echo esc_html($label . ' (' . $slug . ')'); ?>
                                    </label>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                </tbody>
            </table>

            <?php submit_button('Save Menu Visibility'); ?>
        </form>
    </div>
    <?php
}
