<?php
declare(strict_types=1);

namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }

use DailyTarot\Calendar\DayEntryTable;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\PostTypes;

final class Upgrade {

    // One-time migration marker.
    private const OPT_GYPSY_SPLIT_DONE = 'dtarot_upgrade_gypsy_split_2026_01_24';

    /**
     * Runs one-time upgrade migrations.
     *
     * Silent by design: no admin notices, no UI.
     */
    public static function maybeRun(): void {
        if (!function_exists('get_option') || !function_exists('update_option')) return;

        $done = (string)get_option(self::OPT_GYPSY_SPLIT_DONE, '');
        if ($done !== '') return;

        self::migrateGypsySplit();

        // Mark done even if partial; repeated attempts can cause unnecessary churn.
        update_option(self::OPT_GYPSY_SPLIT_DONE, gmdate('c'), false);
    }

    private static function migrateGypsySplit(): void {
        // 1) Decks: legacy "kipper" decks become "gypsy" and image keys are remapped.
        $gypsyDeckIds = self::migrateDecksKipperToGypsy();

        // 2) Meaning packs: legacy "kipper" packs become "gypsy" and meaning keys are remapped.
        $gypsyPackIds = self::migrateMeaningPacksKipperToGypsy();

        // 3) Defaults: if the default Kipper meaning pack now belongs to Gypsy, carry it over.
        if ($gypsyPackIds) {
            self::migrateDefaultMeaningPackKipperToGypsy();
        }

        // 4) Day entries: for migrated Gypsy decks, rewrite stored card_id kipper_XX -> gypsy_XX.
        if ($gypsyDeckIds) {
            self::migrateDayEntriesForGypsyDecks($gypsyDeckIds);
        }
    }

    /** @return array<int,int> */
    private static function migrateDecksKipperToGypsy(): array {
        if (!function_exists('get_posts') || !function_exists('get_post_meta')) return [];

        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
            'fields' => 'ids',
        ]);

        $gypsyIds = [];
        foreach ((array)$decks as $deckId) {
            $deckId = (int)$deckId;
            if ($deckId <= 0) continue;

            $systemRaw = (string)get_post_meta($deckId, '_dtarot_system', true);
            $system = Cards::normalizeSystem($systemRaw);

            $changed = false;

            // Old installs treated Tarot Gypsy as the Kipper system.
            if ($system === Cards::SYSTEM_KIPPER) {
                update_post_meta($deckId, '_dtarot_system', Cards::SYSTEM_GYPSY);
                $system = Cards::SYSTEM_GYPSY;
                $changed = true;
            }

            // Remap stored card image keys to match Gypsy IDs.
            if ($system === Cards::SYSTEM_GYPSY) {
                $imgs = get_post_meta($deckId, '_dtarot_cards', true);
                if (!is_array($imgs)) $imgs = [];

                $out = $imgs;
                foreach ($imgs as $key => $url) {
                    if (!is_string($key)) continue;
                    if (!is_string($url) || trim($url) === '') continue;

                    if (preg_match('/^kipper_(\d{2})$/', $key, $m)) {
                        $newKey = 'gypsy_' . $m[1];
                        if (!isset($out[$newKey]) || !is_string($out[$newKey]) || trim((string)$out[$newKey]) === '') {
                            $out[$newKey] = $url;
                        }
                        unset($out[$key]);
                        $changed = true;
                    }
                }

                if ($changed) {
                    update_post_meta($deckId, '_dtarot_cards', $out);
                }

                $gypsyIds[] = $deckId;
            }
        }

        return array_values(array_unique($gypsyIds));
    }

    /** @return array<int,int> */
    private static function migrateMeaningPacksKipperToGypsy(): array {
        if (!function_exists('get_posts') || !function_exists('get_post_meta')) return [];

        $packs = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
            'fields' => 'ids',
        ]);

        $gypsyIds = [];
        foreach ((array)$packs as $packId) {
            $packId = (int)$packId;
            if ($packId <= 0) continue;

            $systemRaw = (string)get_post_meta($packId, '_dtarot_system', true);
            $system = Cards::normalizeSystem($systemRaw);
            if ($system !== Cards::SYSTEM_KIPPER) continue;

            update_post_meta($packId, '_dtarot_system', Cards::SYSTEM_GYPSY);

            $meanings = get_post_meta($packId, '_dtarot_meanings', true);
            if (!is_array($meanings)) $meanings = [];

            $changed = false;
            $out = $meanings;
            foreach ($meanings as $key => $meaning) {
                if (!is_string($key)) continue;
                if (!is_array($meaning)) continue;

                if (preg_match('/^kipper_(\d{2})$/', $key, $m)) {
                    $newKey = 'gypsy_' . $m[1];
                    if (!isset($out[$newKey]) || !is_array($out[$newKey])) {
                        $out[$newKey] = $meaning;
                    }
                    unset($out[$key]);
                    $changed = true;
                }
            }

            if ($changed) {
                update_post_meta($packId, '_dtarot_meanings', $out);
            }

            $gypsyIds[] = $packId;
        }

        return array_values(array_unique($gypsyIds));
    }

    private static function migrateDefaultMeaningPackKipperToGypsy(): void {
        if (!function_exists('get_option') || !function_exists('update_option')) return;

        $raw = get_option('dtarot_default_meaning_packs_v1', []);
        if (!is_array($raw)) return;

        $kipperId = isset($raw[Cards::SYSTEM_KIPPER]) ? (int)$raw[Cards::SYSTEM_KIPPER] : 0;
        if ($kipperId <= 0) return;

        $ps = Cards::normalizeSystem((string)get_post_meta($kipperId, '_dtarot_system', true));
        if ($ps !== Cards::SYSTEM_GYPSY) return;

        if (empty($raw[Cards::SYSTEM_GYPSY])) {
            $raw[Cards::SYSTEM_GYPSY] = $kipperId;
        }

        // Let validation zero out kipper later; we keep it for traceability.
        update_option('dtarot_default_meaning_packs_v1', $raw, false);
    }

    /** @param array<int,int> $gypsyDeckIds */
    private static function migrateDayEntriesForGypsyDecks(array $gypsyDeckIds): void {
        if (!class_exists(DayEntryTable::class)) return;
        if (!function_exists('get_option')) return;

        $gypsyDeckIds = array_values(array_filter(array_map('intval', $gypsyDeckIds), static fn($v) => $v > 0));
        if (!$gypsyDeckIds) return;

        if (!DayEntryTable::exists()) return;

        global $wpdb;
        if (!isset($wpdb) || !is_object($wpdb)) return;

        $table = DayEntryTable::name();
        if ($table === '') return;

        $idsSql = implode(',', array_map('absint', $gypsyDeckIds));
        if ($idsSql === '') return;

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- deck ids are absint()'d and table is internal.
        $rows = $wpdb->get_results("SELECT date, card_id FROM {$table} WHERE deck_id IN ({$idsSql}) AND card_id LIKE 'kipper\\_%'", ARRAY_A);
        if (!is_array($rows) || !$rows) return;

        foreach ($rows as $row) {
            $date = isset($row['date']) && is_string($row['date']) ? $row['date'] : '';
            $cardId = isset($row['card_id']) && is_string($row['card_id']) ? $row['card_id'] : '';
            if ($date === '' || $cardId === '') continue;

            if (!preg_match('/^kipper_(\d{2})$/', $cardId, $m)) continue;
            $newId = 'gypsy_' . $m[1];

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- one-time migration.
            $wpdb->update(
                $table,
                ['card_id' => $newId],
                ['date' => $date],
                ['%s'],
                ['%s']
            );
        }
    }
}
