<?php
declare(strict_types=1);

namespace DailyTarot\Support;

if (!defined('ABSPATH')) { exit; }

/**
 * Defines free starter decks that can be imported from a remote ZIP URL.
 *
 * The default URLs are intended to point at the WP.org SVN assets folder.
 */
final class StarterDecks {

    /**
     * @return array<int,array{
     *   slug:string,
     *   title:string,
     *   system:string,
     *   cards:int,
     *   size_bytes:int,
     *   zip:string,
     *   back_preview:string,
     *   description:string
     * }>
     */
    public static function all(): array {
        $base = self::baseUrl();

        $decks = [
            [
                'slug' => 'tarot-basic',
                'title' => __('Tarot Basic Starter Deck','daily-tarot'),
                'system' => 'tarot',
                'cards' => 78,
                'size_bytes' => 4912207,
                'zip' => 'dtarot-deck-tarot-basic.zip',
                'back_preview' => 'starter-tarot-basic-back.avif',
                'description' => __('A ready-to-import Tarot deck to help you publish your first Card of the Day quickly.','daily-tarot'),
            ],
            [
                'slug' => 'kipper-basic',
                'title' => __('Kipper Basic Starter Deck','daily-tarot'),
                'system' => 'kipper_fin_de_siecle',
                'cards' => 39,
                'size_bytes' => 1865097,
                'zip' => 'dtarot-deck-kipper-basic.zip',
                'back_preview' => 'starter-kipper-basic-back.webp',
                'description' => __('A ready-to-import Kipper deck for fast setup and testing.','daily-tarot'),
            ],
        ];

        /**
         * Filter starter decks list.
         *
         * @param array $decks
         */
        return (array)apply_filters('dtarot_starter_decks', $decks);
    }

    public static function baseUrl(): string {
        $default = 'https://plugins.svn.wordpress.org/daily-tarot/assets/starter-packs/';
        $url = (string)apply_filters('dtarot_starter_decks_base_url', $default);
        $url = trim($url);
        if ($url === '') return $default;
        if (substr($url, -1) !== '/') $url .= '/';
        return $url;
    }

    /** @param array{zip:string} $deck */
    public static function zipUrl(array $deck): string {
        $zip = isset($deck['zip']) && is_string($deck['zip']) ? trim($deck['zip']) : '';
        if ($zip === '') return '';
        return self::baseUrl() . ltrim($zip, '/');
    }

    /** @param array{back_preview:string} $deck */
    public static function backPreviewUrl(array $deck): string {
        $f = isset($deck['back_preview']) && is_string($deck['back_preview']) ? trim($deck['back_preview']) : '';
        if ($f === '') return '';
        return self::baseUrl() . ltrim($f, '/');
    }

    /**
     * Strict allowlist for starter deck ZIP URLs.
     *
     * @return array<int,string>
     */
    public static function allowedZipUrls(): array {
        $out = [];
        foreach (self::all() as $deck) {
            if (!is_array($deck)) continue;
            $url = self::zipUrl($deck);
            if ($url !== '') $out[] = $url;
        }
        return $out;
    }

    public static function isAllowedZipUrl(string $zipUrl): bool {
        $zipUrl = trim($zipUrl);
        if ($zipUrl === '') return false;

        foreach (self::allowedZipUrls() as $allowed) {
            if (hash_equals($allowed, $zipUrl)) return true;
        }
        return false;
    }

    /**
     * Best-effort lookup of an installed deck post ID by manifest slug.
     */
    public static function findDeckIdBySlug(string $slug): int {
        $slug = sanitize_title($slug);
        if ($slug === '') return 0;

        if (!class_exists(PostTypes::class)) return 0;

        foreach (PostTypes::deckTypes() as $pt) {
            $p = get_page_by_path($slug, 'OBJECT', (string)$pt);
            if ($p && isset($p->ID)) {
                return (int)$p->ID;
            }
        }
        return 0;
    }
}
