<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }


final class SpreadScanner {

    private const OPT = 'dtarot_spread_scan_v1';

    public static function init(): void {
        add_action('save_post', [__CLASS__, 'scanPost'], 10, 2);
    }

    public static function scanPost(int $postId, $post): void {
        if ($postId <= 0 || !is_object($post)) return;
        if (wp_is_post_revision($postId)) return;

        $content = isset($post->post_content) ? (string)$post->post_content : '';
        self::scanContent($postId, $content);
    }

    public static function scanAll(): void {
        $types = get_post_types(['public' => true], 'names');
        if (!is_array($types)) $types = ['post','page'];

        $posts = get_posts([
            'post_type' => $types,
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
            'fields' => 'ids',
        ]);

        foreach ((array)$posts as $postId) {
            $postId = (int)$postId;
            if ($postId <= 0) continue;
            $p = get_post($postId);
            if (!$p) continue;
            self::scanContent($postId, (string)$p->post_content);
        }

        self::updateMeta([
            'last_scan' => time(),
        ]);
    }

    /**
     * @return array<int,array<string,mixed>>
     */
    public static function getAll(): array {
        $raw = get_option(self::OPT, []);
        return is_array($raw) ? $raw : [];
    }

    /**
     * @return array<int,array<string,mixed>>
     */
    public static function getForPost(int $postId): array {
        $all = self::getAll();
        return isset($all[$postId]) && is_array($all[$postId]) ? $all[$postId] : [];
    }

    private static function scanContent(int $postId, string $content): void {
        $shortcodes = self::findShortcodes($content);
        $all = self::getAll();
        $all[$postId] = $shortcodes;

        update_option(self::OPT, $all, false);
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    /**
     * @return array<int,array<string,mixed>>
     */
    private static function findShortcodes(string $content): array {
        if ($content === '') return [];

        $items = [];

        // Shortcodes: [dtarot_spread ...]
        $pattern = get_shortcode_regex(['dtarot_spread']);
        if ($pattern) {
            preg_match_all('/' . $pattern . '/s', $content, $matches, PREG_SET_ORDER | PREG_OFFSET_CAPTURE);
            foreach ($matches as $m) {
                $atts = shortcode_parse_atts($m[3][0]) ?: [];
                $items[] = [
                    'offset' => (int)$m[0][1],
                    'atts' => $atts,
                    'raw' => $m[0][0],
                ];
            }
        }

        // Blocks: <!-- wp:dtarot/spread {...} /--> or paired comment.
        preg_match_all('/<!--\s+wp:dtarot\/spread\s*(\{.*?\})?\s*(?:\/)?-->/s', $content, $bm, PREG_SET_ORDER | PREG_OFFSET_CAPTURE);
        foreach ($bm as $b) {
            $json = isset($b[1][0]) ? trim((string)$b[1][0]) : '';
            $atts = [];
            if ($json !== '') {
                $decoded = json_decode($json, true);
                if (is_array($decoded)) $atts = $decoded;
            }
            $items[] = [
                'offset' => (int)$b[0][1],
                'atts' => $atts,
                'raw' => $b[0][0],
            ];
        }

        if (!$items) return [];

        usort($items, static function(array $a, array $b): int {
            return ($a['offset'] ?? 0) <=> ($b['offset'] ?? 0);
        });

        $out = [];
        $idx = 0;
        foreach ($items as $item) {
            $out[] = [
                'index' => $idx,
                'atts' => isset($item['atts']) && is_array($item['atts']) ? $item['atts'] : [],
                'raw' => isset($item['raw']) ? (string)$item['raw'] : '',
            ];
            $idx++;
        }
        return $out;
    }

    private static function updateMeta(array $meta): void {
        $all = self::getAll();
        $all['_meta'] = $meta;
        update_option(self::OPT, $all, false);
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function lastScanTs(): int {
        $all = self::getAll();
        if (!isset($all['_meta']) || !is_array($all['_meta'])) return 0;
        $ts = isset($all['_meta']['last_scan']) ? (int)$all['_meta']['last_scan'] : 0;
        return $ts > 0 ? $ts : 0;
    }
}
