<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }


final class SpreadMeaningPacks {

    private const OPT = 'dtarot_spread_packs_v1';
    private const TOKENS_OPT = 'dtarot_spread_pack_tokens_v1';

    /**
     * @return array<string,array<string,mixed>>
     */
    public static function all(): array {
        $raw = get_option(self::OPT, []);
        return is_array($raw) ? $raw : [];
    }

    /**
     * @return array<string,mixed>|null
     */
    public static function get(string $id): ?array {
        $id = sanitize_key($id);
        if ($id === '') return null;
        $all = self::all();
        return isset($all[$id]) && is_array($all[$id]) ? $all[$id] : null;
    }

    public static function saveAll(array $packs): void {
        update_option(self::OPT, $packs, false);
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function ensureDefaultPackExists(): void {
        $all = self::all();
        if (isset($all['default']) && is_array($all['default'])) {
            $updated = false;
            $pack = $all['default'];
            $spreads = isset($pack['spreads']) && is_array($pack['spreads']) ? $pack['spreads'] : [];
            $presets = SpreadPresets::all();

            foreach ($presets as $id => $preset) {
                $slots = [];
                foreach ($preset['slots'] as $slot) {
                    $label = isset($slot['label']) && is_string($slot['label']) ? $slot['label'] : '';
                    $slots[] = [
                        'label' => $label,
                        'meaning' => self::defaultMeaningForLabel($label),
                    ];
                }

                if (!isset($spreads[$id]) || !is_array($spreads[$id])) {
                    $spreads[$id] = ['slots' => $slots];
                    $updated = true;
                    continue;
                }

                $existingSlots = isset($spreads[$id]['slots']) && is_array($spreads[$id]['slots']) ? $spreads[$id]['slots'] : [];
                if (count($existingSlots) < count($slots)) {
                    for ($i = count($existingSlots); $i < count($slots); $i++) {
                        $existingSlots[] = $slots[$i];
                    }
                    $spreads[$id]['slots'] = $existingSlots;
                    $updated = true;
                }
            }

            if ($updated) {
                $pack['spreads'] = $spreads;
                $all['default'] = $pack;
                self::saveAll($all);
            }
            return;
        }

        $all['default'] = self::defaultPack();
        self::saveAll($all);
    }

    /**
     * @return array<string,mixed>
     */
    private static function defaultPack(): array {
        $presets = SpreadPresets::all();

        $defaultSpreads = [];
        foreach ($presets as $id => $preset) {
            $slots = [];
            foreach ($preset['slots'] as $slot) {
                $label = isset($slot['label']) && is_string($slot['label']) ? $slot['label'] : '';
                $slots[] = [
                    'label' => $label,
                    'meaning' => self::defaultMeaningForLabel($label),
                ];
            }
            $defaultSpreads[$id] = [
                'slots' => $slots,
            ];
        }

        return [
            'id' => 'default',
            'title' => __('Default Spread Meanings','daily-tarot'),
            'version' => '1.0.0',
            'allow_edit' => false,
            'template' => false,
            'purchase_token' => '',
            'spreads' => $defaultSpreads,
        ];
    }

    private static function defaultMeaningForLabel(string $label): string {
        $label = strtolower(trim($label));
        if ($label === 'past') return __('Reflects the background and influences that shaped the situation.','daily-tarot');
        if ($label === 'present') return __('Shows the current energy and what is most active now.','daily-tarot');
        if ($label === 'future') return __('Indicates the likely direction or outcome if the path continues.','daily-tarot');
        if ($label === 'situation') return __('Describes the core situation and what is most important.','daily-tarot');
        if ($label === 'challenge') return __('Highlights the primary obstacle or tension to work through.','daily-tarot');
        if ($label === 'advice') return __('Offers guidance on how to move forward.','daily-tarot');
        if ($label === 'outcome') return __('Points to the probable result based on current actions.','daily-tarot');
        if ($label === 'center') return __('Represents the heart of the spread and the main theme.','daily-tarot');
        if ($label === 'above') return __('Shows higher influences, ideals, or conscious goals.','daily-tarot');
        if ($label === 'below') return __('Reveals underlying energies, foundations, or hidden factors.','daily-tarot');
        if ($label === 'left') return __('Represents the past or what is moving away.','daily-tarot');
        if ($label === 'right') return __('Represents what is developing or moving toward you.','daily-tarot');
        return __('Describes this position within the spread.','daily-tarot');
    }

    public static function markTokenUsed(string $token): void {
        $token = sanitize_text_field($token);
        if ($token === '') return;
        $used = get_option(self::TOKENS_OPT, []);
        if (!is_array($used)) $used = [];
        $used[$token] = time();
        update_option(self::TOKENS_OPT, $used, false);
    }

    public static function tokenUsed(string $token): bool {
        $token = sanitize_text_field($token);
        if ($token === '') return false;
        $used = get_option(self::TOKENS_OPT, []);
        if (!is_array($used)) return false;
        return isset($used[$token]);
    }
}
