<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



final class ShortcodeSettings {

    private const OPT = 'dtarot_shortcode_v1';

    private const KEY_CARD_POSITION = 'card_position';
    private const KEY_TEXT_MODE = 'text_mode';
    private const KEY_EXCERPT_WORDS = 'excerpt_words';
    private const KEY_READ_MORE_LABEL = 'read_more_label';
    private const KEY_LAYOUT_DEFAULT = 'layout_default';
    private const KEY_SHOW_SHARE = 'show_share';
    private const KEY_SHARE_STYLE = 'share_style';
    private const KEY_THEME_STYLE = 'theme_style';
    private const KEY_EMPTY_CARD_MODE = 'empty_card_mode';
    private const KEY_EMPTY_DECK_ID = 'empty_deck_id';
    private const KEY_EMPTY_TEXT = 'empty_text';

    /** @return array{card_position:string,text_mode:string,excerpt_words:int,read_more_label:string,layout_default:string,show_share:string,share_style:string,theme_style:string,empty_card_mode:string,empty_deck_id:int,empty_text:string} */
    public static function get(): array {
        $raw = get_option(self::OPT, null);
        if (!is_array($raw)) $raw = [];

        $cardPosition = isset($raw[self::KEY_CARD_POSITION]) && is_string($raw[self::KEY_CARD_POSITION]) ? sanitize_key($raw[self::KEY_CARD_POSITION]) : 'left';
        if (!in_array($cardPosition, ['left','right','center'], true)) {
            $cardPosition = 'left';
        }

        $textMode = isset($raw[self::KEY_TEXT_MODE]) && is_string($raw[self::KEY_TEXT_MODE]) ? sanitize_key($raw[self::KEY_TEXT_MODE]) : 'full';
        if (!in_array($textMode, ['full','excerpt','read_more','none'], true)) {
            $textMode = 'full';
        }

        $excerptWords = isset($raw[self::KEY_EXCERPT_WORDS]) ? (int)$raw[self::KEY_EXCERPT_WORDS] : 40;
        if ($excerptWords < 10) $excerptWords = 10;
        if ($excerptWords > 200) $excerptWords = 200;

        $readMoreLabel = isset($raw[self::KEY_READ_MORE_LABEL]) && is_string($raw[self::KEY_READ_MORE_LABEL]) ? trim((string)$raw[self::KEY_READ_MORE_LABEL]) : 'Read more';
        if ($readMoreLabel === '') $readMoreLabel = 'Read more';
        $readMoreLabel = sanitize_text_field($readMoreLabel);

        $layoutDefault = isset($raw[self::KEY_LAYOUT_DEFAULT]) && is_string($raw[self::KEY_LAYOUT_DEFAULT]) ? sanitize_key($raw[self::KEY_LAYOUT_DEFAULT]) : 'minimal';
        if (!in_array($layoutDefault, ['minimal','hero'], true)) {
            $layoutDefault = 'minimal';
        }

        $showShare = isset($raw[self::KEY_SHOW_SHARE]) && is_string($raw[self::KEY_SHOW_SHARE]) ? $raw[self::KEY_SHOW_SHARE] : '0';
        $showShare = ($showShare === '1') ? '1' : '0';

        $shareStyle = isset($raw[self::KEY_SHARE_STYLE]) && is_string($raw[self::KEY_SHARE_STYLE]) ? sanitize_key($raw[self::KEY_SHARE_STYLE]) : 'text';
        if (!in_array($shareStyle, ['text','arcana-sigils','crystal-aura','tarot-card'], true)) {
            $shareStyle = 'text';
        }

        $themeStyle = isset($raw[self::KEY_THEME_STYLE]) && is_string($raw[self::KEY_THEME_STYLE]) ? sanitize_key($raw[self::KEY_THEME_STYLE]) : 'minimal';
        if (!in_array($themeStyle, ['minimal','mystic','modern'], true)) {
            $themeStyle = 'minimal';
        }

        $emptyCardMode = isset($raw[self::KEY_EMPTY_CARD_MODE]) && is_string($raw[self::KEY_EMPTY_CARD_MODE]) ? sanitize_key($raw[self::KEY_EMPTY_CARD_MODE]) : 'random';
        if (!in_array($emptyCardMode, ['random','default'], true)) {
            $emptyCardMode = 'random';
        }

        $emptyDeckId = isset($raw[self::KEY_EMPTY_DECK_ID]) ? (int)$raw[self::KEY_EMPTY_DECK_ID] : 0;
        if ($emptyDeckId < 0) $emptyDeckId = 0;

        $emptyText = isset($raw[self::KEY_EMPTY_TEXT]) && is_string($raw[self::KEY_EMPTY_TEXT])
            ? trim((string)$raw[self::KEY_EMPTY_TEXT])
            : __('No card has been drawn for today. While you wait, here\'s a gentle reminder: "The future depends on what you do today."', 'daily-tarot');
        if ($emptyText === '') {
            $emptyText = __('No card has been drawn for today. While you wait, here\'s a gentle reminder: "The future depends on what you do today."', 'daily-tarot');
        }
        $emptyText = sanitize_textarea_field($emptyText);

        return [
            'card_position' => $cardPosition,
            'text_mode' => $textMode,
            'excerpt_words' => $excerptWords,
            'read_more_label' => $readMoreLabel,
            'layout_default' => $layoutDefault,
            'show_share' => $showShare,
            'share_style' => $shareStyle,
            'theme_style' => $themeStyle,
            'empty_card_mode' => $emptyCardMode,
            'empty_deck_id' => $emptyDeckId,
            'empty_text' => $emptyText,
        ];
    }

    public static function set(
        string $cardPosition,
        string $textMode,
        int $excerptWords,
        string $readMoreLabel,
        string $layoutDefault,
        string $showShare,
        string $shareStyle,
        string $themeStyle,
        string $emptyCardMode,
        int $emptyDeckId,
        string $emptyText
    ): void {
        $cardPosition = sanitize_key($cardPosition);
        if (!in_array($cardPosition, ['left','right','center'], true)) {
            $cardPosition = 'left';
        }

        $textMode = sanitize_key($textMode);
        if (!in_array($textMode, ['full','excerpt','read_more','none'], true)) {
            $textMode = 'full';
        }

        if ($excerptWords < 10) $excerptWords = 10;
        if ($excerptWords > 200) $excerptWords = 200;

        $readMoreLabel = trim($readMoreLabel);
        if ($readMoreLabel === '') $readMoreLabel = 'Read more';
        $readMoreLabel = sanitize_text_field($readMoreLabel);

        $layoutDefault = sanitize_key($layoutDefault);
        if (!in_array($layoutDefault, ['minimal','hero'], true)) {
            $layoutDefault = 'minimal';
        }

        $showShare = ($showShare === '1') ? '1' : '0';

        $shareStyle = sanitize_key($shareStyle);
        if (!in_array($shareStyle, ['text','arcana-sigils','crystal-aura','tarot-card'], true)) {
            $shareStyle = 'text';
        }

        $themeStyle = sanitize_key($themeStyle);
        if (!in_array($themeStyle, ['minimal','mystic','modern'], true)) {
            $themeStyle = 'minimal';
        }

        $emptyCardMode = sanitize_key($emptyCardMode);
        if (!in_array($emptyCardMode, ['random','default'], true)) {
            $emptyCardMode = 'random';
        }

        if ($emptyDeckId < 0) $emptyDeckId = 0;

        $emptyText = trim($emptyText);
        if ($emptyText === '') {
            $emptyText = __('No card has been drawn for today. While you wait, here\'s a gentle reminder: "The future depends on what you do today."', 'daily-tarot');
        }
        $emptyText = sanitize_textarea_field($emptyText);

        $raw = [
            self::KEY_CARD_POSITION => $cardPosition,
            self::KEY_TEXT_MODE => $textMode,
            self::KEY_EXCERPT_WORDS => $excerptWords,
            self::KEY_READ_MORE_LABEL => $readMoreLabel,
            self::KEY_LAYOUT_DEFAULT => $layoutDefault,
            self::KEY_SHOW_SHARE => $showShare,
            self::KEY_SHARE_STYLE => $shareStyle,
            self::KEY_THEME_STYLE => $themeStyle,
            self::KEY_EMPTY_CARD_MODE => $emptyCardMode,
            self::KEY_EMPTY_DECK_ID => $emptyDeckId,
            self::KEY_EMPTY_TEXT => $emptyText,
        ];

        update_option(self::OPT, $raw, false);

        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function handleSave(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_shortcode_save');

        $cardPosition = isset($_POST['dtarot_shortcode_card_position']) ? sanitize_key((string)wp_unslash($_POST['dtarot_shortcode_card_position'])) : 'left';
        $textMode = isset($_POST['dtarot_shortcode_text_mode']) ? sanitize_key((string)wp_unslash($_POST['dtarot_shortcode_text_mode'])) : 'full';
        $excerptWords = isset($_POST['dtarot_shortcode_excerpt_words']) ? absint(wp_unslash($_POST['dtarot_shortcode_excerpt_words'])) : 40;
        $readMoreLabel = isset($_POST['dtarot_shortcode_read_more_label']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_shortcode_read_more_label'])) : 'Read more';
        $layoutDefault = isset($_POST['dtarot_shortcode_layout_default']) ? sanitize_key((string)wp_unslash($_POST['dtarot_shortcode_layout_default'])) : 'minimal';
        $showShare = isset($_POST['dtarot_shortcode_show_share']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_shortcode_show_share'])) : '0';
        $shareStyle = isset($_POST['dtarot_shortcode_share_style']) ? sanitize_key((string)wp_unslash($_POST['dtarot_shortcode_share_style'])) : 'text';
        $themeStyle = isset($_POST['dtarot_shortcode_theme_style']) ? sanitize_key((string)wp_unslash($_POST['dtarot_shortcode_theme_style'])) : 'minimal';
        $emptyCardMode = isset($_POST['dtarot_shortcode_empty_card_mode']) ? sanitize_key((string)wp_unslash($_POST['dtarot_shortcode_empty_card_mode'])) : 'random';
        $emptyDeckId = isset($_POST['dtarot_shortcode_empty_deck_id']) ? absint(wp_unslash($_POST['dtarot_shortcode_empty_deck_id'])) : 0;
        $emptyText = isset($_POST['dtarot_shortcode_empty_text']) ? sanitize_textarea_field((string)wp_unslash($_POST['dtarot_shortcode_empty_text'])) : '';

        self::set($cardPosition, $textMode, $excerptWords, $readMoreLabel, $layoutDefault, $showShare, $shareStyle, $themeStyle, $emptyCardMode, $emptyDeckId, $emptyText);

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-settings',
            'tab' => 'shortcode',
            'msg' => 'shortcode_saved',
        ], admin_url('admin.php')));
        exit;
    }
}
