<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }


final class ShareImageSettings {

    private const OPT = 'dtarot_share_image_settings';

    /** @return array<string,mixed> */
    public static function defaults(): array {
        $siteName = function_exists('get_bloginfo') ? (string)get_bloginfo('name') : '';
        if ($siteName === '') $siteName = 'Daily Tarot';

        return [
            'enabled' => true,
            'theme' => [
                'default_style' => 'mystic',
                'styles' => [
                    'minimal' => [
                        'bg' => '#f7f7f2',
                        'accent' => '#1b1b1b',
                        'font' => 'Georgia, serif',
                        'card_frame' => '#e4e4dc',
                        'border' => '#d7d7cd',
                        'watermark_opacity' => 0.08,
                    ],
                    'mystic' => [
                        'bg' => '#f6efe4',
                        'accent' => '#6a3f23',
                        'font' => '"Palatino Linotype", "Book Antiqua", Palatino, serif',
                        'card_frame' => '#ead8c6',
                        'border' => '#dfc7b1',
                        'watermark_opacity' => 0.12,
                    ],
                    'modern' => [
                        'bg' => '#eef2f7',
                        'accent' => '#264a76',
                        'font' => '"Trebuchet MS", "Lucida Sans Unicode", "Lucida Grande", sans-serif',
                        'card_frame' => '#dde4ef',
                        'border' => '#ccd7e6',
                        'watermark_opacity' => 0.08,
                    ],
                ],
            ],
            'popup' => [
                'layout' => 'left_preview',
                'size' => 'md',
                'show_caption' => true,
            ],
            'branding' => [
                'mode' => 'optional',
                'text' => $siteName,
                'logo_url' => '',
            ],
            'qr' => [
                'enabled' => true,
                'allow_toggle' => true,
                'dest_rule' => 'daily_readable',
                'custom_url' => '',
            ],
            'text' => [
                'allow_toggle' => true,
                'max_chars' => 220,
            ],
            'export' => [
                'resolution' => 1080,
                'format' => 'png',
                'quality' => 92,
            ],
        ];
    }

    /** @return array<string,mixed> */
    public static function get(): array {
        $raw = get_option(self::OPT, null);
        if (!is_array($raw)) $raw = [];

        $defaults = self::defaults();
        $out = $defaults;

        $out['enabled'] = !empty($raw['enabled']);

        if (isset($raw['theme']) && is_array($raw['theme'])) {
            $theme = $raw['theme'];
            $defaultStyle = isset($theme['default_style']) && is_string($theme['default_style']) ? sanitize_key($theme['default_style']) : $defaults['theme']['default_style'];
            if (!isset($defaults['theme']['styles'][$defaultStyle])) {
                $defaultStyle = $defaults['theme']['default_style'];
            }
            $out['theme']['default_style'] = $defaultStyle;
        }

        if (isset($raw['popup']) && is_array($raw['popup'])) {
            $layout = isset($raw['popup']['layout']) && is_string($raw['popup']['layout']) ? sanitize_key($raw['popup']['layout']) : $defaults['popup']['layout'];
            if (!in_array($layout, ['left_preview','top_preview','full_preview'], true)) {
                $layout = $defaults['popup']['layout'];
            }
            $size = isset($raw['popup']['size']) && is_string($raw['popup']['size']) ? sanitize_key($raw['popup']['size']) : $defaults['popup']['size'];
            if (!in_array($size, ['sm','md','lg'], true)) {
                $size = $defaults['popup']['size'];
            }
            $out['popup']['layout'] = $layout;
            $out['popup']['size'] = $size;
            $out['popup']['show_caption'] = !empty($raw['popup']['show_caption']);
        }

        if (isset($raw['branding']) && is_array($raw['branding'])) {
            $mode = isset($raw['branding']['mode']) && is_string($raw['branding']['mode']) ? sanitize_key($raw['branding']['mode']) : $defaults['branding']['mode'];
            if (!in_array($mode, ['optional','required','disabled'], true)) {
                $mode = $defaults['branding']['mode'];
            }
            $out['branding']['mode'] = $mode;
            $text = isset($raw['branding']['text']) && is_string($raw['branding']['text']) ? trim((string)$raw['branding']['text']) : $defaults['branding']['text'];
            $out['branding']['text'] = sanitize_text_field($text);
            $logo = isset($raw['branding']['logo_url']) && is_string($raw['branding']['logo_url']) ? esc_url_raw(trim((string)$raw['branding']['logo_url'])) : '';
            $out['branding']['logo_url'] = $logo;
        }

        if (isset($raw['qr']) && is_array($raw['qr'])) {
            $enabled = !empty($raw['qr']['enabled']);
            $allowToggle = !empty($raw['qr']['allow_toggle']);
            $destRule = isset($raw['qr']['dest_rule']) && is_string($raw['qr']['dest_rule']) ? sanitize_key($raw['qr']['dest_rule']) : $defaults['qr']['dest_rule'];
            if (!in_array($destRule, ['daily_readable','current_page','custom'], true)) {
                $destRule = $defaults['qr']['dest_rule'];
            }
            $customUrl = isset($raw['qr']['custom_url']) && is_string($raw['qr']['custom_url']) ? esc_url_raw(trim((string)$raw['qr']['custom_url'])) : '';
            $out['qr']['enabled'] = $enabled;
            $out['qr']['allow_toggle'] = $allowToggle;
            $out['qr']['dest_rule'] = $destRule;
            $out['qr']['custom_url'] = $customUrl;
        }

        if (isset($raw['text']) && is_array($raw['text'])) {
            $out['text']['allow_toggle'] = !empty($raw['text']['allow_toggle']);
            $maxChars = isset($raw['text']['max_chars']) ? (int)$raw['text']['max_chars'] : (int)$defaults['text']['max_chars'];
            if ($maxChars < 40) $maxChars = 40;
            if ($maxChars > 500) $maxChars = 500;
            $out['text']['max_chars'] = $maxChars;
        }

        if (isset($raw['export']) && is_array($raw['export'])) {
            $res = isset($raw['export']['resolution']) ? (int)$raw['export']['resolution'] : (int)$defaults['export']['resolution'];
            if ($res < 640) $res = 640;
            if ($res > 2400) $res = 2400;
            $format = isset($raw['export']['format']) && is_string($raw['export']['format']) ? sanitize_key($raw['export']['format']) : $defaults['export']['format'];
            if (!in_array($format, ['png','webp'], true)) {
                $format = $defaults['export']['format'];
            }
            $quality = isset($raw['export']['quality']) ? (int)$raw['export']['quality'] : (int)$defaults['export']['quality'];
            if ($quality < 40) $quality = 40;
            if ($quality > 100) $quality = 100;
            $out['export']['resolution'] = $res;
            $out['export']['format'] = $format;
            $out['export']['quality'] = $quality;
        }

        return $out;
    }

    /** @param array<string,mixed> $settings */
    public static function set(array $settings): void {
        $defaults = self::defaults();
        $raw = $defaults;

        $raw['enabled'] = !empty($settings['enabled']);

        if (isset($settings['theme']) && is_array($settings['theme'])) {
            $defaultStyle = isset($settings['theme']['default_style']) && is_string($settings['theme']['default_style'])
                ? sanitize_key($settings['theme']['default_style'])
                : $defaults['theme']['default_style'];
            if (!isset($defaults['theme']['styles'][$defaultStyle])) {
                $defaultStyle = $defaults['theme']['default_style'];
            }
            $raw['theme']['default_style'] = $defaultStyle;
        }

        if (isset($settings['popup']) && is_array($settings['popup'])) {
            $layout = isset($settings['popup']['layout']) && is_string($settings['popup']['layout']) ? sanitize_key($settings['popup']['layout']) : $defaults['popup']['layout'];
            if (!in_array($layout, ['left_preview','top_preview','full_preview'], true)) {
                $layout = $defaults['popup']['layout'];
            }
            $size = isset($settings['popup']['size']) && is_string($settings['popup']['size']) ? sanitize_key($settings['popup']['size']) : $defaults['popup']['size'];
            if (!in_array($size, ['sm','md','lg'], true)) {
                $size = $defaults['popup']['size'];
            }
            $raw['popup']['layout'] = $layout;
            $raw['popup']['size'] = $size;
            $raw['popup']['show_caption'] = !empty($settings['popup']['show_caption']);
        }

        if (isset($settings['branding']) && is_array($settings['branding'])) {
            $mode = isset($settings['branding']['mode']) && is_string($settings['branding']['mode']) ? sanitize_key($settings['branding']['mode']) : $defaults['branding']['mode'];
            if (!in_array($mode, ['optional','required','disabled'], true)) {
                $mode = $defaults['branding']['mode'];
            }
            $raw['branding']['mode'] = $mode;
            $text = isset($settings['branding']['text']) && is_string($settings['branding']['text']) ? trim((string)$settings['branding']['text']) : $defaults['branding']['text'];
            $raw['branding']['text'] = sanitize_text_field($text);
            $logo = isset($settings['branding']['logo_url']) && is_string($settings['branding']['logo_url']) ? esc_url_raw(trim((string)$settings['branding']['logo_url'])) : '';
            $raw['branding']['logo_url'] = $logo;
        }

        if (isset($settings['qr']) && is_array($settings['qr'])) {
            $destRule = isset($settings['qr']['dest_rule']) && is_string($settings['qr']['dest_rule']) ? sanitize_key($settings['qr']['dest_rule']) : $defaults['qr']['dest_rule'];
            if (!in_array($destRule, ['daily_readable','current_page','custom'], true)) {
                $destRule = $defaults['qr']['dest_rule'];
            }
            $customUrl = isset($settings['qr']['custom_url']) && is_string($settings['qr']['custom_url']) ? esc_url_raw(trim((string)$settings['qr']['custom_url'])) : '';
            $raw['qr']['enabled'] = !empty($settings['qr']['enabled']);
            $raw['qr']['allow_toggle'] = !empty($settings['qr']['allow_toggle']);
            $raw['qr']['dest_rule'] = $destRule;
            $raw['qr']['custom_url'] = $customUrl;
        }

        if (isset($settings['text']) && is_array($settings['text'])) {
            $raw['text']['allow_toggle'] = !empty($settings['text']['allow_toggle']);
            $maxChars = isset($settings['text']['max_chars']) ? (int)$settings['text']['max_chars'] : (int)$defaults['text']['max_chars'];
            if ($maxChars < 40) $maxChars = 40;
            if ($maxChars > 500) $maxChars = 500;
            $raw['text']['max_chars'] = $maxChars;
        }

        if (isset($settings['export']) && is_array($settings['export'])) {
            $res = isset($settings['export']['resolution']) ? (int)$settings['export']['resolution'] : (int)$defaults['export']['resolution'];
            if ($res < 640) $res = 640;
            if ($res > 2400) $res = 2400;
            $format = isset($settings['export']['format']) && is_string($settings['export']['format']) ? sanitize_key($settings['export']['format']) : $defaults['export']['format'];
            if (!in_array($format, ['png','webp'], true)) {
                $format = $defaults['export']['format'];
            }
            $quality = isset($settings['export']['quality']) ? (int)$settings['export']['quality'] : (int)$defaults['export']['quality'];
            if ($quality < 40) $quality = 40;
            if ($quality > 100) $quality = 100;
            $raw['export']['resolution'] = $res;
            $raw['export']['format'] = $format;
            $raw['export']['quality'] = $quality;
        }

        update_option(self::OPT, $raw, false);

        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function handleSave(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_share_image_save');

        $settings = [
            'enabled' => isset($_POST['dtarot_share_image_enabled']),
            'theme' => [
                'default_style' => isset($_POST['dtarot_share_image_style']) ? sanitize_key((string)wp_unslash($_POST['dtarot_share_image_style'])) : 'mystic',
            ],
            'popup' => [
                'layout' => isset($_POST['dtarot_share_image_layout']) ? sanitize_key((string)wp_unslash($_POST['dtarot_share_image_layout'])) : 'left_preview',
                'size' => isset($_POST['dtarot_share_image_size']) ? sanitize_key((string)wp_unslash($_POST['dtarot_share_image_size'])) : 'md',
                'show_caption' => isset($_POST['dtarot_share_image_show_caption']),
            ],
            'branding' => [
                'mode' => isset($_POST['dtarot_share_image_branding_mode']) ? sanitize_key((string)wp_unslash($_POST['dtarot_share_image_branding_mode'])) : 'optional',
                'text' => isset($_POST['dtarot_share_image_branding_text']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_share_image_branding_text'])) : '',
                'logo_url' => isset($_POST['dtarot_share_image_branding_logo']) ? esc_url_raw((string)wp_unslash($_POST['dtarot_share_image_branding_logo'])) : '',
            ],
            'qr' => [
                'enabled' => isset($_POST['dtarot_share_image_qr_enabled']),
                'allow_toggle' => isset($_POST['dtarot_share_image_qr_toggle']),
                'dest_rule' => isset($_POST['dtarot_share_image_qr_dest']) ? sanitize_key((string)wp_unslash($_POST['dtarot_share_image_qr_dest'])) : 'daily_readable',
                'custom_url' => isset($_POST['dtarot_share_image_qr_custom']) ? esc_url_raw((string)wp_unslash($_POST['dtarot_share_image_qr_custom'])) : '',
            ],
            'text' => [
                'allow_toggle' => isset($_POST['dtarot_share_image_text_toggle']),
                'max_chars' => isset($_POST['dtarot_share_image_text_max']) ? absint(wp_unslash($_POST['dtarot_share_image_text_max'])) : 220,
            ],
            'export' => [
                'resolution' => isset($_POST['dtarot_share_image_resolution']) ? absint(wp_unslash($_POST['dtarot_share_image_resolution'])) : 1080,
                'format' => isset($_POST['dtarot_share_image_format']) ? sanitize_key((string)wp_unslash($_POST['dtarot_share_image_format'])) : 'png',
                'quality' => isset($_POST['dtarot_share_image_quality']) ? absint(wp_unslash($_POST['dtarot_share_image_quality'])) : 92,
            ],
        ];

        self::set($settings);

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-settings',
            'tab' => 'shortcode',
            'msg' => 'share_image_saved',
        ], admin_url('admin.php')));
        exit;
    }
}
