<?php
declare(strict_types=1);

namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }

final class ReviewPrompt {

    private const OPT_INSTALL_TS = 'dtarot_review_install_ts_v1';
    private const OPT_NEXT_TS = 'dtarot_review_next_ts_v1';
    private const OPT_DONE = 'dtarot_review_done_v1';

    public static function init(): void {
        add_action('admin_notices', [__CLASS__, 'maybeRender']);
    }

    public static function ensureInstallState(): void {
        $install = get_option(self::OPT_INSTALL_TS, 0);
        $install = is_numeric($install) ? (int)$install : 0;
        if ($install <= 0) {
            $install = time();
            update_option(self::OPT_INSTALL_TS, $install, false);
        }

        $next = get_option(self::OPT_NEXT_TS, 0);
        $next = is_numeric($next) ? (int)$next : 0;
        if ($next <= 0) {
            update_option(self::OPT_NEXT_TS, $install + 7 * DAY_IN_SECONDS, false);
        }
    }

    private static function isAllowedScreen(): bool {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only query arg used for screen detection.
        $page = isset($_GET['page']) ? sanitize_key((string)wp_unslash($_GET['page'])) : '';
        if ($page !== '' && str_starts_with($page, 'daily-tarot')) {
            return true;
        }

        if (function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if ($screen && is_object($screen)) {
                $id = (string)($screen->id ?? '');
                if ($id === 'plugins' || $id === 'plugins-network') {
                    return true;
                }
            }
        }

        return false;
    }

    public static function reviewUrl(): string {
        $default = 'https://wordpress.org/support/plugin/daily-tarot/reviews/';
        $url = (string)apply_filters('dtarot_review_url', $default);
        $url = trim($url);
        return $url !== '' ? $url : $default;
    }

    public static function maybeRender(): void {
        if (!is_admin() || !current_user_can('manage_options')) return;

        self::ensureInstallState();

        $done = (bool)get_option(self::OPT_DONE, false);
        if ($done) return;

        $next = get_option(self::OPT_NEXT_TS, 0);
        $next = is_numeric($next) ? (int)$next : 0;
        if ($next > 0 && time() < $next) return;

        if (!self::isAllowedScreen()) return;

        $base = admin_url('admin-post.php?action=dtarot_review_prompt');
        $rateUrl = wp_nonce_url(add_query_arg(['do' => 'rate'], $base), 'dtarot_review_prompt');
        $laterUrl = wp_nonce_url(add_query_arg(['do' => 'later'], $base), 'dtarot_review_prompt');
        $dismissUrl = wp_nonce_url(add_query_arg(['do' => 'dismiss'], $base), 'dtarot_review_prompt');

        echo '<div class="notice notice-info" style="padding:10px 12px;">';
        echo '<p style="margin:0 0 8px;"><strong>' . esc_html__('Enjoying Daily Tarot?','daily-tarot') . '</strong> ' . esc_html__('If you have a minute, a review on WordPress.org really helps.','daily-tarot') . '</p>';
        echo '<p style="margin:0; display:flex; gap:10px; align-items:center; flex-wrap:wrap;">';
        echo '<a class="button button-primary" href="' . esc_url($rateUrl) . '" target="_blank" rel="noopener noreferrer">' . esc_html__('Leave a review on WordPress.org','daily-tarot') . '</a>';
        echo '<a class="button" href="' . esc_url($laterUrl) . '">' . esc_html__('Remind me in a week','daily-tarot') . '</a>';
        echo '<a href="' . esc_url($dismissUrl) . '" style="text-decoration:none;">' . esc_html__('No thanks','daily-tarot') . '</a>';
        echo '</p>';
        echo '</div>';
    }

    public static function handleAction(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_review_prompt');

        self::ensureInstallState();

        $do = isset($_GET['do']) ? sanitize_key((string)wp_unslash($_GET['do'])) : '';
        if ($do === 'rate') {
            update_option(self::OPT_DONE, true, false);
            wp_safe_redirect(self::reviewUrl());
            exit;
        }

        if ($do === 'later') {
            update_option(self::OPT_NEXT_TS, time() + 7 * DAY_IN_SECONDS, false);
        } elseif ($do === 'dismiss') {
            update_option(self::OPT_DONE, true, false);
        }

        $to = wp_get_referer();
        if (!$to) $to = admin_url('admin.php?page=daily-tarot');
        wp_safe_redirect($to);
        exit;
    }
}
