<?php
declare(strict_types=1);

namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }

use DailyTarot\Registry\Cards;

final class RelatedLinks {

    private const OPT = 'dtarot_related_links_v1';

    /** @return array{enabled:string,post_type:string,base_path:string,map:array<string,array{type:string,id?:int,url?:string}>} */
    public static function get(): array {
        $raw = get_option(self::OPT, null);
        if (!is_array($raw)) $raw = [];

        $enabled = (!empty($raw['enabled']) && (string)$raw['enabled'] === '1') ? '1' : '0';
        $postType = isset($raw['post_type']) && is_string($raw['post_type']) ? sanitize_key($raw['post_type']) : 'page';
        if (!in_array($postType, ['page','post','any'], true)) $postType = 'page';

        $basePath = isset($raw['base_path']) && is_string($raw['base_path']) ? trim((string)$raw['base_path']) : '';
        $basePath = trim($basePath, " \t\n\r\0\x0B/");
        $basePath = sanitize_title_with_dashes($basePath);

        $map = [];
        if (isset($raw['map']) && is_array($raw['map'])) {
            foreach ($raw['map'] as $cardId => $row) {
                if (!is_string($cardId)) continue;
                $cardId = sanitize_key($cardId);
                if ($cardId === '') continue;
                if (!is_array($row)) continue;

                $type = isset($row['type']) && is_string($row['type']) ? sanitize_key($row['type']) : '';
                if ($type === 'post') {
                    $id = isset($row['id']) ? (int)$row['id'] : 0;
                    if ($id > 0) {
                        $map[$cardId] = ['type' => 'post', 'id' => $id];
                    }
                } elseif ($type === 'url') {
                    $url = isset($row['url']) && is_string($row['url']) ? esc_url_raw(trim($row['url'])) : '';
                    if ($url !== '') {
                        $map[$cardId] = ['type' => 'url', 'url' => $url];
                    }
                }
            }
        }

        return [
            'enabled' => $enabled,
            'post_type' => $postType,
            'base_path' => $basePath,
            'map' => $map,
        ];
    }

    public static function setBase(string $enabled, string $postType, string $basePath): void {
        $s = self::get();
        $enabled = ($enabled === '1') ? '1' : '0';
        $postType = sanitize_key($postType);
        if (!in_array($postType, ['page','post','any'], true)) $postType = 'page';

        $basePath = trim($basePath);
        $basePath = trim($basePath, " \t\n\r\0\x0B/");
        $basePath = sanitize_title_with_dashes($basePath);

        $s['enabled'] = $enabled;
        $s['post_type'] = $postType;
        $s['base_path'] = $basePath;

        update_option(self::OPT, $s, false);
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function setMappingPost(string $cardId, int $postId): void {
        $cardId = sanitize_key($cardId);
        if ($cardId === '' || $postId <= 0) return;

        $s = self::get();
        $s['map'][$cardId] = ['type' => 'post', 'id' => $postId];
        update_option(self::OPT, $s, false);
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function setMappingUrl(string $cardId, string $url): void {
        $cardId = sanitize_key($cardId);
        $url = esc_url_raw(trim($url));
        if ($cardId === '' || $url === '') return;

        $s = self::get();
        $s['map'][$cardId] = ['type' => 'url', 'url' => $url];
        update_option(self::OPT, $s, false);
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function clearMapping(string $cardId): void {
        $cardId = sanitize_key($cardId);
        if ($cardId === '') return;

        $s = self::get();
        if (isset($s['map'][$cardId])) {
            unset($s['map'][$cardId]);
            update_option(self::OPT, $s, false);
            if (function_exists('wp_cache_delete')) {
                wp_cache_delete(self::OPT, 'options');
                wp_cache_delete('alloptions', 'options');
            }
        }
    }

    /** @return array<int,array{id:int,title:string,type:string,url:string}> */
    public static function suggest(string $cardId): array {
        $cardId = sanitize_key($cardId);
        if ($cardId === '') return [];

        $cardName = Cards::name($cardId);
        $slug = sanitize_title($cardName);
        if ($slug === '') return [];

        $s = self::get();
        $types = self::postTypesForSetting($s['post_type'] ?? 'page');
        $base = isset($s['base_path']) && is_string($s['base_path']) ? (string)$s['base_path'] : '';

        $out = [];
        $seen = [];

        // 1) base_path/slug exact path (best for pages)
        if ($base !== '') {
            $p = get_page_by_path($base . '/' . $slug, OBJECT, $types);
            if ($p && isset($p->ID)) {
                $id = (int)$p->ID;
                if ($id > 0) {
                    $seen[$id] = true;
                    $out[] = self::postInfo($id);
                }
            }
        }

        // 2) exact slug
        $posts = get_posts([
            'post_type' => $types,
            'name' => $slug,
            'post_status' => 'publish',
            'numberposts' => 5,
            'no_found_rows' => true,
        ]);
        foreach ((array)$posts as $p) {
            if (!is_object($p) || !isset($p->ID)) continue;
            $id = (int)$p->ID;
            if ($id <= 0 || isset($seen[$id])) continue;
            $seen[$id] = true;
            $out[] = self::postInfo($id);
        }

        // 3) exact title match (fallback)
        $posts = get_posts([
            'post_type' => $types,
            'post_status' => 'publish',
            'numberposts' => 5,
            'no_found_rows' => true,
            's' => $cardName,
        ]);
        foreach ((array)$posts as $p) {
            if (!is_object($p) || !isset($p->ID)) continue;
            $id = (int)$p->ID;
            if ($id <= 0 || isset($seen[$id])) continue;
            $title = (string)get_the_title($id);
            if ($title !== $cardName) continue;
            $seen[$id] = true;
            $out[] = self::postInfo($id);
        }

        // Filter out empties and cap.
        $out = array_values(array_filter($out, function($row){
            return is_array($row) && !empty($row['id']) && !empty($row['url']);
        }));

        return array_slice($out, 0, 5);
    }

    /**
     * Suggest posts/pages to link based on the current day editor text.
     *
     * @return array<int,array{id:int,title:string,type:string,url:string,anchor:string,source:string}>
     */
    public static function suggestFromText(string $content, string $dailyText = '', int $excludePostId = 0, int $limit = 10): array {
        $contentText = trim((string)$content);
        $dailyText = trim((string)$dailyText);
        $allText = trim($contentText . "\n" . $dailyText);
        if ($allText === '') return [];

        $limit = (int)$limit;
        if ($limit <= 0) $limit = 10;
        if ($limit > 10) $limit = 10;

        $query = self::buildQueryFromText($allText);
        $items = $query !== '' ? self::searchPosts($query, 10, $excludePostId) : [];

        // Attach an anchor phrase per item (best-effort) so the UI can auto-link.
        $out = [];
        $usedAnchors = [];
        foreach ($items as $it) {
            if (!isset($it['id']) || !isset($it['url'])) continue;
            $title = isset($it['title']) ? (string)$it['title'] : '';
            $anchor = self::pickAnchorPhraseFromText($allText, $title, $usedAnchors);
            if ($anchor !== '') {
                $usedAnchors[mb_strtolower($anchor)] = true;
            }
            $source = self::pickAnchorSource($anchor, $contentText, $dailyText);
            $out[] = [
                'id' => (int)$it['id'],
                'title' => $title,
                'type' => isset($it['type']) ? (string)$it['type'] : '',
                'url' => (string)$it['url'],
                'anchor' => $anchor,
                'source' => $source,
            ];
        }

        // If nothing matched, provide 3 random posts/pages so the user can still link quickly.
        if (!$out) {
            $fallback = self::randomPosts(3, $excludePostId);
            $usedAnchors = [];
            foreach ($fallback as $it) {
                $title = (string)($it['title'] ?? '');
                $anchor = self::pickAnchorPhraseFromText($allText, $title, $usedAnchors);
                if ($anchor !== '') {
                    $usedAnchors[mb_strtolower($anchor)] = true;
                }
                $out[] = [
                    'id' => (int)$it['id'],
                    'title' => $title,
                    'type' => (string)($it['type'] ?? ''),
                    'url' => (string)($it['url'] ?? ''),
                    'anchor' => $anchor,
                    'source' => self::pickAnchorSource($anchor, $contentText, $dailyText),
                ];
            }
        }

        return array_slice($out, 0, $limit);
    }

    private static function buildQueryFromText(string $text): string {
        $text = strtolower(wp_strip_all_tags($text));
        if ($text === '') return '';

        // Keep it simple and robust: extract frequent-ish words and search by them.
        $text = preg_replace('/[^a-z0-9\s]+/u', ' ', $text) ?? $text;
        $parts = preg_split('/\s+/', $text, -1, PREG_SPLIT_NO_EMPTY);
        if (!$parts) return '';

        $stop = [
            'the','and','that','this','with','from','your','you','are','for','but','not','have','has','had','will','just','into','over','under','about',
            'today','daily','card','tarot','reading','meaning','meanings','feel','feels','really','very','more','most','when','what','where','which','who',
            'can','could','should','would','may','might','than','then','them','they','their','there','here','its','it\'s','our','ours','we','us','i','me',
        ];
        $stop = array_fill_keys($stop, true);

        $freq = [];
        foreach ($parts as $w) {
            $w = trim((string)$w);
            if ($w === '') continue;
            if (strlen($w) < 4) continue;
            if (isset($stop[$w])) continue;
            if (!isset($freq[$w])) $freq[$w] = 0;
            $freq[$w]++;
        }

        if (!$freq) return '';
        arsort($freq);

        $top = array_slice(array_keys($freq), 0, 8);
        $top = array_values(array_filter($top, function($w){
            return is_string($w) && $w !== '';
        }));
        return trim(implode(' ', $top));
    }

    /**
     * Pick a short (2–3 word) phrase from the text to use as an anchor.
     *
     * @param array<string,bool> $usedAnchors lowercased anchors already used
     */
    private static function pickDefaultAnchorFromText(string $text, array $usedAnchors = []): string {
        $text = trim(wp_strip_all_tags($text));
        if ($text === '') return '';

        $query = self::buildQueryFromText($text);
        if ($query === '') return '';

        $keywords = preg_split('/\s+/', $query, -1, PREG_SPLIT_NO_EMPTY);
        if (!$keywords) return '';

        $phrase = self::extractPhraseFromKeywords($text, $keywords, $usedAnchors);
        if ($phrase !== '') return $phrase;

        // Last resort: first keyword (avoid empty anchors).
        foreach ($keywords as $k) {
            $k = (string)$k;
            if ($k === '') continue;
            if (isset($usedAnchors[mb_strtolower($k)])) continue;
            return $k;
        }
        return '';
    }

    private static function pickAnchorSource(string $anchor, string $contentText, string $dailyText): string {
        $anchor = trim($anchor);
        if ($anchor === '') return 'either';
        $a = mb_strtolower($anchor);
        $inContent = $contentText !== '' && mb_strpos(mb_strtolower($contentText), $a) !== false;
        $inDaily = $dailyText !== '' && mb_strpos(mb_strtolower($dailyText), $a) !== false;
        if ($inContent && !$inDaily) return 'content';
        if ($inDaily && !$inContent) return 'daily_text';
        if ($inDaily && $inContent) return 'either';
        return 'either';
    }


    /**
     * Prefer an anchor that is a short phrase (2–3 words), is present in the user's text,
     * and is not reused across suggestions.
     *
     * @param array<string,bool> $usedAnchors lowercased anchors already used
     */
    private static function pickAnchorPhraseFromText(string $text, string $postTitle, array $usedAnchors = []): string {
        $text = trim(wp_strip_all_tags($text));
        $postTitle = trim($postTitle);
        if ($text === '') return '';

        // 1) If the full title is short (<= 3 words) and appears, use it.
        if ($postTitle !== '') {
            $titleWords = self::splitWords($postTitle);
            if (count($titleWords) > 0 && count($titleWords) <= 3) {
                $m = self::findPhraseInText($text, implode(' ', $titleWords));
                if ($m !== '' && !isset($usedAnchors[mb_strtolower($m)])) {
                    return $m;
                }
            }

            // 2) Try 3-grams then 2-grams from the title.
            $cands = self::titleNgrams($postTitle, 3);
            foreach ($cands as $cand) {
                if ($cand === '') continue;
                $m = self::findPhraseInText($text, $cand);
                if ($m !== '' && !isset($usedAnchors[mb_strtolower($m)])) {
                    return $m;
                }
            }
            $cands = self::titleNgrams($postTitle, 2);
            foreach ($cands as $cand) {
                if ($cand === '') continue;
                $m = self::findPhraseInText($text, $cand);
                if ($m !== '' && !isset($usedAnchors[mb_strtolower($m)])) {
                    return $m;
                }
            }
        }

        // 3) Fallback: keyword-derived phrase from the text (2–3 words), unique.
        return self::pickDefaultAnchorFromText($text, $usedAnchors);
    }

    /** @return array<int,string> */
    private static function splitWords(string $s): array {
        $s = trim($s);
        if ($s === '') return [];
        $s = preg_replace('/[^\p{L}\p{N}]+/u', ' ', $s) ?? $s;
        $s = trim(preg_replace('/\s+/u', ' ', $s) ?? $s);
        if ($s === '') return [];
        $parts = preg_split('/\s+/u', $s, -1, PREG_SPLIT_NO_EMPTY);
        return $parts ? array_values(array_map('strval', $parts)) : [];
    }

    /** @return array<int,string> */
    private static function titleNgrams(string $title, int $n): array {
        $n = max(1, min(3, (int)$n));
        $words = self::splitWords($title);
        if (count($words) < $n) return [];

        $stop = [
            'the','and','or','a','an','to','of','in','on','for','with','at','by','from','as','is','are','was','were','be','been','being','your','you','we','our','us',
        ];
        $stop = array_fill_keys($stop, true);

        $out = [];
        $seen = [];
        $max = count($words) - $n;
        for ($i = 0; $i <= $max; $i++) {
            $slice = array_slice($words, $i, $n);
            $phrase = trim(implode(' ', $slice));
            if ($phrase === '') continue;

            // Avoid phrases that are all stopwords.
            $nonStop = 0;
            foreach ($slice as $w) {
                $lw = strtolower((string)$w);
                if ($lw !== '' && !isset($stop[$lw])) $nonStop++;
            }
            if ($nonStop === 0) continue;

            // Avoid ultra-short phrases like "of a".
            $chars = preg_replace('/\s+/u', '', $phrase) ?? $phrase;
            if (mb_strlen($chars) < 6) continue;

            $key = mb_strtolower($phrase);
            if (isset($seen[$key])) continue;
            $seen[$key] = true;
            $out[] = $phrase;
        }
        return $out;
    }

    private static function findPhraseInText(string $text, string $phrase): string {
        $text = trim($text);
        $phrase = trim($phrase);
        if ($text === '' || $phrase === '') return '';

        $re = '/\b' . preg_quote($phrase, '/') . '\b/iu';
        if (preg_match($re, $text, $m) === 1 && isset($m[0]) && is_string($m[0]) && $m[0] !== '') {
            return (string)$m[0];
        }
        return '';
    }

    /**
     * Pull a 2–3 word phrase from the text around one of the keywords.
     *
     * @param array<int,string> $keywords
     * @param array<string,bool> $usedAnchors
     */
    private static function extractPhraseFromKeywords(string $text, array $keywords, array $usedAnchors = []): string {
        $text = trim($text);
        if ($text === '' || !$keywords) return '';

        foreach ($keywords as $k) {
            $k = trim((string)$k);
            if ($k === '') continue;

            // Prefer a phrase starting at the keyword (keyword + next 1-2 words).
            $re1 = '/\b' . preg_quote($k, '/') . '\b(?:\s+[\p{L}\p{N}][\p{L}\p{N}\-’\']*){1,2}/iu';
            if (preg_match($re1, $text, $m) === 1 && isset($m[0]) && is_string($m[0])) {
                $cand = trim((string)$m[0]);
                if ($cand !== '' && !isset($usedAnchors[mb_strtolower($cand)])) {
                    return $cand;
                }
            }

            // Or a phrase ending at the keyword (prev 1-2 words + keyword).
            $re2 = '/(?:[\p{L}\p{N}][\p{L}\p{N}\-’\']*\s+){1,2}\b' . preg_quote($k, '/') . '\b/iu';
            if (preg_match($re2, $text, $m) === 1 && isset($m[0]) && is_string($m[0])) {
                $cand = trim((string)$m[0]);
                if ($cand !== '' && !isset($usedAnchors[mb_strtolower($cand)])) {
                    return $cand;
                }
            }
        }

        return '';
    }

    /** @return array<int,array{id:int,title:string,type:string,url:string}> */
    private static function searchPosts(string $query, int $limit = 10, int $excludePostId = 0): array {
        $query = trim($query);
        if ($query === '') return [];

        $s = self::get();
        $types = self::postTypesForSetting($s['post_type'] ?? 'page');

        $limit = max(1, min(50, (int)$limit));
        $args = [
            'post_type' => $types,
            'post_status' => 'publish',
            // Avoid exclusionary params (post__not_in/exclude); we filter in PHP below.
            'numberposts' => $excludePostId > 0 ? min(50, $limit + 5) : $limit,
            'no_found_rows' => true,
            's' => $query,
        ];

        $posts = get_posts($args);

        $items = [];
        foreach ((array)$posts as $p) {
            if (!is_object($p) || !isset($p->ID)) continue;
            $id = (int)$p->ID;
            if ($id <= 0) continue;
            if ($excludePostId > 0 && $id === (int)$excludePostId) continue;
            $url = get_permalink($id);
            $url = is_string($url) ? (string)$url : '';
            if ($url === '') continue;
            $items[] = [
                'id' => $id,
                'title' => (string)get_the_title($id),
                'type' => (string)get_post_type($id),
                'url' => $url,
            ];
            if (count($items) >= $limit) break;
        }
        return $items;
    }

    /** @return array<int,array{id:int,title:string,type:string,url:string}> */
    private static function randomPosts(int $limit = 3, int $excludePostId = 0): array {
        $s = self::get();
        $types = self::postTypesForSetting($s['post_type'] ?? 'page');

        $limit = max(1, min(10, (int)$limit));
        $args = [
            'post_type' => $types,
            'post_status' => 'publish',
            // Avoid exclusionary params (post__not_in/exclude); we filter in PHP below.
            'numberposts' => $excludePostId > 0 ? min(10, $limit + 3) : $limit,
            'no_found_rows' => true,
            'orderby' => 'rand',
        ];

        $posts = get_posts($args);

        $items = [];
        foreach ((array)$posts as $p) {
            if (!is_object($p) || !isset($p->ID)) continue;
            $id = (int)$p->ID;
            if ($id <= 0) continue;
            if ($excludePostId > 0 && $id === (int)$excludePostId) continue;
            $url = get_permalink($id);
            $url = is_string($url) ? (string)$url : '';
            if ($url === '') continue;
            $items[] = [
                'id' => $id,
                'title' => (string)get_the_title($id),
                'type' => (string)get_post_type($id),
                'url' => $url,
            ];
            if (count($items) >= $limit) break;
        }
        return $items;
    }

    /** @return array{id:int,title:string,type:string,url:string} */
    private static function postInfo(int $postId): array {
        $postId = (int)$postId;
        $url = get_permalink($postId);
        $url = is_string($url) ? (string)$url : '';
        $type = get_post_type($postId);
        $type = is_string($type) ? (string)$type : '';
        return [
            'id' => $postId,
            'title' => (string)get_the_title($postId),
            'type' => $type,
            'url' => $url,
        ];
    }

    /** @return array<int,string> */
    private static function postTypesForSetting(string $setting): array {
        $setting = sanitize_key($setting);
        if ($setting === 'any') return ['page','post'];
        if ($setting === 'post') return ['post'];
        return ['page'];
    }

    public static function selectedLink(string $cardId): array {
        $cardId = sanitize_key($cardId);
        $s = self::get();
        $row = isset($s['map'][$cardId]) && is_array($s['map'][$cardId]) ? $s['map'][$cardId] : [];

        $type = isset($row['type']) && is_string($row['type']) ? sanitize_key($row['type']) : '';
        if ($type === 'post') {
            $id = isset($row['id']) ? (int)$row['id'] : 0;
            if ($id > 0) {
                $url = get_permalink($id);
                $url = is_string($url) ? (string)$url : '';
                return [
                    'type' => 'post',
                    'id' => $id,
                    'title' => (string)get_the_title($id),
                    'url' => $url,
                ];
            }
        }

        if ($type === 'url') {
            $url = isset($row['url']) && is_string($row['url']) ? esc_url_raw(trim($row['url'])) : '';
            if ($url !== '') {
                return [
                    'type' => 'url',
                    'id' => 0,
                    'title' => $url,
                    'url' => $url,
                ];
            }
        }

        return ['type' => 'none', 'id' => 0, 'title' => '', 'url' => ''];
    }

    public static function resolvedUrl(string $cardId): string {
        $s = self::get();
        if (!isset($s['enabled']) || (string)$s['enabled'] !== '1') return '';

        $sel = self::selectedLink($cardId);
        if (!empty($sel['url']) && is_string($sel['url'])) {
            return (string)$sel['url'];
        }

        $sugs = self::suggest($cardId);
        if (!empty($sugs[0]['url'])) {
            return (string)$sugs[0]['url'];
        }

        return '';
    }

    public static function renderBlock(string $cardId, string $cardName): string {
        $url = self::resolvedUrl($cardId);
        if ($url === '') return '';

        $label = sprintf(
            /* translators: %s is a card name */
            __('Learn more about %s','daily-tarot'),
            $cardName
        );

        return '<div class="dtarot-related-links">'
            . '<a class="dtarot-related-link" href="' . esc_url($url) . '">' . esc_html($label) . '</a>'
            . '</div>';
    }

    public static function handleSave(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_related_links_save');

        $enabled = isset($_POST['dtarot_related_links_enabled']) ? '1' : '0';
        $postType = isset($_POST['dtarot_related_links_post_type']) ? sanitize_key((string)wp_unslash($_POST['dtarot_related_links_post_type'])) : 'page';
        $basePath = isset($_POST['dtarot_related_links_base_path']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_related_links_base_path'])) : '';

        self::setBase($enabled, $postType, $basePath);

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-settings',
            'tab' => 'shortcode',
            'msg' => 'related_links_saved',
        ], admin_url('admin.php')));
        exit;
    }
}
