<?php
declare(strict_types=1);

namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }

final class RateLimit {

    public static function defaultRetryAfterSeconds(int $windowSeconds): int {
        $windowSeconds = max(1, (int)$windowSeconds);
        return min(3600, $windowSeconds);
    }

    /**
     * @param string $action unique action key (e.g. 'share_track')
     * @param int $limit max requests in window
     * @param int $windowSeconds window size
     */
    public static function hit(string $action, int $limit, int $windowSeconds): bool {
        $action = strtolower(trim($action));
        if ($action === '') return true;

        $enabled = (bool)apply_filters('dtarot_rate_limit_enabled', true, $action);
        if (!$enabled) return true;

        $limit = max(1, (int)$limit);
        $windowSeconds = max(1, (int)$windowSeconds);

        $key = self::key($action, $windowSeconds);
        if ($key === '') return true;

        $count = (int)get_transient($key);
        $count++;
        set_transient($key, $count, $windowSeconds);

        return $count <= $limit;
    }

    private static function key(string $action, int $windowSeconds): string {
        $ip = isset($_SERVER['REMOTE_ADDR'])
            ? sanitize_text_field((string)wp_unslash($_SERVER['REMOTE_ADDR']))
            : '';
        $ua = isset($_SERVER['HTTP_USER_AGENT'])
            ? sanitize_text_field((string)wp_unslash($_SERVER['HTTP_USER_AGENT']))
            : '';

        /**
         * Filter: dtarot_rate_limit_ip
         * Allows use of trusted proxy headers (X-Forwarded-For) if desired.
         */
        $ip = (string)apply_filters('dtarot_rate_limit_ip', $ip, $action);

        $userPart = '';
        if (function_exists('is_user_logged_in') && is_user_logged_in() && function_exists('get_current_user_id')) {
            $uid = (int)get_current_user_id();
            if ($uid > 0) {
                $userPart = 'user:' . $uid;
            }
        }

        $raw = $action . '|' . $windowSeconds . '|' . $userPart . '|' . $ip . '|' . substr($ua, 0, 80);

        /**
         * Filter: dtarot_rate_limit_identifier
         * Alternate to dtarot_rate_limit_key; return a stable identifier string.
         */
        $ident = (string)apply_filters('dtarot_rate_limit_identifier', '', $action);
        if (trim($ident) !== '') {
            $raw = $action . '|' . $windowSeconds . '|' . trim($ident);
        }

        /**
         * Filter: dtarot_rate_limit_key
         * Allows hosts to customize the identifier (e.g., include user id).
         */
        $raw = (string)apply_filters('dtarot_rate_limit_key', $raw, $action);

        $raw = trim($raw);
        if ($raw === '') return '';

        return 'dtarot_rl_' . md5($raw);
    }
}
