<?php
declare(strict_types=1);

namespace DailyTarot\Support;

if (!defined('ABSPATH')) { exit; }

/**
 * Central registry for CPT keys.
 *
 * We use a plugin-specific prefix (dtarot_) as the canonical post type key.
 * Legacy dt_* keys are still supported for backward compatibility and are
 * migrated automatically.
 */
final class PostTypes {

    public const DECK = 'dtarot_deck';
    public const MEANING_PACK = 'dtarot_meaning_pack';
    public const READING_TYPE = 'dtarot_reading_type';
    public const BOOKING = 'dtarot_booking';

    // Legacy (pre-prefix-unification) keys.
    public const LEGACY_DECK = 'dt_deck';
    public const LEGACY_MEANING_PACK = 'dt_meaning_pack';
    public const LEGACY_READING_TYPE = 'dt_reading_type';
    public const LEGACY_BOOKING = 'dt_booking';

    private const MIGRATED_OPT = 'dtarot_post_types_migrated_v1';

    /** @return list<string> */
    public static function deckTypes(): array { return [self::DECK, self::LEGACY_DECK]; }

    /** @return list<string> */
    public static function meaningPackTypes(): array { return [self::MEANING_PACK, self::LEGACY_MEANING_PACK]; }

    /** @return list<string> */
    public static function readingTypeTypes(): array { return [self::READING_TYPE, self::LEGACY_READING_TYPE]; }

    /** @return list<string> */
    public static function bookingTypes(): array { return [self::BOOKING, self::LEGACY_BOOKING]; }

    public static function isDeckType(string $postType): bool { return in_array($postType, self::deckTypes(), true); }
    public static function isMeaningPackType(string $postType): bool { return in_array($postType, self::meaningPackTypes(), true); }
    public static function isReadingTypeType(string $postType): bool { return in_array($postType, self::readingTypeTypes(), true); }
    public static function isBookingType(string $postType): bool { return in_array($postType, self::bookingTypes(), true); }

    /**
     * Migrates legacy CPT keys to the canonical dtarot_ keys.
     *
     * Safe to call multiple times.
     */
    public static function migrateLegacyPostTypes(): void {
        // Avoid running DB migrations on anonymous frontend requests.
        // Admin screens, cron, and WP-CLI are appropriate contexts.
        $isCli = defined('WP_CLI') && WP_CLI;
        if (!$isCli && !is_admin() && !(function_exists('wp_doing_cron') && wp_doing_cron())) {
            return;
        }

        if ((string)get_option(self::MIGRATED_OPT, '') === '1') return;

        global $wpdb;
        if (!isset($wpdb) || !is_object($wpdb)) {
            update_option(self::MIGRATED_OPT, '1', false);
            return;
        }

        $map = [
            self::LEGACY_DECK => self::DECK,
            self::LEGACY_MEANING_PACK => self::MEANING_PACK,
            self::LEGACY_READING_TYPE => self::READING_TYPE,
            self::LEGACY_BOOKING => self::BOOKING,
        ];

        foreach ($map as $from => $to) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->query(
                $wpdb->prepare(
                    "UPDATE {$wpdb->posts} SET post_type = %s WHERE post_type = %s",
                    $to,
                    $from
                )
            );
        }

        update_option(self::MIGRATED_OPT, '1', false);
    }
}
