<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



final class OnlineVisitors {

    private const OPT = 'dtarot_online_v1';

    /**
     * Record a lightweight "online" ping for the current request.
     */
    public static function record(): void {
        if (is_admin()) return;

        $token = self::visitorToken();
        if ($token === '') return;

        $now = time();
        $window = self::windowSeconds();

        $all = get_option(self::OPT, []);
        if (!is_array($all)) $all = [];

        $all = self::prune($all, $now, $window);
        $all[$token] = $now;

        self::trimMax($all);

        update_option(self::OPT, $all, false);

        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function countOnline(): int {
        $now = time();
        $window = self::windowSeconds();

        $all = get_option(self::OPT, []);
        if (!is_array($all)) $all = [];

        $before = count($all);
        $all = self::prune($all, $now, $window);
        $after = count($all);

        if ($after !== $before) {
            update_option(self::OPT, $all, false);
            if (function_exists('wp_cache_delete')) {
                wp_cache_delete(self::OPT, 'options');
                wp_cache_delete('alloptions', 'options');
            }
        }

        return $after;
    }

    public static function windowSeconds(): int {
        $window = (int) apply_filters('dtarot_online_window', 300);
        if ($window < 60) $window = 60;
        if ($window > 3600) $window = 3600;
        return $window;
    }

    private static function visitorToken(): string {
        $ip = '';
        if (isset($_SERVER['REMOTE_ADDR'])) {
            $ip = sanitize_text_field((string)wp_unslash($_SERVER['REMOTE_ADDR']));
            $ip = trim($ip);
            if ($ip !== '' && function_exists('filter_var') && filter_var($ip, FILTER_VALIDATE_IP) === false) {
                $ip = '';
            }
        }
        if ($ip === '') return '';

        $ua = '';
        if (isset($_SERVER['HTTP_USER_AGENT'])) {
            $ua = sanitize_text_field((string)wp_unslash($_SERVER['HTTP_USER_AGENT']));
            $ua = trim($ua);
        }
        $raw = $ip . '|' . $ua;

        return (string) wp_hash($raw, 'dtarot_online');
    }

    /** @param array<string,int> $all */
    private static function prune(array $all, int $now, int $window): array {
        $cutoff = $now - $window;
        foreach ($all as $key => $ts) {
            $t = (int)$ts;
            if ($t <= 0 || $t < $cutoff) {
                unset($all[$key]);
            }
        }
        return $all;
    }

    /** @param array<string,int> $all */
    private static function trimMax(array &$all): void {
        $max = (int) apply_filters('dtarot_online_max', 1500);
        if ($max < 100) $max = 100;
        if (count($all) <= $max) return;

        asort($all);
        $remove = max(0, count($all) - $max);
        if ($remove <= 0) return;

        $keys = array_keys($all);
        for ($i = 0; $i < $remove; $i++) {
            $k = $keys[$i] ?? '';
            if (is_string($k) && $k !== '') {
                unset($all[$k]);
            }
        }
    }
}
