<?php
declare(strict_types=1);

namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }

final class Log {

    private const OPT = 'dtarot_log_v1';

    /**
     * @return array<int,array{time:string,level:string,event:string,message:string,context:array<string,mixed>}>
     */
    public static function all(): array {
        $raw = get_option(self::OPT, []);
        if (!is_array($raw)) return [];
        return $raw;
    }

    /**
     * @param array<string,mixed> $context
     */
    public static function add(string $level, string $event, string $message, array $context = []): void {
        $level = strtolower(trim($level));
        if ($level === '') $level = 'info';
        $event = sanitize_key($event);
        if ($event === '') $event = 'event';
        $message = trim($message);

        $max = (int)apply_filters('dtarot_log_max_entries', 200);
        if ($max < 50) $max = 50;
        if ($max > 2000) $max = 2000;

        // Avoid storing huge blobs.
        foreach ($context as $k => $v) {
            if (is_string($v) && strlen($v) > 2000) {
                $context[$k] = substr($v, 0, 2000) . '…';
            }
        }

        $items = self::all();
        $items[] = [
            'time' => gmdate('c'),
            'level' => $level,
            'event' => $event,
            'message' => $message,
            'context' => $context,
        ];

        if (count($items) > $max) {
            $items = array_slice($items, -$max);
        }

        update_option(self::OPT, $items, false);

        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function clear(): void {
        delete_option(self::OPT);
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }
}
