<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }


/**
 * Freemius-driven license access helpers.
 */
final class License {

    public static function isPro(): bool {
        $fs = self::freemius();
        if ($fs) {
            if (method_exists($fs, 'has_active_valid_license')) {
                return (bool)$fs->has_active_valid_license();
            }
            if (method_exists($fs, 'can_use_premium_code')) {
                return (bool)$fs->can_use_premium_code();
            }
            if (method_exists($fs, 'is_paying')) {
                return (bool)$fs->is_paying();
            }
        }

        return false;
    }

    /**
     * @return array{status:string,plan:string}
     */
    public static function status(): array {
        $fs = self::freemius();
        if (!$fs) return ['status' => 'inactive', 'plan' => ''];
        $isPro = self::isPro();
        return [
            'status' => $isPro ? 'active' : 'inactive',
            'plan' => self::freemiusPlanLabel($fs),
        ];
    }

    public static function handleSync(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));
        check_admin_referer('dtarot_license_sync');

        $fs = self::freemius();
        $canSync = $fs && method_exists($fs, '_sync_licenses');
        if ($canSync && method_exists($fs, 'get_user')) {
            $user = $fs->get_user();
            if (!is_object($user) || empty($user->id)) {
                $canSync = false;
            }
        }
        if ($canSync) {
            $fs->_sync_licenses();
        }

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-settings',
            'tab' => 'pro',
            'msg' => $canSync ? 'license_synced' : 'license_sync_unavailable',
        ], admin_url('admin.php')));
        exit;
    }

    private static function freemius(): ?object {
        if (!function_exists('dtarot_fs')) return null;
        $fs = dtarot_fs();
        return is_object($fs) ? $fs : null;
    }

    private static function freemiusPlanLabel(object $fs): string {
        if (method_exists($fs, 'get_plan')) {
            $plan = $fs->get_plan();
            if (is_object($plan)) {
                if (isset($plan->title) && is_string($plan->title)) return $plan->title;
                if (isset($plan->name) && is_string($plan->name)) return $plan->name;
                if (isset($plan->id)) return (string)$plan->id;
            }
            if (is_array($plan)) {
                if (isset($plan['title']) && is_string($plan['title'])) return $plan['title'];
                if (isset($plan['name']) && is_string($plan['name'])) return $plan['name'];
                if (isset($plan['id'])) return (string)$plan['id'];
            }
        }
        return '';
    }
}
