<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }


use DailyTarot\Support\License;

/**
 * Centralized feature gating.
 *
 * Keep gating logic here so future Pro/free splits don't leak into UI code.
 */
final class FeatureFlags {

    /**
     * Returns true if a named feature is enabled.
     *
     * @param string $feature e.g. "ai", "analytics", "email"
     */
    public static function enabled(string $feature): bool {
        $feature = strtolower(trim($feature));

        // Default: core features only.
        $isPro = (class_exists(License::class) && License::isPro());

        $enabled = match ($feature) {
            'core', 'calendar', 'decks', 'meaning_packs', 'shortcode' => true,

            // Pro-only (for now): AI + Analytics.
            'ai', 'analytics' => $isPro,

            // Pro is a site-level license (not user roles).
            'pro' => $isPro,

            // Provider settings are Pro-only.
            'ai_provider' => $isPro,
            default => false,
        };

        /**
         * Filter: dtarot_feature_enabled
         * Allows site owners (and later licensing) to enable/disable features.
         */
        return (bool) apply_filters('dtarot_feature_enabled', $enabled, $feature);
    }
}
