<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }


final class EmailCtaStore {
    private const OPT = 'dtarot_email_cta_signups_v1';
    private const MAX_ROWS = 500;

    /**
     * @return array<string,array{email:string,first_at:string,last_at:string,count:int}>
     */
    public static function getAll(): array {
        $raw = get_option(self::OPT, []);
        if (!is_array($raw)) return [];

        $out = [];
        foreach ($raw as $email => $row) {
            if (!is_string($email) || !is_array($row)) continue;
            $email = sanitize_email($email);
            if ($email === '') continue;
            $first = isset($row['first_at']) && is_string($row['first_at']) ? (string)$row['first_at'] : '';
            $last = isset($row['last_at']) && is_string($row['last_at']) ? (string)$row['last_at'] : '';
            $count = isset($row['count']) ? (int)$row['count'] : 1;
            $out[$email] = [
                'email' => $email,
                'first_at' => $first,
                'last_at' => $last,
                'count' => $count > 0 ? $count : 1,
            ];
        }
        return $out;
    }

    public static function add(string $email): void {
        $email = sanitize_email($email);
        if ($email === '') return;

        $all = self::getAll();
        $now = gmdate('c');

        if (isset($all[$email])) {
            $row = $all[$email];
            $row['last_at'] = $now;
            $row['count'] = isset($row['count']) ? (int)$row['count'] + 1 : 2;
            $all[$email] = $row;
        } else {
            $all[$email] = [
                'email' => $email,
                'first_at' => $now,
                'last_at' => $now,
                'count' => 1,
            ];
        }

        if (count($all) > self::MAX_ROWS) {
            uasort($all, static function ($a, $b) {
                $aTime = isset($a['last_at']) ? strtotime((string)$a['last_at']) : 0;
                $bTime = isset($b['last_at']) ? strtotime((string)$b['last_at']) : 0;
                return $aTime <=> $bTime;
            });
            $all = array_slice($all, -self::MAX_ROWS, null, true);
        }

        update_option(self::OPT, $all, false);
    }

    public static function handleExport(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_email_cta_export');

        $rows = self::getAll();
        $filename = 'dtarot-email-cta-' . gmdate('Ymd-His') . '.csv';

        nocache_headers();
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=' . $filename);

        $escape = static function (string $value): string {
            $value = str_replace('"', '""', $value);
            if (preg_match('/[",\r\n]/', $value)) {
                return '"' . $value . '"';
            }
            return $value;
        };

        $lines = [];
        $lines[] = implode(',', [
            $escape('email'),
            $escape('first_at'),
            $escape('last_at'),
            $escape('count'),
        ]);
        foreach ($rows as $row) {
            $lines[] = implode(',', [
                $escape((string)$row['email']),
                $escape((string)$row['first_at']),
                $escape((string)$row['last_at']),
                $escape((string)$row['count']),
            ]);
        }
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV file download.
        echo implode("\n", $lines);
        exit;
    }
}
