<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



final class EmailCtaSettings {

    private const OPT = 'dtarot_email_cta_v1';

    private const KEY_PROVIDER = 'provider';
    private const KEY_ACTION_URL = 'action_url';
    private const KEY_HONEYPOT_NAME = 'honeypot_name';
    private const KEY_RECIPIENT = 'recipient';
    private const KEY_MAILCHIMP_API_KEY = 'mailchimp_api_key';
    private const KEY_MAILCHIMP_LIST_ID = 'mailchimp_list_id';

    /** @return array{provider:string,action_url:string,honeypot_name:string,recipient:string,has_mailchimp_api_key:bool,mailchimp_list_id:string} */
    public static function get(): array {
        $raw = get_option(self::OPT, null);
        if (!is_array($raw)) $raw = [];

        $provider = isset($raw[self::KEY_PROVIDER]) && is_string($raw[self::KEY_PROVIDER]) ? sanitize_key($raw[self::KEY_PROVIDER]) : '';
        if (!in_array($provider, ['', 'mailchimp', 'wp'], true)) {
            $provider = '';
        }

        $actionUrl = isset($raw[self::KEY_ACTION_URL]) && is_string($raw[self::KEY_ACTION_URL]) ? trim($raw[self::KEY_ACTION_URL]) : '';
        $honeypot = isset($raw[self::KEY_HONEYPOT_NAME]) && is_string($raw[self::KEY_HONEYPOT_NAME]) ? sanitize_key($raw[self::KEY_HONEYPOT_NAME]) : '';
        $recipient = isset($raw[self::KEY_RECIPIENT]) && is_string($raw[self::KEY_RECIPIENT]) ? sanitize_email($raw[self::KEY_RECIPIENT]) : '';
        if ($recipient === '') $recipient = sanitize_email((string)get_option('admin_email', ''));

        $apiKey = isset($raw[self::KEY_MAILCHIMP_API_KEY]) && is_string($raw[self::KEY_MAILCHIMP_API_KEY]) ? trim($raw[self::KEY_MAILCHIMP_API_KEY]) : '';
        $listId = isset($raw[self::KEY_MAILCHIMP_LIST_ID]) && is_string($raw[self::KEY_MAILCHIMP_LIST_ID]) ? trim($raw[self::KEY_MAILCHIMP_LIST_ID]) : '';

        return [
            'provider' => $provider,
            'action_url' => $actionUrl,
            'honeypot_name' => $honeypot,
            'recipient' => $recipient,
            'has_mailchimp_api_key' => $apiKey !== '',
            'mailchimp_list_id' => $listId,
        ];
    }

    public static function set(string $provider, string $actionUrl, string $honeypotName, string $recipient, string $mailchimpApiKey, string $mailchimpListId): void {
        $provider = sanitize_key($provider);
        if (!in_array($provider, ['', 'mailchimp', 'wp'], true)) {
            $provider = '';
        }

        $actionUrl = trim($actionUrl);
        if ($provider === '' || $provider === 'wp') {
            $actionUrl = '';
        } elseif ($actionUrl !== '') {
            $actionUrl = (string)esc_url_raw($actionUrl);
        }

        $honeypotName = sanitize_key($honeypotName);
        $recipient = sanitize_email($recipient);
        $mailchimpApiKey = trim($mailchimpApiKey);
        $mailchimpListId = trim($mailchimpListId);

        $raw = get_option(self::OPT, null);
        if (!is_array($raw)) $raw = [];

        $existingKey = isset($raw[self::KEY_MAILCHIMP_API_KEY]) && is_string($raw[self::KEY_MAILCHIMP_API_KEY])
            ? trim($raw[self::KEY_MAILCHIMP_API_KEY])
            : '';
        if ($mailchimpApiKey === '' && $existingKey !== '') {
            $mailchimpApiKey = $existingKey;
        }

        $raw = [
            self::KEY_PROVIDER => $provider,
            self::KEY_ACTION_URL => $actionUrl,
            self::KEY_HONEYPOT_NAME => $honeypotName,
            self::KEY_RECIPIENT => $recipient,
            self::KEY_MAILCHIMP_API_KEY => $mailchimpApiKey,
            self::KEY_MAILCHIMP_LIST_ID => $mailchimpListId,
        ];

        update_option(self::OPT, $raw, false);

        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function handleSave(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_email_cta_save');

        $provider = isset($_POST['dtarot_email_cta_provider']) ? sanitize_key((string)wp_unslash($_POST['dtarot_email_cta_provider'])) : '';
        $actionUrl = isset($_POST['dtarot_email_cta_action_url']) ? esc_url_raw((string)wp_unslash($_POST['dtarot_email_cta_action_url'])) : '';
        $honeypot = isset($_POST['dtarot_email_cta_honeypot_name']) ? sanitize_key((string)wp_unslash($_POST['dtarot_email_cta_honeypot_name'])) : '';
        $recipient = isset($_POST['dtarot_email_cta_recipient']) ? sanitize_email((string)wp_unslash($_POST['dtarot_email_cta_recipient'])) : '';
        $mailchimpApiKey = isset($_POST['dtarot_email_cta_mailchimp_api_key']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_email_cta_mailchimp_api_key'])) : '';
        $mailchimpListId = isset($_POST['dtarot_email_cta_mailchimp_list_id']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_email_cta_mailchimp_list_id'])) : '';

        self::set($provider, $actionUrl, $honeypot, $recipient, $mailchimpApiKey, $mailchimpListId);

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-settings',
            'tab' => 'cta',
            'msg' => 'email_cta_saved',
        ], admin_url('admin.php')));
        exit;
    }

    public static function getMailchimpApiKey(): string {
        $raw = get_option(self::OPT, null);
        if (!is_array($raw)) return '';
        $key = isset($raw[self::KEY_MAILCHIMP_API_KEY]) && is_string($raw[self::KEY_MAILCHIMP_API_KEY]) ? (string)$raw[self::KEY_MAILCHIMP_API_KEY] : '';
        return trim($key);
    }

    public static function getMailchimpListId(): string {
        $raw = get_option(self::OPT, null);
        if (!is_array($raw)) return '';
        $id = isset($raw[self::KEY_MAILCHIMP_LIST_ID]) && is_string($raw[self::KEY_MAILCHIMP_LIST_ID]) ? (string)$raw[self::KEY_MAILCHIMP_LIST_ID] : '';
        return trim($id);
    }

    public static function handleSubmit(): void {
        $redirect = wp_get_referer();
        if (!is_string($redirect) || $redirect === '') {
            $redirect = home_url('/');
        }

        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field((string)wp_unslash($_POST['_wpnonce'])), 'dtarot_email_cta_submit')) {
            wp_safe_redirect(add_query_arg(['dtarot_email_cta' => 'error'], $redirect));
            exit;
        }

        $s = self::get();
        if ($s['provider'] !== 'wp') {
            wp_safe_redirect($redirect);
            exit;
        }

        $honeypotName = isset($s['honeypot_name']) ? sanitize_key($s['honeypot_name']) : '';
        if ($honeypotName !== '' && isset($_POST[$honeypotName]) && trim((string)wp_unslash($_POST[$honeypotName])) !== '') {
            wp_safe_redirect(add_query_arg(['dtarot_email_cta' => 'ok'], $redirect));
            exit;
        }

        $emailRaw = isset($_POST['dtarot_email']) ? (string)wp_unslash($_POST['dtarot_email']) : '';
        $email = sanitize_email($emailRaw);
        if ($email === '' || !is_email($email)) {
            wp_safe_redirect(add_query_arg(['dtarot_email_cta' => 'invalid'], $redirect));
            exit;
        }

        $to = $s['recipient'] !== '' ? $s['recipient'] : sanitize_email((string)get_option('admin_email', ''));
        if ($to === '') {
            wp_safe_redirect(add_query_arg(['dtarot_email_cta' => 'error'], $redirect));
            exit;
        }

        if (class_exists(EmailCtaStore::class)) {
            EmailCtaStore::add($email);
        }

        $subject = sprintf(__('New email signup: %s','daily-tarot'), $email);
        $body = sprintf(__('CTA signup email: %s','daily-tarot'), $email);
        wp_mail($to, $subject, $body);

        wp_safe_redirect(add_query_arg(['dtarot_email_cta' => 'ok'], $redirect));
        exit;
    }
}
