<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



use DailyTarot\Registry\Cards;
use DailyTarot\Support\PostTypes;

final class DefaultMeaningPacks {

    private const OPTION_KEY = 'dtarot_default_meaning_packs_v1';

    /**
     * @return array<string,int> Map: system => packId
     */
    public static function getAll(): array {
        $raw = get_option(self::OPTION_KEY, []);
        if (!is_array($raw)) $raw = [];

        $out = [];
        foreach (Cards::systems() as $system => $_label) {
            $sid = is_string($system) ? Cards::normalizeSystem($system) : '';
            if ($sid === '') continue;
            $out[$sid] = 0;
        }

        foreach ($raw as $system => $packId) {
            if (!is_string($system)) continue;
            $s = Cards::normalizeSystem($system);
            if ($s === '') continue;
            $out[$s] = is_numeric($packId) ? (int)$packId : 0;
        }

        // Validate stored IDs.
        foreach ($out as $system => $packId) {
            if ($packId <= 0) {
                $out[$system] = 0;
                continue;
            }
            if (!self::isPackForSystem($packId, $system)) {
                $out[$system] = 0;
            }
        }

        return $out;
    }

    public static function get(string $system): int {
        $system = Cards::normalizeSystem($system);
        if ($system === '') return 0;

        $all = self::getAll();
        $packId = isset($all[$system]) ? (int)$all[$system] : 0;
        return $packId > 0 ? $packId : 0;
    }

    public static function set(string $system, int $packId): bool {
        $system = Cards::normalizeSystem($system);
        if ($system === '' || $packId <= 0) return false;
        if (!self::isPackForSystem($packId, $system)) return false;

        $all = self::getAll();
        $all[$system] = $packId;

        update_option(self::OPTION_KEY, $all, false);
        return true;
    }

    public static function handleSetDefault(): void {
        if (!current_user_can('edit_posts')) {
            wp_die(esc_html__('Not allowed.','daily-tarot'));
        }

        $packId = isset($_GET['pack_id']) ? (int)sanitize_text_field((string)wp_unslash($_GET['pack_id'])) : 0;
        if ($packId <= 0) {
            wp_die(esc_html__('Missing meaning pack id.','daily-tarot'));
        }

        if (!current_user_can('edit_post', $packId)) {
            wp_die(esc_html__('Not allowed.','daily-tarot'));
        }

        check_admin_referer('dtarot_set_default_pack_' . $packId);

        $system = Cards::normalizeSystem((string)get_post_meta($packId, '_dtarot_system', true));
        if ($system === '') {
            wp_die(esc_html__('This meaning pack has no system.','daily-tarot'));
        }

        self::set($system, $packId);

        $ref = wp_get_referer();
        if (!is_string($ref) || $ref === '') {
            $ref = admin_url('edit.php?post_type=' . PostTypes::MEANING_PACK);
        }

        wp_safe_redirect(add_query_arg(['dtarot_default_updated' => '1'], $ref));
        exit;
    }

    private static function isPackForSystem(int $packId, string $system): bool {
        if ($packId <= 0) return false;
        $system = Cards::normalizeSystem($system);
        if ($system === '') return false;

        $p = get_post($packId);
        if (!$p || !isset($p->post_type) || !PostTypes::isMeaningPackType((string)$p->post_type)) return false;

        $ps = Cards::normalizeSystem((string)get_post_meta($packId, '_dtarot_system', true));
        return $ps === $system;
    }
}
