<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



use DailyTarot\Registry\Cards;
use DailyTarot\Support\PostTypes;

final class DefaultDecks {

    private const OPTION_KEY = 'dtarot_default_decks_v1';

    private const GROUP_TAROT = 'tarot';
    private const GROUP_LENORMAND = 'lenormand';
    private const GROUP_KIPPER = 'kipper';

    /**
     * @return array<string,int> Map: group => deckId
     */
    public static function getAll(): array {
        $raw = get_option(self::OPTION_KEY, []);
        if (!is_array($raw)) $raw = [];

        // Defaults are stored per-family (one Tarot, one Lenormand, one Kipper),
        // even though there can be multiple Kipper system variants.
        $out = [
            self::GROUP_TAROT => 0,
            self::GROUP_LENORMAND => 0,
            self::GROUP_KIPPER => 0,
        ];

        // Migration / compatibility:
        // - old shape: system => deckId
        // - new shape: group => deckId
        foreach ($raw as $key => $deckId) {
            if (!is_string($key)) continue;
            $id = is_numeric($deckId) ? (int)$deckId : 0;
            if ($id <= 0) continue;

            $group = self::normalizeGroupKey($key);
            if ($group === '') {
                $system = Cards::normalizeSystem($key);
                $group = self::groupForSystem($system);
            }
            if ($group === '') continue;

            // Keep the last one we see for a group.
            $out[$group] = $id;
        }

        // Validate stored IDs (group-aware).
        foreach ($out as $group => $deckId) {
            if ($deckId <= 0) {
                $out[$group] = 0;
                continue;
            }

            $deckSystem = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
            if (self::groupForSystem($deckSystem) !== $group) {
                $out[$group] = 0;
                continue;
            }

            // Also ensure the post is actually a deck.
            $p = get_post($deckId);
            if (!$p || !isset($p->post_type) || !PostTypes::isDeckType((string)$p->post_type)) {
                $out[$group] = 0;
            }
        }

        return $out;
    }

    public static function get(string $system): int {
        $system = Cards::normalizeSystem($system);
        if ($system === '') return 0;

        $group = self::groupForSystem($system);
        if ($group === '') return 0;

        $all = self::getAll();
        $deckId = isset($all[$group]) ? (int)$all[$group] : 0;
        if ($deckId <= 0) return 0;

        // System-safe: only return the default if it matches the requested system exactly.
        // This prevents a default Kipper (36) deck being used for Kipper Fin de Siècle (39) pages.
        $ps = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
        return ($ps === $system) ? $deckId : 0;
    }

    public static function set(string $system, int $deckId): bool {
        $system = Cards::normalizeSystem($system);
        if ($system === '' || $deckId <= 0) return false;
        if (!self::isDeckForSystem($deckId, $system)) return false;

        $group = self::groupForSystem($system);
        if ($group === '') return false;

        $all = self::getAll();
        $all[$group] = $deckId;

        update_option(self::OPTION_KEY, $all, false);
        return true;
    }

    public static function handleSetDefault(): void {
        if (!current_user_can('edit_posts')) {
            wp_die(esc_html__('Not allowed.','daily-tarot'));
        }

        $deckId = isset($_GET['deck_id']) ? (int)sanitize_text_field((string)wp_unslash($_GET['deck_id'])) : 0;
        if ($deckId <= 0) {
            wp_die(esc_html__('Missing deck id.','daily-tarot'));
        }

        if (!current_user_can('edit_post', $deckId)) {
            wp_die(esc_html__('Not allowed.','daily-tarot'));
        }

        check_admin_referer('dtarot_set_default_deck_' . $deckId);

        $system = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
        if ($system === '') {
            wp_die(esc_html__('This deck has no system.','daily-tarot'));
        }

        self::set($system, $deckId);

        $ref = wp_get_referer();
        if (!is_string($ref) || $ref === '') {
            $ref = admin_url('edit.php?post_type=' . PostTypes::DECK);
        }

        wp_safe_redirect(add_query_arg(['dtarot_default_updated' => '1'], $ref));
        exit;
    }

    private static function isDeckForSystem(int $deckId, string $system): bool {
        if ($deckId <= 0) return false;
        $system = Cards::normalizeSystem($system);
        if ($system === '') return false;

        $p = get_post($deckId);
        if (!$p || !isset($p->post_type) || !PostTypes::isDeckType((string)$p->post_type)) return false;

        $ps = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
        return $ps === $system;
    }

    private static function normalizeGroupKey(string $key): string {
        $k = sanitize_key($key);
        if (in_array($k, [self::GROUP_TAROT, self::GROUP_LENORMAND, self::GROUP_KIPPER], true)) {
            return $k;
        }
        return '';
    }

    private static function groupForSystem(string $system): string {
        $system = Cards::normalizeSystem($system);
        if ($system === Cards::SYSTEM_LENORMAND) return self::GROUP_LENORMAND;
        if (in_array($system, [Cards::SYSTEM_KIPPER, Cards::SYSTEM_GYPSY, Cards::SYSTEM_KIPPER_FIN_DE_SIECLE], true)) {
            return self::GROUP_KIPPER;
        }
        return self::GROUP_TAROT;
    }
}
