<?php
declare(strict_types=1);

namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }

use DailyTarot\Frontend\ReadableRoutes;
use DailyTarot\Seo\Sitemap;

/**
 * Cache purge integration for popular caching plugins.
 *
 * Goal: keep the public readable daily URL fresh when a day is published/updated.
 */
final class CachePurge {

    private const TARGETS_TRANSIENT = 'dtarot_cache_purge_targets_v1';

    public static function init(): void {
        // Invalidate our cached list of pages to purge when content changes.
        add_action('save_post', [__CLASS__, 'invalidateTargets'], 10, 2);
        add_action('deleted_post', [__CLASS__, 'invalidateTargets'], 10, 1);
        add_action('trashed_post', [__CLASS__, 'invalidateTargets'], 10, 1);
        add_action('untrashed_post', [__CLASS__, 'invalidateTargets'], 10, 1);
    }

    /** @param int|\WP_Post $post */
    public static function invalidateTargets($post): void {
        // Avoid recursion/extra work during autosaves.
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        delete_transient(self::TARGETS_TRANSIENT);
    }

    public static function purgeDate(string $date): void {
        $date = trim($date);
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) return;

        if (!class_exists(ReadableRoutes::class)) return;

        $urls = [];

        $dailyUrl = ReadableRoutes::urlForDate($date);
        if ($dailyUrl !== '') $urls[] = $dailyUrl;

        // Also purge any pages that embed our output (shortcode/block), plus home/front page.
        foreach (self::autoPurgeUrls() as $u) {
            if ($u !== '') $urls[] = $u;
        }

        $urls = array_values(array_unique(array_filter(array_map('trim', $urls))));
        foreach ($urls as $u) {
            self::purgeUrl($u);
        }
    }

    /**
     * Auto-detect pages/posts that likely render Daily Tarot output.
     *
     * @return array<int,string> list of URLs
     */
    private static function autoPurgeUrls(): array {
        $cached = get_transient(self::TARGETS_TRANSIENT);
        if (is_array($cached)) {
            return array_values(array_filter(array_map('strval', $cached)));
        }

        $urls = [];

        // Home URL (common landing page).
        $home = (string)home_url('/');
        if ($home !== '') $urls[] = $home;

        // Plugin sitemap.
        if (class_exists(Sitemap::class)) {
            $s = Sitemap::url();
            if ($s !== '') $urls[] = $s;
        }

        // Static front page / posts page.
        $frontId = (int)get_option('page_on_front', 0);
        if ($frontId > 0) {
            $u = get_permalink($frontId);
            if (is_string($u) && $u !== '') $urls[] = $u;
        }
        $postsId = (int)get_option('page_for_posts', 0);
        if ($postsId > 0) {
            $u = get_permalink($postsId);
            if (is_string($u) && $u !== '') $urls[] = $u;
        }

        // Pages/posts containing our shortcode or Gutenberg block.
        $ids = self::findPostsContainingDailyTarot();
        foreach ($ids as $id) {
            $u = get_permalink($id);
            if (is_string($u) && $u !== '') $urls[] = $u;
        }

        $urls = array_values(array_unique(array_filter(array_map('trim', $urls))));
        set_transient(self::TARGETS_TRANSIENT, $urls, 6 * HOUR_IN_SECONDS);
        return $urls;
    }

    /**
     * Finds published posts/pages containing Daily Tarot embeds.
     *
     * @return array<int,int> post IDs
     */
    private static function findPostsContainingDailyTarot(): array {
        global $wpdb;
        if (!$wpdb || !isset($wpdb->posts)) return [];

        // Shortcode and block markers.
        $likeShortcode = '%[daily_tarot%';
        $likeBlock = '%wp:dtarot/daily-tarot%';

                // Limit to keep this light; we cache results in a transient (see self::TARGETS_TRANSIENT).
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Prepared query; results cached in a transient.
                $rows = $wpdb->get_col($wpdb->prepare(
                        "SELECT ID FROM {$wpdb->posts}
                         WHERE post_status = 'publish'
                             AND post_type IN ('page','post')
                             AND (post_content LIKE %s OR post_content LIKE %s)
                         ORDER BY post_modified_gmt DESC
                         LIMIT 50",
                        $likeShortcode,
                        $likeBlock
                ));
        if (!is_array($rows)) return [];

        $ids = [];
        foreach ($rows as $r) {
            $id = (int)$r;
            if ($id > 0) $ids[] = $id;
        }
        return $ids;
    }

    public static function purgeUrl(string $url): void {
        $url = trim($url);
        if ($url === '') return;

        /**
         * Allow custom site-specific cache purge handlers.
         *
         * @param string $url
         */
        do_action('dtarot_cache_purge_url', $url);

        // WP Rocket
        if (function_exists('rocket_clean_files')) {
            try {
                rocket_clean_files([$url]);
            } catch (\Throwable $e) {
                // no-op
            }
        } elseif (function_exists('rocket_clean_file')) {
            try {
                rocket_clean_file($url);
            } catch (\Throwable $e) {
                // no-op
            }
        }

        // LiteSpeed Cache
        // Docs/usage varies by version; these actions are the most common.
        if (has_action('litespeed_purge_url')) {
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Third-party hook (LiteSpeed).
            do_action('litespeed_purge_url', $url);
        } else {
            // Fallback: generic purge action.
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Third-party hook (LiteSpeed).
            do_action('litespeed_purge', 'url ' . $url);
        }

        // W3 Total Cache
        if (function_exists('w3tc_flush_url')) {
            try {
                w3tc_flush_url($url);
            } catch (\Throwable $e) {
                // no-op
            }
        }

        // Generic page cache plugins sometimes expose this.
        if (function_exists('wp_cache_clear_cache')) {
            // This clears all cache; avoid unless site has no URL-specific API.
            // Intentionally not calling by default.
        }
    }
}
