<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



use DailyTarot\Support\FeatureFlags;

final class BookingSettings {

    private const OPTION = 'dtarot_booking_settings';
    private const MAX_OPTION_BYTES = 2000000;

    public static function defaults(): array {
        return [
            'mode' => 'request',
            'timezone' => '',
            'weekdays' => ['mon','tue','wed','thu','fri'],
            'start_time' => '09:00',
            'end_time' => '17:00',
            'buffer_before' => 0,
            'buffer_after' => 0,
            'slot_interval' => 15,
            'blackout_dates' => '',
            'payment_mode' => 'none',
            'payment_provider' => 'paypal',
            'paypal_url' => '',
            'stripe_url' => '',
            'style' => 'modern',
            'admin_email' => (string)get_option('admin_email', ''),
            'from_name' => (string)get_bloginfo('name'),
            'from_email' => (string)get_option('admin_email', ''),
            'templates' => [
                'user_request_subject' => __('Your reading request is received','daily-tarot'),
                'user_request_body' => __("Hi {name},\n\nYour request has been received. We'll confirm your reading shortly.\n\nReading: {reading}\nTime: {date} {time} ({timezone})\n\nThank you,\n{site}",'daily-tarot'),
                'user_confirmed_subject' => __('Your reading is confirmed','daily-tarot'),
                'user_confirmed_body' => __("Hi {name},\n\nYour reading is confirmed.\n\nReading: {reading}\nTime: {date} {time} ({timezone})\n\nPayment link: {payment_url}\n\nThank you,\n{site}",'daily-tarot'),
                'user_declined_subject' => __('Your reading request','daily-tarot'),
                'user_declined_body' => __("Hi {name},\n\nThanks for your request. Unfortunately we are not available for that time.\n\nReading: {reading}\nRequested: {date} {time} ({timezone})\n\nPlease feel welcome to choose another time.\n\n{site}",'daily-tarot'),
                'admin_new_subject' => __('New reading request','daily-tarot'),
                'admin_new_body' => __("New booking from {name} ({email}).\n\nReading: {reading}\nTime: {date} {time} ({timezone})\nStatus: {status}\nPayment provider: {payment_provider}\nPayment link: {payment_url}\n\nQuestion:\n{question}\n\nAdmin time: {admin_date} {admin_time} ({admin_timezone})",'daily-tarot'),
            ],
        ];
    }

    public static function get(): array {
        $stored = [];
        if (isset($GLOBALS['wpdb']) && is_object($GLOBALS['wpdb'])) {
            $len = (int)$GLOBALS['wpdb']->get_var(
                $GLOBALS['wpdb']->prepare(
                    "SELECT LENGTH(option_value) FROM {$GLOBALS['wpdb']->options} WHERE option_name = %s",
                    self::OPTION
                )
            );
            if ($len > 0 && $len <= self::MAX_OPTION_BYTES) {
                $stored = get_option(self::OPTION, []);
            }
        } else {
            $stored = get_option(self::OPTION, []);
        }
        if (!is_array($stored)) $stored = [];
        $settings = array_replace_recursive(self::defaults(), $stored);
        if (class_exists(FeatureFlags::class) && !FeatureFlags::enabled('pro')) {
            $settings['payment_mode'] = 'none';
        }
        return $settings;
    }

    public static function getMode(): string {
        $s = self::get();
        $mode = isset($s['mode']) ? sanitize_key((string)$s['mode']) : 'request';
        return in_array($mode, ['request','instant'], true) ? $mode : 'request';
    }

    public static function getTimezone(): \DateTimeZone {
        $s = self::get();
        $tz = isset($s['timezone']) ? trim((string)$s['timezone']) : '';
        if ($tz === '') {
            if (function_exists('wp_timezone')) {
                return wp_timezone();
            }
            $tz = (string)get_option('timezone_string', 'UTC');
        }
        try {
            return new \DateTimeZone($tz);
        } catch (\Exception $e) {
            return new \DateTimeZone('UTC');
        }
    }

    public static function handleSave(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));
        check_admin_referer('dtarot_booking_settings');

        $current = self::get();

        $mode = isset($_POST['booking_mode']) ? sanitize_key((string)wp_unslash($_POST['booking_mode'])) : (string)$current['mode'];
        if (!in_array($mode, ['request','instant'], true)) $mode = (string)$current['mode'];

        $timezone = isset($_POST['timezone']) ? sanitize_text_field((string)wp_unslash($_POST['timezone'])) : (string)$current['timezone'];
        $weekdays = $current['weekdays'];
        if (isset($_POST['weekdays']) && is_array($_POST['weekdays'])) {
            $weekdays = array_map('sanitize_key', wp_unslash($_POST['weekdays']));
            $weekdays = array_values(array_intersect($weekdays, ['mon','tue','wed','thu','fri','sat','sun']));
        }

        $start = isset($_POST['start_time']) ? sanitize_text_field((string)wp_unslash($_POST['start_time'])) : (string)$current['start_time'];
        $end = isset($_POST['end_time']) ? sanitize_text_field((string)wp_unslash($_POST['end_time'])) : (string)$current['end_time'];

        $bufferBefore = isset($_POST['buffer_before']) ? (int)wp_unslash($_POST['buffer_before']) : (int)$current['buffer_before'];
        $bufferAfter = isset($_POST['buffer_after']) ? (int)wp_unslash($_POST['buffer_after']) : (int)$current['buffer_after'];
        $slotInterval = isset($_POST['slot_interval']) ? (int)wp_unslash($_POST['slot_interval']) : (int)$current['slot_interval'];
        if ($slotInterval < 5) $slotInterval = 5;

        $blackout = isset($_POST['blackout_dates']) ? sanitize_textarea_field((string)wp_unslash($_POST['blackout_dates'])) : (string)$current['blackout_dates'];

        $paymentMode = isset($_POST['payment_mode']) ? sanitize_key((string)wp_unslash($_POST['payment_mode'])) : (string)$current['payment_mode'];
        if (!in_array($paymentMode, ['none','before','after'], true)) $paymentMode = (string)$current['payment_mode'];
        $paymentProvider = isset($_POST['payment_provider']) ? sanitize_key((string)wp_unslash($_POST['payment_provider'])) : (string)$current['payment_provider'];
        if (!in_array($paymentProvider, ['paypal','stripe','both'], true)) $paymentProvider = (string)$current['payment_provider'];
        $paypalUrl = isset($_POST['paypal_url']) ? esc_url_raw((string)wp_unslash($_POST['paypal_url'])) : (string)$current['paypal_url'];
        $stripeUrl = isset($_POST['stripe_url']) ? esc_url_raw((string)wp_unslash($_POST['stripe_url'])) : (string)$current['stripe_url'];

        $isPro = class_exists(FeatureFlags::class) && FeatureFlags::enabled('pro');
        if (!$isPro) {
            $paymentMode = 'none';
        }

        $style = isset($_POST['style']) ? sanitize_key((string)wp_unslash($_POST['style'])) : (string)$current['style'];
        if (!in_array($style, ['modern','mystic','minimal'], true)) {
            $style = (string)$current['style'];
        }

        $adminEmail = isset($_POST['admin_email']) ? sanitize_email((string)wp_unslash($_POST['admin_email'])) : (string)$current['admin_email'];
        $fromName = isset($_POST['from_name']) ? sanitize_text_field((string)wp_unslash($_POST['from_name'])) : (string)$current['from_name'];
        $fromEmail = isset($_POST['from_email']) ? sanitize_email((string)wp_unslash($_POST['from_email'])) : (string)$current['from_email'];

        $templates = isset($current['templates']) && is_array($current['templates'])
            ? $current['templates']
            : self::defaults()['templates'];
        foreach ($templates as $key => $val) {
            if (!isset($_POST[$key])) continue;
            $raw = (string)wp_unslash($_POST[$key]);
            if (str_ends_with($key, '_subject')) {
                $templates[$key] = sanitize_text_field($raw);
            } else {
                $templates[$key] = wp_kses_post($raw);
            }
        }

        $data = [
            'mode' => $mode,
            'timezone' => $timezone,
            'weekdays' => $weekdays,
            'start_time' => $start,
            'end_time' => $end,
            'buffer_before' => max(0, $bufferBefore),
            'buffer_after' => max(0, $bufferAfter),
            'slot_interval' => $slotInterval,
            'blackout_dates' => $blackout,
            'payment_mode' => $paymentMode,
            'payment_provider' => $paymentProvider,
            'paypal_url' => $paypalUrl,
            'stripe_url' => $stripeUrl,
            'style' => $style,
            'admin_email' => $adminEmail,
            'from_name' => $fromName,
            'from_email' => $fromEmail,
            'templates' => $templates,
        ];

        update_option(self::OPTION, $data, false);

        wp_safe_redirect(add_query_arg([
            'page' => isset($_POST['redirect_page']) ? sanitize_key((string)wp_unslash($_POST['redirect_page'])) : 'daily-tarot-settings',
            'tab' => isset($_POST['redirect_tab']) ? sanitize_key((string)wp_unslash($_POST['redirect_tab'])) : 'bookings',
            'dtarot_booking' => 'saved',
        ], admin_url('admin.php')));
        exit;
    }

    public static function paymentUrlFor(string $provider): string {
        $s = self::get();
        $provider = sanitize_key($provider);
        if ($provider === 'stripe') {
            return (string)($s['stripe_url'] ?? '');
        }
        return (string)($s['paypal_url'] ?? '');
    }
}
