<?php
declare(strict_types=1);


namespace DailyTarot\Support;
if (!defined('ABSPATH')) { exit; }


use DailyTarot\Support\PostTypes;

final class BookingAvailability {

    private const STATUS_BLOCKING = ['pending','approved','completed'];

    public static function parseBlackoutDates(string $raw): array {
        $items = preg_split('/[\s,]+/', trim($raw));
        if (!is_array($items)) return [];
        $out = [];
        foreach ($items as $item) {
            $item = trim((string)$item);
            if ($item === '') continue;
            if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $item)) continue;
            $out[$item] = true;
        }
        return array_keys($out);
    }

    public static function getReadingDuration(int $readingTypeId): int {
        $duration = (int)get_post_meta($readingTypeId, '_dtarot_reading_duration', true);
        if ($duration <= 0) $duration = 30;
        return $duration;
    }

    public static function getAvailabilitySettings(): array {
        $s = BookingSettings::get();
        return [
            'weekdays' => isset($s['weekdays']) && is_array($s['weekdays']) ? $s['weekdays'] : [],
            'start_time' => isset($s['start_time']) ? (string)$s['start_time'] : '09:00',
            'end_time' => isset($s['end_time']) ? (string)$s['end_time'] : '17:00',
            'buffer_before' => isset($s['buffer_before']) ? (int)$s['buffer_before'] : 0,
            'buffer_after' => isset($s['buffer_after']) ? (int)$s['buffer_after'] : 0,
            'slot_interval' => isset($s['slot_interval']) ? (int)$s['slot_interval'] : 15,
            'blackout_dates' => isset($s['blackout_dates']) ? (string)$s['blackout_dates'] : '',
        ];
    }

    /**
     * @return array<int,array{start_utc:int,end_utc:int,start_admin:string,end_admin:string}>
     */
    public static function slotsForDate(string $date, int $readingTypeId): array {
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) return [];

        $tz = BookingSettings::getTimezone();
        $settings = self::getAvailabilitySettings();
        $blackouts = self::parseBlackoutDates((string)$settings['blackout_dates']);
        if (in_array($date, $blackouts, true)) return [];

        $dtDay = \DateTimeImmutable::createFromFormat('Y-m-d', $date, $tz);
        $weekday = $dtDay ? strtolower((string)$dtDay->format('D')) : '';
        $map = ['mon'=>'mon','tue'=>'tue','wed'=>'wed','thu'=>'thu','fri'=>'fri','sat'=>'sat','sun'=>'sun'];
        $dayKey = $map[$weekday] ?? '';
        $allowedDays = is_array($settings['weekdays']) ? $settings['weekdays'] : [];
        if ($dayKey === '' || (!empty($allowedDays) && !in_array($dayKey, $allowedDays, true))) {
            return [];
        }

        $startTime = (string)$settings['start_time'];
        $endTime = (string)$settings['end_time'];

        $start = \DateTimeImmutable::createFromFormat('Y-m-d H:i', $date . ' ' . $startTime, $tz);
        $end = \DateTimeImmutable::createFromFormat('Y-m-d H:i', $date . ' ' . $endTime, $tz);
        if (!$start || !$end) return [];
        if ($end <= $start) return [];

        $duration = self::getReadingDuration($readingTypeId);
        $bufferBefore = max(0, (int)$settings['buffer_before']);
        $bufferAfter = max(0, (int)$settings['buffer_after']);
        $interval = max(5, (int)$settings['slot_interval']);

        $block = max($duration + $bufferBefore + $bufferAfter, $interval);
        $slots = [];

        $nowUtc = time();
        $windowStartUtc = (int)$start->setTimezone(new \DateTimeZone('UTC'))->format('U');
        $windowEndUtc = (int)$end->setTimezone(new \DateTimeZone('UTC'))->format('U');
        $blocked = self::blockedIntervals($windowStartUtc, $windowEndUtc);

        $current = $start;
        while (true) {
            $slotStart = $current;
            $slotEnd = $slotStart->modify('+' . $duration . ' minutes');
            if ($slotEnd > $end) break;

            $startUtc = (int)$slotStart->setTimezone(new \DateTimeZone('UTC'))->format('U');
            $endUtc = (int)$slotEnd->setTimezone(new \DateTimeZone('UTC'))->format('U');

            if ($endUtc > $nowUtc && !self::overlapsBlocked($startUtc, $endUtc, $blocked)) {
                $slots[] = [
                    'start_utc' => $startUtc,
                    'end_utc' => $endUtc,
                    'start_admin' => $slotStart->format('Y-m-d H:i'),
                    'end_admin' => $slotEnd->format('Y-m-d H:i'),
                ];
            }

            $current = $current->modify('+' . $block . ' minutes');
        }

        return apply_filters('dtarot_booking_slots', $slots, $date, $readingTypeId, $settings);
    }

    public static function isBlocked(int $startUtc, int $endUtc): bool {
        $blocked = self::blockedIntervals($startUtc, $endUtc);
        return self::overlapsBlocked($startUtc, $endUtc, $blocked);
    }

    /**
     * @return array<int,array{start:int,end:int}>
     */
    private static function blockedIntervals(int $startUtc, int $endUtc): array {
        $q = new \WP_Query([
            'post_type' => PostTypes::bookingTypes(),
            'post_status' => 'publish',
            'posts_per_page' => 200,
            'meta_query' => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Bounded window query (max 200).
                [
                    'key' => '_dtarot_booking_status',
                    'value' => self::STATUS_BLOCKING,
                    'compare' => 'IN',
                ],
                [
                    'key' => '_dtarot_booking_start_utc',
                    'value' => $endUtc,
                    'type' => 'NUMERIC',
                    'compare' => '<',
                ],
                [
                    'key' => '_dtarot_booking_end_utc',
                    'value' => $startUtc,
                    'type' => 'NUMERIC',
                    'compare' => '>',
                ],
            ],
        ]);

        if (!$q->have_posts()) return [];
        $out = [];
        foreach ($q->posts as $post) {
            if (!is_object($post)) continue;
            $sid = (int)get_post_meta($post->ID, '_dtarot_booking_start_utc', true);
            $eid = (int)get_post_meta($post->ID, '_dtarot_booking_end_utc', true);
            if ($sid <= 0 || $eid <= 0) continue;
            $out[] = ['start' => $sid, 'end' => $eid];
        }
        return $out;
    }

    /**
     * @param array<int,array{start:int,end:int}> $blocked
     */
    private static function overlapsBlocked(int $startUtc, int $endUtc, array $blocked): bool {
        foreach ($blocked as $row) {
            $s = (int)$row['start'];
            $e = (int)$row['end'];
            if ($s < $endUtc && $e > $startUtc) return true;
        }
        return false;
    }

    public static function validateSlot(int $startUtc, int $readingTypeId): bool {
        $duration = self::getReadingDuration($readingTypeId);
        $endUtc = $startUtc + ($duration * 60);
        if ($startUtc <= time()) return false;

        $tz = BookingSettings::getTimezone();
        $dt = (new \DateTimeImmutable('@' . $startUtc))->setTimezone($tz);
        $date = $dt->format('Y-m-d');

        $settings = self::getAvailabilitySettings();
        $blackouts = self::parseBlackoutDates((string)$settings['blackout_dates']);
        if (in_array($date, $blackouts, true)) return false;

        $weekday = strtolower((string)$dt->format('D'));
        $map = ['mon'=>'mon','tue'=>'tue','wed'=>'wed','thu'=>'thu','fri'=>'fri','sat'=>'sat','sun'=>'sun'];
        $dayKey = $map[$weekday] ?? '';
        $allowedDays = is_array($settings['weekdays']) ? $settings['weekdays'] : [];
        if ($dayKey === '' || (!empty($allowedDays) && !in_array($dayKey, $allowedDays, true))) {
            return false;
        }

        $startTime = (string)$settings['start_time'];
        $endTime = (string)$settings['end_time'];
        $startDay = \DateTimeImmutable::createFromFormat('Y-m-d H:i', $date . ' ' . $startTime, $tz);
        $endDay = \DateTimeImmutable::createFromFormat('Y-m-d H:i', $date . ' ' . $endTime, $tz);
        if (!$startDay || !$endDay) return false;

        $startLocal = (int)$dt->format('U');
        $endLocal = $endUtc;
        $windowStart = (int)$startDay->setTimezone(new \DateTimeZone('UTC'))->format('U');
        $windowEnd = (int)$endDay->setTimezone(new \DateTimeZone('UTC'))->format('U');
        if ($startLocal < $windowStart || $endLocal > $windowEnd) return false;

        if (self::isBlocked($startUtc, $endUtc)) return false;

        $ok = true;
        return (bool)apply_filters('dtarot_booking_validate_slot', $ok, $startUtc, $readingTypeId);
    }
}
