<?php
declare(strict_types=1);

namespace DailyTarot\Seo;
if (!defined('ABSPATH')) { exit; }

use DailyTarot\Calendar\DayEntryService;
use DailyTarot\Frontend\ReadableRoutes;
use DailyTarot\Meaning\MeaningPackRepository;
use DailyTarot\Reading\ReadingComposer;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\PostTypes;

final class Yoast {

    public static function init(): void {
        if (!self::isYoastActive()) return;

        // Title + description.
        add_filter('wpseo_title', [__CLASS__, 'filter_title']);
        add_filter('wpseo_metadesc', [__CLASS__, 'filter_metadesc']);

        // Canonical.
        add_filter('wpseo_canonical', [__CLASS__, 'filter_canonical']);

        // OpenGraph.
        add_filter('wpseo_opengraph_title', [__CLASS__, 'filter_og_title']);
        add_filter('wpseo_opengraph_desc', [__CLASS__, 'filter_og_desc']);
        add_filter('wpseo_opengraph_url', [__CLASS__, 'filter_og_url']);
        add_filter('wpseo_opengraph_image', [__CLASS__, 'filter_og_image']);
        add_filter('wpseo_opengraph_image_url', [__CLASS__, 'filter_og_image']);

        // Twitter.
        add_filter('wpseo_twitter_title', [__CLASS__, 'filter_twitter_title']);
        add_filter('wpseo_twitter_description', [__CLASS__, 'filter_twitter_desc']);
        add_filter('wpseo_twitter_image', [__CLASS__, 'filter_twitter_image']);
    }

    private static function isYoastActive(): bool {
        return defined('WPSEO_VERSION') || class_exists('WPSEO_Options');
    }

    private static function context(): array {
        if (class_exists(ReadableRoutes::class) && ReadableRoutes::isReadableRequest()) {
            $date = (string)get_query_var('dtarot_date');
            $entry = DayEntryService::getPublished($date);
            if (!$entry) return ['type' => 'none'];

            $arr = $entry->toArray();
            $cardId = (string)($arr['card'] ?? '');
            $deckId = (int)($arr['deck'] ?? 0);
            $packId = (int)($arr['pack'] ?? 0);

            $ts = strtotime($date . ' 00:00:00');
            $pretty = $date;
            if ($ts !== false) {
                $pretty = function_exists('wp_date') ? (string)wp_date('F j, Y', $ts) : (string)date_i18n('F j, Y', $ts);
            }

            $cardName = Cards::name($cardId);
            $title = $cardName !== '' ? ($cardName . ' — ' . $pretty) : $pretty;

            $meaning = MeaningPackRepository::getMeaning($packId, $cardId);
            $desc = ReadingComposer::buildMetaDescription($arr, $meaning, 160);

            $canonical = ReadableRoutes::urlForDate($date);
            $imgUrl = self::imageForReadable($arr, $deckId, $cardId);

            return [
                'type' => 'readable',
                'title' => $title,
                'desc' => $desc,
                'canonical' => $canonical,
                'img' => $imgUrl,
            ];
        }

        if (class_exists(ReadableRoutes::class) && ReadableRoutes::isCardRequest()) {
            $system = (string)get_query_var('dtarot_system');
            $cardId = (string)get_query_var('dtarot_card');
            $system = Cards::normalizeSystem($system);
            $cardId = sanitize_text_field($cardId);
            if ($system === '' || $cardId === '') return ['type' => 'none'];

            $title = Cards::name($cardId);

            [$deckId, $packId] = self::getCardOverrides($system);
            if ($deckId <= 0) $deckId = self::findFirstDeckForSystem($system);
            if ($packId <= 0) $packId = self::findFirstPackForSystem($system);

            $meaning = $packId > 0 ? MeaningPackRepository::getMeaning($packId, $cardId) : MeaningPackRepository::emptyMeaning();
            $desc = ReadingComposer::buildMetaDescription([], $meaning, 160);

            $canonical = ReadableRoutes::urlForCard($system, $cardId);
            $imgUrl = self::imageForCard($deckId, $cardId);

            return [
                'type' => 'card',
                'title' => $title,
                'desc' => $desc,
                'canonical' => $canonical,
                'img' => $imgUrl,
            ];
        }

        return ['type' => 'none'];
    }

    private static function imageForReadable(array $entryArr, int $deckId, string $cardId): string {
        $overrideUrl = isset($entryArr['image_override_url']) ? (string)$entryArr['image_override_url'] : '';
        if ($overrideUrl !== '') return esc_url($overrideUrl);

        if ($deckId <= 0 || $cardId === '') return '';
        $imgs = get_post_meta($deckId, '_dtarot_cards', true);
        if (!is_array($imgs) || !$imgs) return '';
        foreach (Cards::kipperGypsyAliases($cardId) as $id) {
            if (empty($imgs[$id]) || !is_string($imgs[$id])) continue;
            $u = trim((string)$imgs[$id]);
            if ($u !== '') return esc_url($u);
        }
        return '';
    }

    private static function imageForCard(int $deckId, string $cardId): string {
        if ($deckId <= 0 || $cardId === '') return '';
        $imgs = get_post_meta($deckId, '_dtarot_cards', true);
        if (!is_array($imgs) || !$imgs) return '';
        foreach (Cards::kipperGypsyAliases($cardId) as $id) {
            if (empty($imgs[$id]) || !is_string($imgs[$id])) continue;
            $u = trim((string)$imgs[$id]);
            if ($u !== '') return esc_url($u);
        }
        return '';
    }

    /** @return array{0:int,1:int} [deckId, packId] */
    private static function getCardOverrides(string $system): array {
        $system = Cards::normalizeSystem($system);
        if ($system === '') return [0, 0];

        $deckId = 0;
        $packId = 0;

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only override via query args; no state is mutated.
        if (isset($_GET['deck_id'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only override via query args; no state is mutated.
            $deckId = (int)sanitize_text_field((string)wp_unslash($_GET['deck_id']));
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only override via query args; no state is mutated.
        if (isset($_GET['pack_id'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only override via query args; no state is mutated.
            $packId = (int)sanitize_text_field((string)wp_unslash($_GET['pack_id']));
        }

        if ($deckId > 0 && !self::isDeckForSystem($deckId, $system)) {
            $deckId = 0;
        }
        if ($packId > 0 && !self::isPackForSystem($packId, $system)) {
            $packId = 0;
        }

        return [$deckId, $packId];
    }

    private static function isDeckForSystem(int $deckId, string $system): bool {
        if ($deckId <= 0) return false;
        $p = get_post($deckId);
        if (!$p || !isset($p->post_type) || !PostTypes::isDeckType((string)$p->post_type)) return false;
        $ps = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
        return $ps === $system;
    }

    private static function isPackForSystem(int $packId, string $system): bool {
        if ($packId <= 0) return false;
        $p = get_post($packId);
        if (!$p || !isset($p->post_type) || !PostTypes::isMeaningPackType((string)$p->post_type)) return false;
        $ps = Cards::normalizeSystem((string)get_post_meta($packId, '_dtarot_system', true));
        return $ps === $system;
    }

    private static function findFirstDeckForSystem(string $system): int {
        $system = Cards::normalizeSystem($system);
        if ($system === '') return 0;

        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => 50,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'date',
            'order' => 'DESC',
            'fields' => 'ids',
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        ]);

        foreach ((array)$decks as $did) {
            $did = (int)$did;
            if ($did <= 0) continue;
            $ps = Cards::normalizeSystem((string)get_post_meta($did, '_dtarot_system', true));
            if ($ps === $system) return $did;
        }

        return 0;
    }

    private static function findFirstPackForSystem(string $system): int {
        $system = Cards::normalizeSystem($system);
        if ($system === '') return 0;

        $packs = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'numberposts' => 50,
            'post_status' => ['publish','draft'],
            'orderby' => 'date',
            'order' => 'DESC',
            'fields' => 'ids',
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        ]);

        foreach ((array)$packs as $pid) {
            $pid = (int)$pid;
            if ($pid <= 0) continue;
            $ps = Cards::normalizeSystem((string)get_post_meta($pid, '_dtarot_system', true));
            if ($ps === $system) return $pid;
        }

        return 0;
    }

    public static function filter_title(string $title): string {
        $ctx = self::context();
        if (($ctx['type'] ?? 'none') === 'none') return $title;
        $t = isset($ctx['title']) && is_string($ctx['title']) ? $ctx['title'] : '';
        return $t !== '' ? $t : $title;
    }

    public static function filter_metadesc(string $desc): string {
        $ctx = self::context();
        if (($ctx['type'] ?? 'none') === 'none') return $desc;
        $d = isset($ctx['desc']) && is_string($ctx['desc']) ? $ctx['desc'] : '';
        return $d !== '' ? $d : $desc;
    }

    public static function filter_canonical(string $canonical): string {
        $ctx = self::context();
        if (($ctx['type'] ?? 'none') === 'none') return $canonical;
        $c = isset($ctx['canonical']) && is_string($ctx['canonical']) ? $ctx['canonical'] : '';
        return $c !== '' ? $c : $canonical;
    }

    public static function filter_og_title(string $title): string {
        return self::filter_title($title);
    }

    public static function filter_og_desc(string $desc): string {
        return self::filter_metadesc($desc);
    }

    public static function filter_og_url($url): string {
        // Yoast can pass null into this filter; normalize defensively.
        if (!is_string($url)) {
            $url = '';
        }
        return self::filter_canonical($url);
    }

    public static function filter_og_image(string $img): string {
        $ctx = self::context();
        if (($ctx['type'] ?? 'none') === 'none') return $img;
        $u = isset($ctx['img']) && is_string($ctx['img']) ? $ctx['img'] : '';
        return $u !== '' ? $u : $img;
    }

    public static function filter_twitter_title(string $title): string {
        return self::filter_title($title);
    }

    public static function filter_twitter_desc(string $desc): string {
        return self::filter_metadesc($desc);
    }

    public static function filter_twitter_image(string $img): string {
        return self::filter_og_image($img);
    }
}
