<?php
declare(strict_types=1);

namespace DailyTarot\Seo;
if (!defined('ABSPATH')) { exit; }

use DailyTarot\Calendar\DayEntryService;
use DailyTarot\Frontend\ReadableRoutes;
use DailyTarot\Support\CacheVersion;

final class Sitemap {

    private const CACHE_GROUP = 'dtarot';

    private static function cacheVer(): int {
        return class_exists(CacheVersion::class) ? CacheVersion::get() : 1;
    }

    public static function init(): void {
        add_action('init', [__CLASS__, 'registerRewrites']);
        add_filter('query_vars', [__CLASS__, 'addQueryVars']);
        add_action('template_redirect', [__CLASS__, 'maybeRender']);
    }

    public static function registerRewrites(): void {
        // /dtarot-sitemap.xml
        add_rewrite_rule('^dtarot-sitemap\.xml$', 'index.php?dtarot_sitemap=1', 'top');

        // Optional index + page support.
        add_rewrite_rule('^dtarot-sitemap-index\.xml$', 'index.php?dtarot_sitemap=1&dtarot_sitemap_mode=index', 'top');
        add_rewrite_rule('^dtarot-sitemap-(\d+)\.xml$', 'index.php?dtarot_sitemap=1&dtarot_sitemap_page=$matches[1]', 'top');
    }

    /** @param array<int,string> $vars */
    public static function addQueryVars(array $vars): array {
        $vars[] = 'dtarot_sitemap';
        $vars[] = 'dtarot_sitemap_mode';
        $vars[] = 'dtarot_sitemap_page';
        return $vars;
    }

    public static function url(): string {
        $useIndex = (bool)apply_filters('dtarot_sitemap_use_index', false);
        return $useIndex ? home_url('/dtarot-sitemap-index.xml') : home_url('/dtarot-sitemap.xml');
    }

    public static function pageUrl(int $page): string {
        $page = max(1, (int)$page);
        return home_url('/dtarot-sitemap-' . $page . '.xml');
    }

    public static function maybeRender(): void {
        $v = get_query_var('dtarot_sitemap');
        if (!is_string($v) && !is_numeric($v)) return;
        if ((string)$v !== '1') return;

        if (!class_exists(ReadableRoutes::class) || !class_exists(DayEntryService::class)) {
            status_header(404);
            return;
        }

        $limit = (int)apply_filters('dtarot_sitemap_limit', 365);
        if ($limit <= 0) $limit = 365;
        if ($limit > 2000) $limit = 2000;

        $perSitemap = (int)apply_filters('dtarot_sitemap_per_sitemap', 2000);
        if ($perSitemap <= 0) $perSitemap = 2000;
        if ($perSitemap > 2000) $perSitemap = 2000;

        $useIndex = (bool)apply_filters('dtarot_sitemap_use_index', false);
        $mode = get_query_var('dtarot_sitemap_mode');
        $pageVar = get_query_var('dtarot_sitemap_page');
        $page = (is_numeric($pageVar) ? (int)$pageVar : 1);
        if ($page <= 0) $page = 1;

        $ttl = (int)apply_filters('dtarot_sitemap_cache_ttl', 6 * HOUR_IN_SECONDS);
        if ($ttl < 60) $ttl = 60;
        if ($ttl > 24 * HOUR_IN_SECONDS) $ttl = 24 * HOUR_IN_SECONDS;

        $cacheKey = 'dtarot_sitemap_xml_' . self::cacheVer() . '_' . $limit . '_' . ($useIndex ? 'idx' : 'single') . '_' . $perSitemap . '_' . $page;
        $xml = get_transient($cacheKey);
        if (!is_string($xml) || $xml === '') {
            $fromDate = function_exists('wp_date') ? (string)wp_date('Y-m-d') : (string)current_time('Y-m-d');
            $items = DayEntryService::latestPublished($limit, $fromDate);

            $dates = array_keys($items);
            $total = count($dates);

            if ($useIndex && is_string($mode) && $mode === 'index') {
                $pages = (int)ceil($total / max(1, $perSitemap));
                if ($pages < 1) $pages = 1;

                $out = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
                $out .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
                for ($p = 1; $p <= $pages; $p++) {
                    $out .= "  <sitemap>\n";
                    $out .= '    <loc>' . esc_url(self::pageUrl($p)) . "</loc>\n";
                    $out .= "  </sitemap>\n";
                }
                $out .= "</sitemapindex>\n";
                $xml = $out;
                set_transient($cacheKey, $xml, $ttl);
            } else {
                $slice = $dates;
                if ($useIndex) {
                    $offset = ($page - 1) * $perSitemap;
                    $slice = array_slice($dates, $offset, $perSitemap);
                }

                $includeImages = (bool)apply_filters('dtarot_sitemap_include_images', false);

                $out = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
                $out .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"'
                    . ($includeImages ? ' xmlns:image="http://www.google.com/schemas/sitemap-image/1.1"' : '')
                    . '>' . "\n";

                foreach ($slice as $date) {
                    if (!is_string($date) || $date === '') continue;
                    $loc = ReadableRoutes::urlForDate($date);
                    if ($loc === '') continue;

                    $lastmod = $date;
                    if (class_exists(CacheVersion::class)) {
                        $ts = CacheVersion::getDayLastModified($date);
                        if ($ts > 0) {
                            $lastmod = function_exists('wp_date') ? (string)wp_date('c', $ts) : (string)gmdate('c', $ts);
                        }
                    }

                    $out .= "  <url>\n";
                    $out .= '    <loc>' . esc_url($loc) . "</loc>\n";
                    $out .= '    <lastmod>' . esc_html($lastmod) . "</lastmod>\n";

                    if ($includeImages) {
                        $img = (string)apply_filters('dtarot_sitemap_image_url', '', $date, $items[$date] ?? null);
                        $img = trim($img);
                        if ($img !== '') {
                            $out .= "    <image:image><image:loc>" . esc_url($img) . "</image:loc></image:image>\n";
                        }
                    }

                    $out .= "  </url>\n";
                }

                $out .= "</urlset>\n";
                $xml = $out;
                set_transient($cacheKey, $xml, $ttl);
            }
        }

        // Headers.
        header('Content-Type: application/xml; charset=UTF-8', true);

        $noindex = (bool)apply_filters('dtarot_sitemap_noindex', false);
        if ($noindex) {
            header('X-Robots-Tag: noindex', true);
        }

        // Conditional GET.
        $etag = 'W/"dtarot-sitemap-' . md5($xml) . '"';
        header('ETag: ' . $etag);

        $inm = isset($_SERVER['HTTP_IF_NONE_MATCH'])
            ? sanitize_text_field((string)wp_unslash($_SERVER['HTTP_IF_NONE_MATCH']))
            : '';
        if ($inm !== '' && trim($inm) === $etag) {
            status_header(304);
            exit;
        }

        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- XML payload is generated/escaped above.
        echo $xml;
        exit;
    }
}
