<?php
declare(strict_types=1);

namespace DailyTarot\Seo;
if (!defined('ABSPATH')) { exit; }

use DailyTarot\Calendar\DayEntryService;
use DailyTarot\Frontend\ReadableRoutes;
use DailyTarot\Meaning\MeaningPackRepository;
use DailyTarot\Reading\ReadingComposer;
use DailyTarot\Registry\Cards;

final class Schema {

    public static function init(): void {
        add_action('wp_head', [__CLASS__, 'maybeOutput'], 20);
    }

    private static function seoPluginActive(): bool {
        return defined('WPSEO_VERSION') || class_exists('WPSEO_Options') || defined('RANK_MATH_VERSION') || class_exists('RankMath\\Helper');
    }

    public static function maybeOutput(): void {
        if (!class_exists(ReadableRoutes::class) || !ReadableRoutes::isReadableRequest()) return;

        // Default: avoid double-schema when an SEO plugin is active.
        $allow = (bool)apply_filters('dtarot_schema_output_enabled', !self::seoPluginActive());
        if (!$allow) return;

        $date = (string)get_query_var('dtarot_date');
        $entry = DayEntryService::getPublished($date);
        if (!$entry) return;

        $arr = $entry->toArray();
        $cardId = (string)($arr['card'] ?? '');
        $packId = (int)($arr['pack'] ?? 0);
        $deckId = (int)($arr['deck'] ?? 0);

        $title = Cards::name($cardId);
        $ts = strtotime($date . ' 00:00:00');
        $pretty = $date;
        if ($ts !== false) {
            $pretty = function_exists('wp_date') ? (string)wp_date('F j, Y', $ts) : (string)date_i18n('F j, Y', $ts);
        }

        $pageTitle = $title !== '' ? ($title . ' — ' . $pretty) : $pretty;

        $meaning = MeaningPackRepository::getMeaning($packId, $cardId);
        $desc = ReadingComposer::buildMetaDescription($arr, $meaning, 160);

        $url = ReadableRoutes::urlForDate($date);
        $img = '';
        $overrideUrl = isset($arr['image_override_url']) ? (string)$arr['image_override_url'] : '';
        if ($overrideUrl !== '') {
            $img = esc_url($overrideUrl);
        } elseif ($deckId > 0 && $cardId !== '') {
            $imgs = get_post_meta($deckId, '_dtarot_cards', true);
            if (is_array($imgs) && $imgs) {
                foreach (Cards::kipperGypsyAliases($cardId) as $id) {
                    if (empty($imgs[$id]) || !is_string($imgs[$id])) continue;
                    $u = trim((string)$imgs[$id]);
                    if ($u !== '') {
                        $img = esc_url($u);
                        break;
                    }
                }
            }
        }

        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'Article',
            'headline' => $pageTitle,
            'description' => $desc,
            'url' => $url,
            'datePublished' => $date,
            'dateModified' => $date,
        ];

        if ($img !== '') {
            $schema['image'] = [$img];
        }

        $siteName = (string)get_bloginfo('name');
        if ($siteName !== '') {
            $schema['publisher'] = [
                '@type' => 'Organization',
                'name' => $siteName,
            ];
        }

        echo "\n" . '<script type="application/ld+json">' . wp_json_encode($schema) . '</script>' . "\n";
    }
}
